import React, { useMemo, useState } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  MenuItem,
  Typography,
  Box,
  Alert,
  CircularProgress,
  IconButton,
  InputAdornment,
} from '@mui/material';
import Visibility from '@mui/icons-material/Visibility';
import VisibilityOff from '@mui/icons-material/VisibilityOff';
import axios from 'axios';

import { BASE_URL } from '../../const';

const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    backgroundColor: theme.palette.background.paper,
    borderRadius: 12,
    width: '500px',
    maxWidth: '90vw',
  },
}));

const Field = styled(TextField)(({ theme }) => ({
  marginBottom: theme.spacing(2),
}));

const SERVICE_OPTIONS = [
  {
    value: 'aws_bedrock',
    label: 'AWS Bedrock',
  },
];

const DEFAULT_EXTRA_BODY = `{
  "anthropic_version": "bedrock-2023-05-31"
}`;

const BEDROCK_ARN_REGEX = /^arn:(aws[a-z0-9-]*):bedrock:[a-z0-9-]+:(?:\d{12})?:[-A-Za-z0-9/_+=,.@:]+$/i;

const trimOrEmpty = value => (typeof value === 'string' ? value.trim() : '');

const isBedrockArn = value => BEDROCK_ARN_REGEX.test(trimOrEmpty(value));

const inferRegionFromArn = arn => {
  const trimmed = trimOrEmpty(arn);
  if (!isBedrockArn(trimmed)) return '';
  const parts = trimmed.split(':');
  if (parts.length > 4) {
    return parts[3] || '';
  }
  return '';
};

const inferRegionFromEndpoint = rawValue => {
  const endpoint = trimOrEmpty(rawValue);
  if (!endpoint) return '';
  if (isBedrockArn(endpoint)) {
    return inferRegionFromArn(endpoint);
  }
  const match = endpoint.match(/bedrock-runtime\.([a-z0-9-]+)\.amazonaws\.com/i);
  return match ? match[1] : '';
};

const inferModelIdFromArn = arn => {
  const trimmed = trimOrEmpty(arn);
  if (!isBedrockArn(trimmed)) return '';
  const parts = trimmed.split(':', 6);
  if (parts.length < 6) return '';
  const resource = parts[5];
  if (!resource) return '';
  if (resource.includes('/')) {
    const resourceParts = resource.split('/', 2);
    return resourceParts[1] || '';
  }
  return resource;
};

const normalizeEndpointUrl = value => {
  const trimmed = trimOrEmpty(value);
  if (!trimmed) return '';
  try {
    const parsed = new URL(trimmed);
    if (parsed.protocol !== 'https:') return '';
    if (parsed.username || parsed.password) return '';
    const host = (parsed.hostname || '').toLowerCase();
    if (!host) return '';
    const isAmazonDomain = host === 'amazonaws.com' || host.endsWith('.amazonaws.com');
    if (!isAmazonDomain) return '';
    if (parsed.port && parsed.port !== '' && parsed.port !== '443') return '';
    if (parsed.pathname && parsed.pathname !== '/' && parsed.pathname !== '') return '';
    if (parsed.search || parsed.hash) return '';
    if (/^\d+\.\d+\.\d+\.\d+$/.test(host) || host.includes(':')) return '';
    return parsed.origin;
  } catch (err) {
    return '';
  }
};

const formatErrorMessage = err => {
  if (err?.response?.data?.detail) return err.response.data.detail;
  if (err?.message) return err.message;
  return 'An unexpected error occurred while creating the endpoint.';
};

const AddExternalEndpointModal = ({
  open,
  onClose,
  onCreated,
  refreshModelList,
}) => {
  const [modelName, setModelName] = useState('');
  const [service, setService] = useState('aws_bedrock');
  const [modelId, setModelId] = useState('');
  const [region, setRegion] = useState('');
  const [endpointUrl, setEndpointUrl] = useState('');
  const [profileArn, setProfileArn] = useState('');
  const [credentialSecret, setCredentialSecret] = useState('');
  const [showCredentialSecret, setShowCredentialSecret] = useState(false);
  const [extraBody, setExtraBody] = useState(DEFAULT_EXTRA_BODY);
  const [description, setDescription] = useState('');
  const [submitting, setSubmitting] = useState(false);
  const [error, setError] = useState(null);

  const normalizedProfileArn = useMemo(() => {
    const explicitArn = trimOrEmpty(profileArn);
    if (isBedrockArn(explicitArn)) return explicitArn;

    const endpointCandidate = trimOrEmpty(endpointUrl);
    if (isBedrockArn(endpointCandidate)) return endpointCandidate;

    const modelIdCandidate = trimOrEmpty(modelId);
    if (isBedrockArn(modelIdCandidate)) return modelIdCandidate;

    return '';
  }, [profileArn, endpointUrl, modelId]);

  const autoDetectedRegion = useMemo(() => {
    const fromEndpoint = inferRegionFromEndpoint(endpointUrl);
    if (fromEndpoint) return fromEndpoint;
    return inferRegionFromArn(normalizedProfileArn);
  }, [endpointUrl, normalizedProfileArn]);

  const resolvedRegion = useMemo(() => {
    const explicit = trimOrEmpty(region);
    if (explicit) return explicit;
    return autoDetectedRegion;
  }, [region, autoDetectedRegion]);

  const resolvedModelId = useMemo(() => {
    const trimmed = trimOrEmpty(modelId);
    if (trimmed && !trimmed.toLowerCase().startsWith('arn:')) {
      return trimmed;
    }
    const fromSelf = inferModelIdFromArn(trimmed);
    if (fromSelf) return fromSelf;
    return inferModelIdFromArn(normalizedProfileArn);
  }, [modelId, normalizedProfileArn]);

  const hasManualRegion = Boolean(trimOrEmpty(region));

  const resetState = () => {
    setModelName('');
    setModelId('');
    setRegion('');
    setEndpointUrl('');
    setProfileArn('');
    setCredentialSecret('');
    setShowCredentialSecret(false);
    setExtraBody(DEFAULT_EXTRA_BODY);
    setDescription('');
    setError(null);
  };

  const handleDialogClose = () => {
    if (submitting) {
      return;
    }
    onClose?.();
  };

  const handleCancel = () => {
    if (submitting) {
      return;
    }
    resetState();
    onClose?.();
  };

  const handleReset = () => {
    if (submitting) {
      return;
    }
    resetState();
  };

  const handleSubmit = async () => {
    if (!modelName.trim()) {
      setError('Model name is required.');
      return;
    }

    const explicitModelId = trimOrEmpty(modelId);
    if (explicitModelId && explicitModelId.toLowerCase().startsWith('arn:') && !isBedrockArn(explicitModelId)) {
      setError('Model ARN must reference an AWS Bedrock resource.');
      return;
    }

    const explicitProfileArn = trimOrEmpty(profileArn);
    if (explicitProfileArn && !isBedrockArn(explicitProfileArn)) {
      setError('Inference profile ARN must reference AWS Bedrock.');
      return;
    }

    const finalModelId = resolvedModelId;
    if (!finalModelId) {
      setError('Provide a model ID or an inference profile ARN.');
      return;
    }

    const finalRegion = resolvedRegion;
    if (!finalRegion) {
      setError('AWS region is required. Specify it directly or provide an endpoint URL or ARN that includes it.');
      return;
    }

    let parsedExtraBody = {};
    if (extraBody && extraBody.trim()) {
      try {
        parsedExtraBody = JSON.parse(extraBody);
      } catch (parseErr) {
        setError('Extra body must be valid JSON.');
        return;
      }
    }

    const finalCredentialSecret = trimOrEmpty(credentialSecret);
    const finalProfileArn = normalizedProfileArn || undefined;

    const endpointCandidate = trimOrEmpty(endpointUrl);
    let usableEndpointUrl;
    if (endpointCandidate) {
      if (endpointCandidate.toLowerCase().startsWith('arn:')) {
        if (!isBedrockArn(endpointCandidate)) {
          setError('Endpoint ARN must reference AWS Bedrock.');
          return;
        }
      } else {
        usableEndpointUrl = normalizeEndpointUrl(endpointCandidate);
        if (!usableEndpointUrl) {
          setError('Endpoint URL must be an HTTPS host under amazonaws.com with no path, query, or fragment.');
          return;
        }
      }
    }

    if (!finalCredentialSecret) {
      const proceed = window.confirm(
        'If you create an external model with no credential, clients will have to pass the credential on every call. Kamiwaza will pass through but ignore the bearer token. Press OK to create, or Cancel to change this setup.'
      );
      if (!proceed) {
        return;
      }
    }

    setSubmitting(true);
    setError(null);

    try {
      const externalEndpointConfig = {
        service,
        model_id: finalModelId,
        region: finalRegion,
        extra_body: parsedExtraBody,
        display_name: modelName.trim(),
      };

      if (usableEndpointUrl) {
        externalEndpointConfig.endpoint_url = usableEndpointUrl;
      }
      if (finalProfileArn) {
        externalEndpointConfig.inference_profile_arn = finalProfileArn;
      }
      if (finalCredentialSecret) {
        externalEndpointConfig.credential_secret = finalCredentialSecret;
      }

      const defaultConfigPayload = {
        name: `${modelName.trim()} External Endpoint`,
        default: true,
        description: description.trim() || 'AWS Bedrock external endpoint configuration',
        config: {
          engine_name: 'aws_bedrock',
          external_endpoint: JSON.stringify(externalEndpointConfig),
        },
        system_config: {},
      };

      const modelPayload = {
        name: modelName.trim(),
        description: description.trim() || 'External inference endpoint',
        hub: 'external',
        purpose: 'external',
        modelfamily: 'external',
        repo_modelId: finalModelId,
        source_repository: SERVICE_OPTIONS.find(option => option.value === service)?.label || 'External',
        private: true,
        default_config: defaultConfigPayload,
      };

      const modelResponse = await axios.post(`${BASE_URL}/models/`, modelPayload);
      const modelData = modelResponse.data;
      if (!modelData?.id) {
        throw new Error('Model creation did not return an identifier.');
      }

      if (typeof refreshModelList === 'function') {
        await Promise.resolve(refreshModelList());
      }

      if (typeof onCreated === 'function') {
        onCreated();
      }

      resetState();
      onClose?.();
    } catch (err) {
      setError(formatErrorMessage(err));
    } finally {
      setSubmitting(false);
    }
  };

  return (
    <StyledDialog open={open} onClose={handleDialogClose} maxWidth="sm">
      <DialogTitle>
        <Typography variant="h6" sx={{ fontWeight: 600 }}>
          Add External Inference Endpoint
        </Typography>
        <Typography variant="body2" color="text.secondary">
          Configure an AWS Bedrock-backed model
        </Typography>
      </DialogTitle>
      <DialogContent>
        {error && (
          <Alert severity="error" sx={{ mb: 2 }}>
            {error}
          </Alert>
        )}
        <Field
          fullWidth
          label="Display Name"
          value={modelName}
          onChange={event => setModelName(event.target.value)}
          disabled={submitting}
        />
        <Field
          select
          fullWidth
          label="Service"
          value={service}
          onChange={event => setService(event.target.value)}
          disabled
        >
          {SERVICE_OPTIONS.map(option => (
            <MenuItem key={option.value} value={option.value}>
              {option.label}
            </MenuItem>
          ))}
        </Field>
        <Field
          fullWidth
          label="Model ID or Inference Profile ID / ARN"
          value={modelId}
          onChange={event => setModelId(event.target.value)}
          helperText={resolvedModelId
            ? `Will use model ID: ${resolvedModelId}`
            : 'Paste a Bedrock model ID (e.g. global.anthropic.claude-sonnet-4-5-20250929-v1:0) or an inference profile ARN.'}
          disabled={submitting}
        />
        <Field
          fullWidth
          label="AWS Region"
          value={region}
          onChange={event => setRegion(event.target.value)}
          helperText={autoDetectedRegion
            ? `Auto-detected region: ${autoDetectedRegion}${hasManualRegion ? ' (manual override applied)' : ''}`
            : 'Specify explicitly or provide an endpoint URL or ARN containing the region.'}
          disabled={submitting}
        />
        <Field
          fullWidth
          label="Endpoint URL (optional)"
          value={endpointUrl}
          onChange={event => setEndpointUrl(event.target.value)}
          placeholder="https://bedrock-runtime.us-east-2.amazonaws.com"
          disabled={submitting}
        />
        <Field
          fullWidth
          label="Inference Profile ARN (optional)"
          value={profileArn}
          onChange={event => setProfileArn(event.target.value)}
          helperText={normalizedProfileArn && normalizedProfileArn !== trimOrEmpty(profileArn)
            ? `Detected ARN: ${normalizedProfileArn}`
            : 'Optional if you already provided an ARN elsewhere.'}
          disabled={submitting}
        />
        <Field
          fullWidth
          label="Credential Secret or URN"
          value={credentialSecret}
          onChange={event => setCredentialSecret(event.target.value)}
          type={showCredentialSecret ? 'text' : 'password'}
          helperText="Provide a secret URN or plaintext credential (we will store it securely)."
          disabled={submitting}
          InputProps={{
            endAdornment: (
              <InputAdornment position="end">
                <IconButton
                  size="small"
                  edge="end"
                  onClick={() => setShowCredentialSecret(prev => !prev)}
                  aria-label={showCredentialSecret ? 'Hide credential secret' : 'Show credential secret'}
                >
                  {showCredentialSecret ? <VisibilityOff fontSize="small" /> : <Visibility fontSize="small" />}
                </IconButton>
              </InputAdornment>
            ),
          }}
        />
        <Field
          fullWidth
          label="Extra Body JSON"
          value={extraBody}
          onChange={event => setExtraBody(event.target.value)}
          multiline
          minRows={4}
          disabled={submitting}
          helperText="Merged into every request to Bedrock (when absent)."
        />
        <Field
          fullWidth
          label="Description"
          value={description}
          onChange={event => setDescription(event.target.value)}
          multiline
          minRows={2}
          disabled={submitting}
        />
      </DialogContent>
      <DialogActions sx={{ px: 3, pb: 2 }}>
        <Button onClick={handleReset} disabled={submitting}>
          Reset
        </Button>
        <Button onClick={handleCancel} disabled={submitting}>
          Cancel
        </Button>
        <Button
          variant="contained"
          onClick={handleSubmit}
          disabled={submitting}
        >
          {submitting ? (
            <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
              <CircularProgress size={18} />
              Saving
            </Box>
          ) : (
            'Save Endpoint'
          )}
        </Button>
      </DialogActions>
    </StyledDialog>
  );
};

AddExternalEndpointModal.propTypes = {
  open: PropTypes.bool.isRequired,
  onClose: PropTypes.func,
  onCreated: PropTypes.func,
  refreshModelList: PropTypes.func,
};

AddExternalEndpointModal.defaultProps = {
  onClose: undefined,
  onCreated: undefined,
  refreshModelList: undefined,
};

export default AddExternalEndpointModal;
