import React from 'react';
import PropTypes from 'prop-types';
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  IconButton,
  Chip,
  Typography,
  Box,
  Tooltip,
} from '@mui/material';
import { styled } from '@mui/material/styles';
import VisibilityIcon from '@mui/icons-material/Visibility';
import DeleteIcon from '@mui/icons-material/Delete';
import WarningAmberIcon from '@mui/icons-material/WarningAmber';
import CheckCircleIcon from '@mui/icons-material/CheckCircle';
import ErrorIcon from '@mui/icons-material/Error';

// Styled components
const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: theme.spacing(1.5),
  border: '1px solid rgba(255, 255, 255, 0.08)',
}));

const StyledTableHead = styled(TableHead)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.05)',
  '& .MuiTableCell-head': {
    fontWeight: 600,
    color: 'rgba(255, 255, 255, 0.9)',
    borderBottom: '1px solid rgba(255, 255, 255, 0.1)',
  },
}));

const StyledTableRow = styled(TableRow)(({ theme }) => ({
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
  },
  '&:last-child td': {
    borderBottom: 'none',
  },
}));

const DeploymentTypeChip = styled(Chip)(({ theme, dtype }) => {
  const colors = {
    vllm: '#4CAF50',
    llamacpp: '#2196F3',
    // TODO: Uncomment when these deployment types are implemented
    // mlx: '#FF9800',
    // ampere: '#9C27B0',
    // app_garden: '#00BCD4',
    // tool_shed: '#FFC107',
  };
  
  return {
    backgroundColor: colors[dtype] || '#757575',
    color: 'white',
    fontWeight: 500,
  };
});

const EmptyState = styled(Box)(({ theme }) => ({
  display: 'flex',
  flexDirection: 'column',
  alignItems: 'center',
  justifyContent: 'center',
  padding: theme.spacing(8),
  color: 'rgba(255, 255, 255, 0.5)',
}));

const LogsTable = ({ logs, onViewLog, onDeleteLog }) => {
  // Format file size
  const formatSize = (bytes) => {
    if (bytes < 1024) return `${bytes} B`;
    if (bytes < 1024 * 1024) return `${(bytes / 1024).toFixed(1)} KB`;
    return `${(bytes / (1024 * 1024)).toFixed(1)} MB`;
  };

  // Format date
  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleString('en-US', {
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  // Get status icon
  const getStatusIcon = (log) => {
    if (log.capture_active) {
      return (
        <Tooltip title="Actively capturing logs">
          <CheckCircleIcon color="success" fontSize="small" />
        </Tooltip>
      );
    }
    if (log.patterns_detected?.runtime_error || log.patterns_detected?.container_crash) {
      return (
        <Tooltip title="Errors detected in logs">
          <ErrorIcon color="error" fontSize="small" />
        </Tooltip>
      );
    }
    if (log.is_orphaned) {
      return (
        <Tooltip title="Deployment no longer exists">
          <WarningAmberIcon color="warning" fontSize="small" />
        </Tooltip>
      );
    }
    // Default status - log exists and is normal
    return (
      <Tooltip title="Log file available">
        <CheckCircleIcon color="disabled" fontSize="small" />
      </Tooltip>
    );
  };

  if (!logs || logs.length === 0) {
    return (
      <StyledTableContainer component={Paper} elevation={0}>
        <EmptyState>
          <Typography variant="h6" gutterBottom>
            No logs found
          </Typography>
          <Typography variant="body2">
            Logs will appear here when deployments are running
          </Typography>
        </EmptyState>
      </StyledTableContainer>
    );
  }

  return (
    <StyledTableContainer component={Paper} elevation={0}>
      <Table>
        <StyledTableHead>
          <TableRow>
            <TableCell width="40">Status</TableCell>
            <TableCell width="300">Deployment ID</TableCell>
            <TableCell width="120">Type</TableCell>
            <TableCell width="100">Size</TableCell>
            <TableCell width="140">Modified</TableCell>
            <TableCell width="120" align="center">Actions</TableCell>
          </TableRow>
        </StyledTableHead>
        <TableBody>
          {logs.map((log) => (
            <StyledTableRow key={`${log.deployment_type}-${log.deployment_id}`}>
              <TableCell>
                <Box sx={{ display: 'flex', alignItems: 'center' }}>
                  {getStatusIcon(log)}
                </Box>
              </TableCell>
              <TableCell>
                <Box>
                  <Typography variant="body2" sx={{ fontFamily: 'monospace' }}>
                    {log.deployment_id}
                  </Typography>
                  {log.is_orphaned && (
                    <Typography variant="caption" color="warning.main">
                      Orphaned (deployment not found)
                    </Typography>
                  )}
                </Box>
              </TableCell>
              <TableCell>
                <DeploymentTypeChip
                  label={log.deployment_type.toUpperCase()}
                  size="small"
                  dtype={log.deployment_type}
                />
              </TableCell>
              <TableCell>
                <Typography variant="body2">
                  {formatSize(log.file_size)}
                </Typography>
              </TableCell>
              <TableCell>
                <Typography variant="body2">
                  {formatDate(log.modified_at)}
                </Typography>
              </TableCell>
              <TableCell align="center">
                <Tooltip title="View logs">
                  <IconButton
                    size="small"
                    onClick={() => onViewLog(log)}
                    sx={{ mr: 1 }}
                  >
                    <VisibilityIcon fontSize="small" />
                  </IconButton>
                </Tooltip>
                <Tooltip title="Delete log">
                  <IconButton
                    size="small"
                    onClick={() => onDeleteLog(log)}
                    color="error"
                  >
                    <DeleteIcon fontSize="small" />
                  </IconButton>
                </Tooltip>
              </TableCell>
            </StyledTableRow>
          ))}
        </TableBody>
      </Table>
    </StyledTableContainer>
  );
};

LogsTable.propTypes = {
  logs: PropTypes.arrayOf(
    PropTypes.shape({
      deployment_id: PropTypes.string.isRequired,
      deployment_type: PropTypes.string.isRequired,
      file_size: PropTypes.number.isRequired,
      modified_at: PropTypes.string.isRequired,
      capture_active: PropTypes.bool,
      is_orphaned: PropTypes.bool,
      patterns_detected: PropTypes.shape({
        runtime_error: PropTypes.bool,
        container_crash: PropTypes.bool,
      }),
    })
  ).isRequired,
  onViewLog: PropTypes.func.isRequired,
  onDeleteLog: PropTypes.func.isRequired,
};

export default LogsTable;