import React from 'react';
import PropTypes from 'prop-types';
import {
  Box,
  TextField,
  ToggleButton,
  ToggleButtonGroup,
  InputAdornment,
} from '@mui/material';
import { styled } from '@mui/material/styles';
import SearchIcon from '@mui/icons-material/Search';
import ViewModuleIcon from '@mui/icons-material/ViewModule';
import ViewListIcon from '@mui/icons-material/ViewList';

// Styled components
const FilterContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  gap: theme.spacing(2),
  marginBottom: theme.spacing(3),
  flexWrap: 'wrap',
  alignItems: 'center',
}));

const StyledToggleButtonGroup = styled(ToggleButtonGroup)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  '& .MuiToggleButton-root': {
    color: 'rgba(255, 255, 255, 0.7)',
    border: 'none',
    padding: theme.spacing(0.75, 2), // Added padding: 6px vertical, 16px horizontal
    '&.Mui-selected': {
      backgroundColor: theme.palette.primary.main,
      color: 'white',
      '&:hover': {
        backgroundColor: theme.palette.primary.dark,
      },
    },
    '&:hover': {
      backgroundColor: 'rgba(255, 255, 255, 0.08)',
    },
  },
}));

const SearchField = styled(TextField)(({ theme }) => ({
  flexGrow: 1,
  maxWidth: 400,
  '& .MuiOutlinedInput-root': {
    backgroundColor: 'rgba(255, 255, 255, 0.03)',
    '&:hover': {
      backgroundColor: 'rgba(255, 255, 255, 0.05)',
    },
    '&.Mui-focused': {
      backgroundColor: 'rgba(255, 255, 255, 0.08)',
    },
  },
}));

const LogsFilter = ({
  selectedType,
  onTypeChange,
  searchTerm,
  onSearchChange,
  logTypes,
  viewMode = 'list',
  onViewModeChange,
}) => {
  return (
    <FilterContainer>
      {/* Log Type Filter */}
      <StyledToggleButtonGroup
        value={selectedType}
        exclusive
        onChange={(e, value) => value && onTypeChange(value)}
        size="small"
      >
        {logTypes.map((type) => (
          <ToggleButton key={type.value} value={type.value}>
            {type.label}
          </ToggleButton>
        ))}
      </StyledToggleButtonGroup>

      {/* Search Field */}
      <SearchField
        variant="outlined"
        size="small"
        placeholder="Search deployment ID..."
        value={searchTerm}
        onChange={(e) => onSearchChange(e.target.value)}
        InputProps={{
          startAdornment: (
            <InputAdornment position="start">
              <SearchIcon fontSize="small" />
            </InputAdornment>
          ),
        }}
      />

      {/* View Mode Toggle (optional, for future use) */}
      {onViewModeChange && (
        <StyledToggleButtonGroup
          value={viewMode}
          exclusive
          onChange={(e, value) => value && onViewModeChange(value)}
          size="small"
        >
          <ToggleButton value="list">
            <ViewListIcon fontSize="small" />
          </ToggleButton>
          <ToggleButton value="grid">
            <ViewModuleIcon fontSize="small" />
          </ToggleButton>
        </StyledToggleButtonGroup>
      )}
    </FilterContainer>
  );
};

LogsFilter.propTypes = {
  selectedType: PropTypes.string.isRequired,
  onTypeChange: PropTypes.func.isRequired,
  searchTerm: PropTypes.string.isRequired,
  onSearchChange: PropTypes.func.isRequired,
  logTypes: PropTypes.arrayOf(
    PropTypes.shape({
      value: PropTypes.string.isRequired,
      label: PropTypes.string.isRequired,
    })
  ).isRequired,
  viewMode: PropTypes.string,
  onViewModeChange: PropTypes.func,
};

LogsFilter.defaultProps = {
  viewMode: 'list',
  onViewModeChange: null,
};

export default LogsFilter;