import React, { useState, useEffect, useRef } from 'react';
import PropTypes from 'prop-types';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  Box,
  Typography,
  TextField,
  IconButton,
  Chip,
  Alert,
  CircularProgress,
  Tooltip,
  ToggleButton,
} from '@mui/material';
import { styled } from '@mui/material/styles';
import CloseIcon from '@mui/icons-material/Close';
import ContentCopyIcon from '@mui/icons-material/ContentCopy';
import DownloadIcon from '@mui/icons-material/Download';
import RefreshIcon from '@mui/icons-material/Refresh';
import SearchIcon from '@mui/icons-material/Search';
import WrapTextIcon from '@mui/icons-material/WrapText';
import { logsService } from '../../services/logsService';

// Styled components
const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    backgroundColor: '#1a1a1a',
    maxWidth: '90vw',
    width: '90vw',
    height: '85vh',
    display: 'flex',
    flexDirection: 'column',
  },
}));

const StyledDialogTitle = styled(DialogTitle)(({ theme }) => ({
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'center',
  padding: theme.spacing(2),
  borderBottom: '1px solid rgba(255, 255, 255, 0.1)',
}));

const TitleContent = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(2),
}));

const ToolbarActions = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(1),
}));

const StyledDialogContent = styled(DialogContent)(({ theme }) => ({
  padding: 0,
  flexGrow: 1,
  display: 'flex',
  flexDirection: 'column',
  overflow: 'hidden',
}));

const SearchBar = styled(Box)(({ theme }) => ({
  padding: theme.spacing(1, 2),
  borderBottom: '1px solid rgba(255, 255, 255, 0.1)',
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(2),
}));

const LogContent = styled(Box)(({ theme, wrapText }) => ({
  flexGrow: 1,
  overflow: 'auto',
  backgroundColor: '#0d0d0d',
  padding: theme.spacing(2),
  fontFamily: 'Monaco, Menlo, "Ubuntu Mono", monospace',
  fontSize: '0.875rem',
  lineHeight: 1.6,
  whiteSpace: wrapText ? 'pre-wrap' : 'pre',
  wordBreak: wrapText ? 'break-word' : 'normal',
  '& .highlight': {
    backgroundColor: '#FFEB3B',
    color: '#000',
  },
  '& .error-line': {
    backgroundColor: 'rgba(244, 67, 54, 0.1)',
    borderLeft: '3px solid #f44336',
    paddingLeft: theme.spacing(1),
  },
  '& .warning-line': {
    backgroundColor: 'rgba(255, 152, 0, 0.1)',
    borderLeft: '3px solid #ff9800',
    paddingLeft: theme.spacing(1),
  },
}));

const LineNumber = styled('span')(({ theme }) => ({
  color: 'rgba(255, 255, 255, 0.3)',
  marginRight: theme.spacing(2),
  userSelect: 'none',
  minWidth: '4ch',
  display: 'inline-block',
}));

const StatusBar = styled(Box)(({ theme }) => ({
  padding: theme.spacing(1, 2),
  borderTop: '1px solid rgba(255, 255, 255, 0.1)',
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'center',
  backgroundColor: 'rgba(255, 255, 255, 0.02)',
}));

const LogViewer = ({ log, open, onClose }) => {
  const [logContent, setLogContent] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [wrapText, setWrapText] = useState(true);
  const [autoRefresh, setAutoRefresh] = useState(false);
  const contentRef = useRef(null);
  const refreshIntervalRef = useRef(null);

  // Fetch log content
  const fetchLogContent = async () => {
    if (!log) return;
    
    setLoading(true);
    setError(null);
    
    try {
      const response = await logsService.getLogContent(
        log.deployment_type,
        log.deployment_id
      );
      setLogContent(response);
    } catch (err) {
      setError(err.response?.data?.detail || 'Failed to load log content');
    } finally {
      setLoading(false);
    }
  };

  // Initial load
  useEffect(() => {
    if (open && log) {
      fetchLogContent();
    }
  }, [open, log]);

  // Auto-refresh
  useEffect(() => {
    if (autoRefresh && open) {
      refreshIntervalRef.current = setInterval(() => {
        fetchLogContent();
      }, 5000); // Refresh every 5 seconds
    } else {
      if (refreshIntervalRef.current) {
        clearInterval(refreshIntervalRef.current);
        refreshIntervalRef.current = null;
      }
    }
    
    return () => {
      if (refreshIntervalRef.current) {
        clearInterval(refreshIntervalRef.current);
      }
    };
  }, [autoRefresh, open]);

  // Copy to clipboard
  const handleCopy = () => {
    if (logContent?.logs) {
      navigator.clipboard.writeText(logContent.logs.join('\n'));
    }
  };

  // Download log file
  const handleDownload = () => {
    if (logContent?.logs) {
      const blob = new Blob([logContent.logs.join('\n')], { type: 'text/plain' });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = `${log.deployment_id}.log`;
      a.click();
      URL.revokeObjectURL(url);
    }
  };

  // Highlight search terms and errors
  const formatLogLine = (line, lineNumber) => {
    const lowerLine = line.toLowerCase();
    let className = '';
    
    // Detect error/warning lines
    if (lowerLine.includes('error') || lowerLine.includes('exception') || 
        lowerLine.includes('failed') || lowerLine.includes('traceback')) {
      className = 'error-line';
    } else if (lowerLine.includes('warning') || lowerLine.includes('warn')) {
      className = 'warning-line';
    }
    
    // Highlight search term
    if (searchTerm && lowerLine.includes(searchTerm.toLowerCase())) {
      const regex = new RegExp(`(${searchTerm})`, 'gi');
      const parts = line.split(regex);
      return (
        <div key={lineNumber} className={className}>
          <LineNumber>{lineNumber}</LineNumber>
          {parts.map((part, i) => 
            regex.test(part) ? (
              <span key={i} className="highlight">{part}</span>
            ) : (
              <span key={i}>{part}</span>
            )
          )}
        </div>
      );
    }
    
    return (
      <div key={lineNumber} className={className}>
        <LineNumber>{lineNumber}</LineNumber>
        {line}
      </div>
    );
  };

  if (!log) return null;

  return (
    <StyledDialog open={open} onClose={onClose} maxWidth={false}>
      <StyledDialogTitle>
        <TitleContent>
          <Typography variant="h6">Log Viewer</Typography>
          <Chip
            label={log.deployment_type.toUpperCase()}
            size="small"
            color="primary"
          />
          <Typography variant="body2" sx={{ fontFamily: 'monospace' }}>
            {log.deployment_id}
          </Typography>
        </TitleContent>
        
        <ToolbarActions>
          <ToggleButton
            value="wrap"
            selected={wrapText}
            onChange={() => setWrapText(!wrapText)}
            size="small"
          >
            <WrapTextIcon fontSize="small" />
          </ToggleButton>
          
          <ToggleButton
            value="refresh"
            selected={autoRefresh}
            onChange={() => setAutoRefresh(!autoRefresh)}
            size="small"
          >
            <RefreshIcon fontSize="small" />
          </ToggleButton>
          
          <Tooltip title="Copy to clipboard">
            <IconButton onClick={handleCopy} size="small">
              <ContentCopyIcon fontSize="small" />
            </IconButton>
          </Tooltip>
          
          <Tooltip title="Download">
            <IconButton onClick={handleDownload} size="small">
              <DownloadIcon fontSize="small" />
            </IconButton>
          </Tooltip>
          
          <IconButton onClick={onClose} size="small">
            <CloseIcon />
          </IconButton>
        </ToolbarActions>
      </StyledDialogTitle>
      
      <StyledDialogContent>
        {/* Search Bar */}
        <SearchBar>
          <SearchIcon fontSize="small" />
          <TextField
            variant="standard"
            placeholder="Search in logs..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            fullWidth
            InputProps={{
              disableUnderline: false,
            }}
          />
        </SearchBar>
        
        {/* Log Content */}
        {loading ? (
          <Box sx={{ display: 'flex', justifyContent: 'center', p: 4 }}>
            <CircularProgress />
          </Box>
        ) : error ? (
          <Box sx={{ p: 2 }}>
            <Alert severity="error">{error}</Alert>
          </Box>
        ) : (
          <LogContent ref={contentRef} wrapText={wrapText}>
            {logContent?.logs?.map((line, index) => 
              formatLogLine(line, index + 1)
            )}
            {logContent?.is_compressed && (
              <Box sx={{ mt: 2, p: 2, backgroundColor: 'rgba(255, 152, 0, 0.1)' }}>
                <Typography variant="caption" color="warning.main">
                  Note: This log has been compressed. Some lines may have been omitted.
                </Typography>
              </Box>
            )}
          </LogContent>
        )}
        
        {/* Status Bar */}
        <StatusBar>
          <Box sx={{ display: 'flex', gap: 2 }}>
            <Typography variant="caption">
              Lines: {logContent?.total_lines || 0}
            </Typography>
            {logContent?.is_compressed && (
              <Chip label="Compressed" size="small" color="warning" />
            )}
            {logContent?.capture_active && (
              <Chip label="Active" size="small" color="success" />
            )}
          </Box>
          
          {logContent?.patterns_detected && (
            <Box sx={{ display: 'flex', gap: 1 }}>
              {Object.entries(logContent.patterns_detected)
                .filter(([, detected]) => detected)
                .map(([pattern]) => (
                  <Chip
                    key={pattern}
                    label={pattern.replace('_', ' ')}
                    size="small"
                    color="error"
                  />
                ))}
            </Box>
          )}
        </StatusBar>
      </StyledDialogContent>
    </StyledDialog>
  );
};

LogViewer.propTypes = {
  log: PropTypes.shape({
    deployment_id: PropTypes.string.isRequired,
    deployment_type: PropTypes.string.isRequired,
  }).isRequired,
  open: PropTypes.bool.isRequired,
  onClose: PropTypes.func.isRequired,
};

export default LogViewer;