import React, { useState, useEffect } from 'react';
import {
  Container,
  Typography,
  // Paper, // Removed - storage stats no longer displayed
  Box,
  Chip,
  CircularProgress,
  Alert,
  IconButton,
  Tooltip,
  // LinearProgress, // Removed - storage stats no longer displayed
} from '@mui/material';
import { styled } from '@mui/material/styles';
import RefreshIcon from '@mui/icons-material/Refresh';
import DeleteSweepIcon from '@mui/icons-material/DeleteSweep';
// import StorageIcon from '@mui/icons-material/Storage'; // Removed - storage stats no longer displayed
import WarningAmberIcon from '@mui/icons-material/WarningAmber';

import LogsFilter from './LogsFilter';
import LogsTable from './LogsTable';
import LogViewer from './LogViewer';
import { logsService } from '../../services/logsService';

// Styled components
const StyledContainer = styled(Container)(({ theme }) => ({
  paddingTop: theme.spacing(3),
  paddingBottom: theme.spacing(3),
  minHeight: 'calc(100vh - 100px)',
}));

const PageHeader = styled(Box)(({ theme }) => ({
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'center',
  marginBottom: theme.spacing(3),
  padding: theme.spacing(2),
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: theme.spacing(1.5),
  border: '1px solid rgba(255, 255, 255, 0.08)',
}));

const HeaderLeft = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(2),
}));

const HeaderRight = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(2),
}));

// Storage stats components - Removed per user request
// const StatsCard = styled(Paper)(({ theme }) => ({
//   padding: theme.spacing(2),
//   backgroundColor: 'rgba(255, 255, 255, 0.03)',
//   borderRadius: theme.spacing(1.5),
//   border: '1px solid rgba(255, 255, 255, 0.08)',
//   marginBottom: theme.spacing(3),
// }));

// const StatItem = styled(Box)(({ theme }) => ({
//   display: 'flex',
//   alignItems: 'center',
//   gap: theme.spacing(1),
//   marginBottom: theme.spacing(1),
// }));

const DeploymentLogs = () => {
  const [logs, setLogs] = useState([]);
  const [loading, setLoading] = useState(true);
  const [selectedType, setSelectedType] = useState('all');
  const [selectedLog, setSelectedLog] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  // const [stats, setStats] = useState(null); // Removed - storage stats no longer displayed
  const [error, setError] = useState(null);
  const [refreshing, setRefreshing] = useState(false);

  // Log type definitions with icons
  const logTypes = [
    { value: 'all', label: 'All' },
    { value: 'vllm', label: 'vLLM' },
    { value: 'llamacpp', label: 'llama.cpp' },
    // TODO: Uncomment when these deployment types are implemented
    // { value: 'mlx', label: 'MLX' },
    // { value: 'ampere', label: 'Ampere' },
    // { value: 'app_garden', label: 'App Garden' },
    // { value: 'tool_shed', label: 'Tool Shed' },
  ];

  // Fetch logs based on selected type
  const fetchLogs = async (showLoading = true) => {
    if (showLoading) setLoading(true);
    setError(null);
    
    try {
      const response = await logsService.listLogs(selectedType);
      setLogs(response.logs || []);
    } catch (err) {
      setError(err.response?.data?.detail || 'Failed to load logs');
      setLogs([]);
    } finally {
      setLoading(false);
      setRefreshing(false);
    }
  };

  // Fetch storage statistics - Removed per user request
  // const fetchStats = async () => {
  //   try {
  //     const response = await logsService.getStats();
  //     setStats(response);
  //   } catch (err) {
  //     console.error('Failed to fetch stats:', err);
  //   }
  // };

  // Initial load and refresh on type change
  useEffect(() => {
    fetchLogs();
    // fetchStats(); // Removed - storage stats no longer displayed
  }, [selectedType]);

  // Auto-refresh every 30 seconds
  useEffect(() => {
    const interval = setInterval(() => {
      fetchLogs(false); // Don't show loading spinner for auto-refresh
    }, 30000);
    
    return () => clearInterval(interval);
  }, [selectedType]);

  // Handle manual refresh
  const handleRefresh = () => {
    setRefreshing(true);
    fetchLogs(false);
    // fetchStats(); // Removed - storage stats no longer displayed
  };

  // Handle log selection for viewing
  const handleViewLog = async (log) => {
    setSelectedLog(log);
  };

  // Handle log deletion
  const handleDeleteLog = async (log) => {
    if (window.confirm(`Delete log for ${log.deployment_id}?`)) {
      try {
        await logsService.deleteLog(log.deployment_type, log.deployment_id);
        fetchLogs(false); // Refresh the list
        // fetchStats(); // Removed - storage stats no longer displayed
      } catch (err) {
        alert('Failed to delete log: ' + (err.response?.data?.detail || err.message));
      }
    }
  };

  // Filter logs based on search term
  const filteredLogs = logs.filter(log => {
    if (!searchTerm) return true;
    return log.deployment_id.toLowerCase().includes(searchTerm.toLowerCase());
  });

  // Count orphaned logs
  const orphanedCount = logs.filter(log => log.is_orphaned).length;

  // Calculate storage usage percentage - Removed since storage stats no longer displayed
  // const storagePercentage = stats ? (stats.total_size_bytes / (1024 * 1024 * 1024)) * 10 : 0; // Assuming 10GB limit

  return (
    <StyledContainer maxWidth="xl">
      {/* Page Header */}
      <PageHeader>
        <HeaderLeft>
          <Typography variant="h4" sx={{ fontWeight: 600 }}>
            Deployment Logs
          </Typography>
          <Chip 
            label={`${filteredLogs.length} logs`} 
            color="primary" 
            size="small" 
          />
          {orphanedCount > 0 && (
            <Chip 
              label={`${orphanedCount} orphaned`} 
              color="warning" 
              size="small"
              icon={<WarningAmberIcon />}
            />
          )}
        </HeaderLeft>
        
        <HeaderRight>
          <Tooltip title="Refresh">
            <IconButton onClick={handleRefresh} disabled={refreshing}>
              <RefreshIcon className={refreshing ? 'rotating' : ''} />
            </IconButton>
          </Tooltip>
          
          <Tooltip title="Clean up old logs">
            <IconButton disabled>
              <DeleteSweepIcon />
            </IconButton>
          </Tooltip>
        </HeaderRight>
      </PageHeader>

      {/* Storage Statistics - Removed per user request */}
      {/* {stats && (
        <StatsCard elevation={0}>
          <StatItem>
            <StorageIcon color="primary" />
            <Typography variant="body2">
              Storage: {(stats.total_size_bytes / (1024 * 1024)).toFixed(1)} MB used
            </Typography>
          </StatItem>
          <LinearProgress 
            variant="determinate" 
            value={storagePercentage} 
            sx={{ height: 8, borderRadius: 4 }}
          />
          <Box sx={{ display: 'flex', gap: 2, mt: 2 }}>
            <Typography variant="caption" color="text.secondary">
              Total logs: {stats.total_logs}
            </Typography>
            <Typography variant="caption" color="text.secondary">
              Orphaned: {stats.orphaned_count}
            </Typography>
          </Box>
        </StatsCard>
      )} */}

      {/* Filters */}
      <LogsFilter
        selectedType={selectedType}
        onTypeChange={setSelectedType}
        searchTerm={searchTerm}
        onSearchChange={setSearchTerm}
        logTypes={logTypes}
      />

      {/* Error Display */}
      {error && (
        <Alert severity="error" sx={{ mb: 3 }}>
          {error}
        </Alert>
      )}

      {/* Loading State */}
      {loading ? (
        <Box sx={{ display: 'flex', justifyContent: 'center', py: 8 }}>
          <CircularProgress />
        </Box>
      ) : (
        /* Logs Table */
        <LogsTable
          logs={filteredLogs}
          onViewLog={handleViewLog}
          onDeleteLog={handleDeleteLog}
        />
      )}

      {/* Log Viewer Modal */}
      {selectedLog && (
        <LogViewer
          log={selectedLog}
          open={Boolean(selectedLog)}
          onClose={() => setSelectedLog(null)}
        />
      )}
    </StyledContainer>
  );
};

export default DeploymentLogs;