import React, { useState, useEffect, useContext } from 'react';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Grid,
  Card,
  CardContent,
  CardActions,
  Button,
  Switch,
  FormControlLabel,
  IconButton,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  DialogContentText
} from '@mui/material';
import { useNavigate } from 'react-router-dom';
import ModelTrainingIcon from '@mui/icons-material/ModelTraining';
import AppsIcon from '@mui/icons-material/Apps';
import CodeIcon from '@mui/icons-material/Code';
import ApiIcon from '@mui/icons-material/Api';
import HandymanIcon from '@mui/icons-material/Handyman';
import InfoIcon from '@mui/icons-material/Info';
import { useTour } from '../../contexts/TourContext';
import { AuthContext } from '../../context/AuthContext';
import News from '../news/News.jsx';

const WelcomeSection = styled(Box)(({ theme }) => ({
  textAlign: 'center',
  marginBottom: theme.spacing(6),
  padding: theme.spacing(4),
  background: 'linear-gradient(135deg, rgba(0, 192, 127, 0.1) 0%, rgba(255, 255, 255, 0.05) 100%)',
  borderRadius: theme.spacing(2),
  border: '1px solid rgba(255, 255, 255, 0.1)',
}));

const ActionCard = styled(Card)(({ theme }) => ({
  height: '100%',
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  border: '1px solid rgba(255, 255, 255, 0.1)',
  borderRadius: theme.spacing(2),
  transition: 'all 0.3s ease',
  display: 'flex',
  flexDirection: 'column',
  '&:hover': {
    transform: 'translateY(-4px)',
    boxShadow: '0 8px 32px rgba(0, 192, 127, 0.2)',
    borderColor: theme.palette.primary.main,
  }
}));

const IconWrapper = styled(Box)(({ theme }) => ({
  width: 80,
  height: 80,
  borderRadius: '50%',
  background: 'linear-gradient(135deg, rgba(0, 192, 127, 0.2) 0%, rgba(0, 192, 127, 0.1) 100%)',
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'center',
  margin: '0 auto',
  marginBottom: theme.spacing(2),
  '& svg': {
    fontSize: 40,
    color: theme.palette.primary.main,
  }
}));

const DeveloperModeToggle = styled(Box)(({ theme }) => ({
  position: 'fixed',
  bottom: theme.spacing(3),
  right: theme.spacing(3),
  backgroundColor: 'rgba(0, 0, 0, 0.8)',
  padding: theme.spacing(2),
  borderRadius: theme.spacing(2),
  border: '1px solid rgba(255, 255, 255, 0.1)',
  backdropFilter: 'blur(10px)',
  zIndex: 1000,
}));

const Homepage = () => {
  const navigate = useNavigate();
  const { startTour } = useTour();
  const { user } = useContext(AuthContext);
  const [advancedMode, setAdvancedMode] = useState(false);
  const [showDisableDialog, setShowDisableDialog] = useState(false);

  // Check if user has admin role
  const isAdmin = user && Array.isArray(user.roles) && user.roles.includes('admin');

  useEffect(() => {
    // Check if advanced mode is enabled in localStorage
    const savedMode = localStorage.getItem('advancedDeveloperMode');
    setAdvancedMode(savedMode === 'true');

    // Start homepage tour if in novice mode (only for non-admins or when advanced mode is off)
    if (savedMode !== 'true') {
      const timer = setTimeout(() => {
        startTour('homepage');
      }, 500);
      return () => clearTimeout(timer);
    }
  }, [startTour]);

  const handleAdvancedModeToggle = (event) => {
    const newValue = event.target.checked;
    if (!newValue && advancedMode) {
      // If turning off, show confirmation dialog
      setShowDisableDialog(true);
    } else {
      setAdvancedMode(newValue);
      localStorage.setItem('advancedDeveloperMode', newValue.toString());
      // Reload page to apply changes
      window.location.reload();
    }
  };

  const handleDisableConfirm = () => {
    setAdvancedMode(false);
    localStorage.setItem('advancedDeveloperMode', 'false');
    setShowDisableDialog(false);
    // Reload page to apply changes
    window.location.reload();
  };

  // Features shown to regular (non-admin) users
  const userFeatures = [
    {
      title: 'Models',
      description: 'Search for open models, download, and deploy them',
      icon: <ModelTrainingIcon />,
      action: () => navigate('/models'),
      buttonText: 'Browse Models'
    },
    {
      title: 'App Garden',
      description: 'Deploy apps that work with Kamiwaza models out of the box',
      icon: <AppsIcon />,
      action: () => navigate('/apps'),
      buttonText: 'Explore Apps'
    },
    {
      title: 'Tool Shed',
      description: 'Deploy tool servers for AI agents and workflows',
      icon: <HandymanIcon />,
      action: () => navigate('/tool-shed'),
      buttonText: 'Explore Tools'
    }
  ];

  // Features shown to admin users (current full set)
  const adminFeatures = [
    {
      title: 'Models',
      description: 'Search for open models, download, and deploy them',
      icon: <ModelTrainingIcon />,
      action: () => navigate('/models'),
      buttonText: 'Browse Models'
    },
    {
      title: 'App Garden',
      description: 'Deploy apps that work with Kamiwaza models out of the box',
      icon: <AppsIcon />,
      action: () => navigate('/apps'),
      buttonText: 'Explore Apps'
    },
    {
      title: 'Notebooks',
      description: 'Write code in a Jupyter notebook environment',
      icon: <CodeIcon />,
      action: () => window.open(`${window.location.origin}/lab`, '_blank'),
      buttonText: 'Open Notebooks'
    },
    {
      title: 'API Docs',
      description: 'Peruse Kamiwaza\'s full REST spec',
      icon: <ApiIcon />,
      action: () => window.open(`${window.location.origin}/api/docs`, '_blank'),
      buttonText: 'View API Docs'
    }
  ];

  // Select features based on user role
  const features = isAdmin ? adminFeatures : userFeatures;

  return (
    <Box sx={{ maxWidth: '100%', mx: 'auto', py: 3, px: 2 }}>
      <WelcomeSection data-tour="welcome-hero">
        <Typography variant="h2" gutterBottom sx={{ fontWeight: 700, color: 'primary.main' }}>
          Welcome to Kamiwaza
        </Typography>
        <Typography variant="h5" sx={{ color: 'text.secondary', mt: 2, maxWidth: 800, mx: 'auto' }}>
          Intelligence. Everywhere.
        </Typography>
      </WelcomeSection>

      <Grid container spacing={4}>
        {features.map((feature, index) => (
          <Grid item xs={12} sm={6} md={isAdmin ? 3 : 4} key={index}>
            <ActionCard data-tour={`${feature.title.toLowerCase().replace(' ', '-')}-card`}>
              <CardContent sx={{ flexGrow: 1, textAlign: 'center', pt: 4 }}>
                <IconWrapper>
                  {feature.icon}
                </IconWrapper>
                <Typography variant="h5" component="h3" gutterBottom sx={{ fontWeight: 600 }}>
                  {feature.title}
                </Typography>
                <Typography variant="body1" sx={{ color: 'text.secondary', mb: 2 }}>
                  {feature.description}
                </Typography>
              </CardContent>
              <CardActions sx={{ justifyContent: 'center', pb: 3 }}>
                <Button 
                  variant="contained" 
                  color="primary" 
                  onClick={feature.action}
                  size="large"
                  sx={{ px: 4 }}
                >
                  {feature.buttonText}
                </Button>
              </CardActions>
            </ActionCard>
          </Grid>
        ))}
      </Grid>

      <News variant="preview" />

      {/* Advanced Developer Mode toggle - only shown to admin users */}
      {isAdmin && (
        <>
          <DeveloperModeToggle data-tour="mode-toggle">
            <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
              <FormControlLabel
                control={
                  <Switch
                    checked={advancedMode}
                    onChange={handleAdvancedModeToggle}
                    color="primary"
                  />
                }
                label={
                  <Typography variant="body2">
                    {advancedMode ? 'Advanced Developer Mode: On' : 'Enable Advanced Developer Mode'}
                  </Typography>
                }
              />
              <IconButton
                size="small"
                sx={{ color: 'text.secondary' }}
                onClick={() => {
                  // Show info about advanced mode
                  alert('Advanced Developer Mode provides access to all features including model hub search, detailed configurations, and advanced deployment options.');
                }}
              >
                <InfoIcon fontSize="small" />
              </IconButton>
            </Box>
          </DeveloperModeToggle>

          <Dialog
            open={showDisableDialog}
            onClose={() => setShowDisableDialog(false)}
          >
            <DialogTitle>Disable Advanced Developer Mode?</DialogTitle>
            <DialogContent>
              <DialogContentText>
                This will switch to the simplified interface designed for novice users.
                Some advanced features will be hidden. Do you want to continue?
              </DialogContentText>
            </DialogContent>
            <DialogActions>
              <Button onClick={() => setShowDisableDialog(false)}>Cancel</Button>
              <Button onClick={handleDisableConfirm} color="primary" variant="contained">
                Yes, Disable
              </Button>
            </DialogActions>
          </Dialog>
        </>
      )}
    </Box>
  );
};

export default Homepage;