import React, { useState, useEffect } from 'react';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Container,
  Button,
  Alert,
  CircularProgress,
  Tabs,
  Tab,
  Card,
  CardContent,
  Grid,
  Chip,
  Badge,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Switch,
  FormControlLabel
} from '@mui/material';
import {
  Download as DownloadIcon,
  Update as UpdateIcon,
  Refresh as RefreshIcon,
  Apps as AppsIcon,
  Build as ToolIcon,
} from '@mui/icons-material';
import AppTemplateCard from '../apps/AppTemplateCard';
import ToolTemplateCard from '../tool/ToolTemplateCard';
import AppDeployModal from '../apps/AppDeployModal';
import ToolDeployModal from '../tool/ToolDeployModal';
import api from '../../utils/api';

// Styled components
const PageContainer = styled(Container)(({ theme }) => ({
  paddingTop: theme.spacing(2),
  paddingBottom: theme.spacing(4),
  maxWidth: '1400px',
}));

const PageTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '2.5rem',
  fontWeight: 700,
  marginBottom: theme.spacing(1),
  background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  backgroundClip: 'text',
  textFillColor: 'transparent',
  WebkitBackgroundClip: 'text',
  WebkitTextFillColor: 'transparent',
}));

const PageSubtitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontSize: '1.1rem',
  marginBottom: theme.spacing(3),
}));

const StatusCard = styled(Card)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  background: `linear-gradient(135deg, ${theme.palette.primary.light}15, ${theme.palette.secondary.light}15)`,
  border: `1px solid ${theme.palette.divider}`,
}));

const ActionButton = styled(Button)(({ theme }) => ({
  margin: theme.spacing(0, 1, 1, 0),
  borderRadius: 8,
  fontWeight: 600,
}));

const ImportButton = styled(ActionButton)(({ theme }) => ({
  background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  color: 'white',
  '&:hover': {
    background: 'linear-gradient(90deg, #00a66b, #00a089)',
  },
  '&:disabled': {
    background: theme.palette.grey[300],
    color: theme.palette.grey[500],
  },
}));

const UpdateButton = styled(ActionButton)(({ theme }) => ({
  background: 'linear-gradient(90deg, #ff6b35, #f7931e)',
  color: 'white',
  '&:hover': {
    background: 'linear-gradient(90deg, #e55a2b, #de831a)',
  },
  '&:disabled': {
    background: theme.palette.grey[300],
    color: theme.palette.grey[500],
  },
}));

const UnifiedTemplateManager = () => {
  // State management
  const [currentTab, setCurrentTab] = useState(0);
  const [templates, setTemplates] = useState([]);
  const [deployments, setDeployments] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [success, setSuccess] = useState(null);
  
  // Remote status state
  const [remoteStatus, setRemoteStatus] = useState({
    apps: { new_templates_available: 0, updates_available: 0, updateable_templates: [] },
    tools: { new_templates_available: 0, updates_available: 0, updateable_templates: [] }
  });
  
  // UI state
  const [showStoppedDeployments, setShowStoppedDeployments] = useState(false);
  const [importLoading, setImportLoading] = useState(false);
  const [updateLoading, setUpdateLoading] = useState(false);
  const [selectedTemplate, setSelectedTemplate] = useState(null);
  const [showAppDeployModal, setShowAppDeployModal] = useState(false);
  const [showToolDeployModal, setShowToolDeployModal] = useState(false);
  const [conflictDialog, setConflictDialog] = useState({ open: false, error: null });

  // Load data
  useEffect(() => {
    loadTemplates();
    loadDeployments();
    loadRemoteStatus();
  }, []);

  const loadTemplates = async () => {
    setLoading(true);
    try {
      const response = await api.get('/garden/apps/app_templates');
      setTemplates(response.data);
      setError(null);
    } catch (err) {
      setError('Failed to load templates');
      console.error('Template loading error:', err);
    } finally {
      setLoading(false);
    }
  };

  const loadDeployments = async () => {
    try {
      const [appDeployments, toolDeployments] = await Promise.all([
        api.get('/garden/apps/deployments'),
        api.get('/garden/tool/deployments')
      ]);
      
      setDeployments([
        ...appDeployments.data.map(d => ({ ...d, type: 'app' })),
        ...toolDeployments.data.map(d => ({ ...d, type: 'tool' }))
      ]);
    } catch (err) {
      console.error('Deployments loading error:', err);
    }
  };

  const loadRemoteStatus = async () => {
    try {
      const [appStatus, toolStatus] = await Promise.all([
        api.get('/garden/apps/remote/status'),
        api.get('/garden/tool/remote/status')
      ]);
      
      setRemoteStatus({
        apps: appStatus.data,
        tools: toolStatus.data
      });
    } catch (err) {
      console.error('Remote status loading error:', err);
    }
  };

  const handleImportNew = async (templateType) => {
    setImportLoading(true);
    try {
      const endpoint = templateType === 'apps' 
        ? '/garden/apps/remote/sync' 
        : '/garden/tool/remote/sync';
      
      const response = await api.post(endpoint);
      
      if (response.data.success) {
        setSuccess(`Successfully imported ${response.data.imported_count} new ${templateType} templates`);
        await Promise.all([loadTemplates(), loadRemoteStatus()]);
      } else {
        setError(`Import failed: ${response.data.error}`);
      }
    } catch (err) {
      setError(`Failed to import ${templateType} templates`);
      console.error('Import error:', err);
    } finally {
      setImportLoading(false);
    }
  };

  const handleUpdateTemplate = async (templateId) => {
    setUpdateLoading(true);
    try {
      const response = await api.put(`/garden/apps/app_templates/${templateId}`);
      
      if (response.data.success) {
        setSuccess(`Template updated successfully: ${response.data.message}`);
        await Promise.all([loadTemplates(), loadRemoteStatus()]);
      } else {
        setError(`Update failed: ${response.data.error}`);
      }
    } catch (err) {
      if (err.response?.status === 409) {
        // Handle active deployment conflict
        setConflictDialog({
          open: true,
          error: err.response.data
        });
      } else {
        setError('Failed to update template');
        console.error('Update error:', err);
      }
    } finally {
      setUpdateLoading(false);
    }
  };

  const handleUpdateAll = async (templateType) => {
    const statusKey = templateType === 'apps' ? 'apps' : 'tools';
    const updateableTemplates = remoteStatus[statusKey].updateable_templates || [];
    
    setUpdateLoading(true);
    let successCount = 0;
    let errorCount = 0;
    
    for (const template of updateableTemplates) {
      try {
        await api.put(`/garden/apps/app_templates/${template.id}`);
        successCount++;
      } catch (err) {
        errorCount++;
        console.error(`Failed to update ${template.name}:`, err);
      }
    }
    
    if (successCount > 0) {
      setSuccess(`Updated ${successCount} templates successfully`);
    }
    if (errorCount > 0) {
      setError(`Failed to update ${errorCount} templates`);
    }
    
    await Promise.all([loadTemplates(), loadRemoteStatus()]);
    setUpdateLoading(false);
  };

  const handleDeleteTemplate = async (templateId) => {
    try {
      const response = await api.delete(`/garden/apps/app_templates/${templateId}`);
      
      if (response.data.success) {
        setSuccess(`Template deleted successfully`);
        if (response.data.deleted_deployments > 0) {
          setSuccess(prev => prev + ` (${response.data.deleted_deployments} stopped deployments cleaned up)`);
        }
        await Promise.all([loadTemplates(), loadDeployments()]);
      } else {
        setError(`Delete failed: ${response.data.error}`);
      }
    } catch (err) {
      if (err.response?.status === 409) {
        setConflictDialog({
          open: true,
          error: err.response.data
        });
      } else {
        setError('Failed to delete template');
        console.error('Delete error:', err);
      }
    }
  };

  const handleDeploy = (template) => {
    setSelectedTemplate(template);
    
    // Determine if it's an app or tool template
    const isToolTemplate = template.name.startsWith('tool-') || template.name.startsWith('mcp-');
    
    if (isToolTemplate) {
      setShowToolDeployModal(true);
    } else {
      setShowAppDeployModal(true);
    }
  };

  const filterTemplates = (type) => {
    if (type === 'apps') {
      return templates.filter(t => !t.name.startsWith('tool-') && !t.name.startsWith('mcp-'));
    } else {
      return templates.filter(t => t.name.startsWith('tool-') || t.name.startsWith('mcp-'));
    }
  };

  const filterDeployments = (type) => {
    if (!showStoppedDeployments) {
      return deployments.filter(d => d.type === type && d.status !== 'STOPPED' && d.status !== 'FAILED');
    }
    return deployments.filter(d => d.type === type);
  };

  const getCurrentStatus = () => {
    const statusKey = currentTab === 0 ? 'apps' : 'tools';
    return remoteStatus[statusKey] || { new_templates_available: 0, updates_available: 0 };
  };

  const getCurrentTemplateType = () => currentTab === 0 ? 'apps' : 'tools';

  return (
    <PageContainer>
      <PageTitle>Template Manager</PageTitle>
      <PageSubtitle>
        Manage application and tool templates with unified import, update, and deployment controls
      </PageSubtitle>

      {error && (
        <Alert severity="error" onClose={() => setError(null)} sx={{ mb: 3 }}>
          {error}
        </Alert>
      )}

      {success && (
        <Alert severity="success" onClose={() => setSuccess(null)} sx={{ mb: 3 }}>
          {success}
        </Alert>
      )}

      {/* Status Card */}
      <StatusCard>
        <CardContent>
          <Typography variant="h6" gutterBottom>
            Remote Template Status
          </Typography>
          
          <Grid container spacing={2}>
            <Grid item xs={12} sm={6}>
              <Box display="flex" alignItems="center" gap={2}>
                <AppsIcon color="primary" />
                <Typography variant="body2">
                  Apps: {remoteStatus.apps.new_templates_available} new, {remoteStatus.apps.updates_available} updates available
                </Typography>
              </Box>
            </Grid>
            <Grid item xs={12} sm={6}>
              <Box display="flex" alignItems="center" gap={2}>
                <ToolIcon color="secondary" />
                <Typography variant="body2">
                  Tools: {remoteStatus.tools.new_templates_available} new, {remoteStatus.tools.updates_available} updates available
                </Typography>
              </Box>
            </Grid>
          </Grid>

          <Box mt={2} display="flex" flexWrap="wrap" gap={1}>
            <ImportButton
              startIcon={<DownloadIcon />}
              onClick={() => handleImportNew(getCurrentTemplateType())}
              disabled={importLoading || getCurrentStatus().new_templates_available === 0}
            >
              {importLoading ? <CircularProgress size={20} /> : `Import New ${getCurrentTemplateType()}`}
              {getCurrentStatus().new_templates_available > 0 && (
                <Badge badgeContent={getCurrentStatus().new_templates_available} color="error" sx={{ ml: 1 }} />
              )}
            </ImportButton>

            <UpdateButton
              startIcon={<UpdateIcon />}
              onClick={() => handleUpdateAll(getCurrentTemplateType())}
              disabled={updateLoading || getCurrentStatus().updates_available === 0}
            >
              {updateLoading ? <CircularProgress size={20} /> : `Update All ${getCurrentTemplateType()}`}
              {getCurrentStatus().updates_available > 0 && (
                <Badge badgeContent={getCurrentStatus().updates_available} color="warning" sx={{ ml: 1 }} />
              )}
            </UpdateButton>

            <ActionButton
              startIcon={<RefreshIcon />}
              onClick={loadRemoteStatus}
              variant="outlined"
            >
              Refresh Status
            </ActionButton>
          </Box>
        </CardContent>
      </StatusCard>

      {/* Template Type Tabs */}
      <Box sx={{ borderBottom: 1, borderColor: 'divider', mb: 3 }}>
        <Tabs value={currentTab} onChange={(e, newValue) => setCurrentTab(newValue)}>
          <Tab 
            label={
              <Box display="flex" alignItems="center" gap={1}>
                <AppsIcon />
                Apps
                <Chip 
                  label={filterTemplates('apps').length} 
                  size="small" 
                  color="primary" 
                />
              </Box>
            } 
          />
          <Tab 
            label={
              <Box display="flex" alignItems="center" gap={1}>
                <ToolIcon />
                Tools
                <Chip 
                  label={filterTemplates('tools').length} 
                  size="small" 
                  color="secondary" 
                />
              </Box>
            } 
          />
        </Tabs>
      </Box>

      {/* Show Stopped Deployments Toggle */}
      <Box mb={2}>
        <FormControlLabel
          control={
            <Switch
              checked={showStoppedDeployments}
              onChange={(e) => setShowStoppedDeployments(e.target.checked)}
            />
          }
          label="Show stopped deployments"
        />
      </Box>

      {/* Template Grid */}
      {loading ? (
        <Box display="flex" justifyContent="center" py={4}>
          <CircularProgress />
        </Box>
      ) : (
        <Grid container spacing={3}>
          {filterTemplates(getCurrentTemplateType()).map((template) => {
            const isToolTemplate = template.name.startsWith('tool-') || template.name.startsWith('mcp-');
            const templateDeployments = filterDeployments(isToolTemplate ? 'tool' : 'app')
              .filter(d => d.template_id === template.id);
            
            // Check if template has updates available
            const hasUpdate = getCurrentStatus().updateable_templates?.some(t => t.id === template.id);

            return (
              <Grid item xs={12} sm={6} md={4} key={template.id}>
                {isToolTemplate ? (
                  <ToolTemplateCard
                    template={template}
                    deployments={templateDeployments}
                    onDeploy={() => handleDeploy(template)}
                    onDelete={() => handleDeleteTemplate(template.id)}
                    onUpdate={hasUpdate ? () => handleUpdateTemplate(template.id) : null}
                    hasUpdate={hasUpdate}
                  />
                ) : (
                  <AppTemplateCard
                    template={template}
                    deployments={templateDeployments}
                    onDeploy={() => handleDeploy(template)}
                    onDelete={() => handleDeleteTemplate(template.id)}
                    onUpdate={hasUpdate ? () => handleUpdateTemplate(template.id) : null}
                    hasUpdate={hasUpdate}
                  />
                )}
              </Grid>
            );
          })}
        </Grid>
      )}

      {/* Deploy Modals */}
      {showAppDeployModal && selectedTemplate && (
        <AppDeployModal
          open={showAppDeployModal}
          onClose={() => {
            setShowAppDeployModal(false);
            setSelectedTemplate(null);
          }}
          template={selectedTemplate}
          onDeploymentChange={loadDeployments}
        />
      )}

      {showToolDeployModal && selectedTemplate && (
        <ToolDeployModal
          open={showToolDeployModal}
          onClose={() => {
            setShowToolDeployModal(false);
            setSelectedTemplate(null);
          }}
          template={selectedTemplate}
          onDeploymentChange={loadDeployments}
        />
      )}

      {/* Conflict Dialog */}
      <Dialog
        open={conflictDialog.open}
        onClose={() => setConflictDialog({ open: false, error: null })}
        maxWidth="sm"
        fullWidth
      >
        <DialogTitle>Active Deployments Found</DialogTitle>
        <DialogContent>
          <Typography paragraph>
            {conflictDialog.error?.error}
          </Typography>
          <Typography variant="subtitle2" gutterBottom>
            Active deployments:
          </Typography>
          <Box component="ul" sx={{ pl: 2 }}>
            {conflictDialog.error?.active_deployments?.map((name, index) => (
              <Typography component="li" key={index} variant="body2">
                {name}
              </Typography>
            ))}
          </Box>
          <Typography variant="body2" color="text.secondary" sx={{ mt: 2 }}>
            Please stop these deployments before updating or deleting the template.
          </Typography>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setConflictDialog({ open: false, error: null })}>
            Close
          </Button>
        </DialogActions>
      </Dialog>
    </PageContainer>
  );
};

export default UnifiedTemplateManager;