import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Link } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  Table, 
  TableHead, 
  TableBody, 
  TableRow, 
  TableCell,
  TableContainer,
  Chip,
  Tooltip
} from '@mui/material';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import { BASE_URL } from '../../const';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';

// Styled components matching the Models tables
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  marginBottom: theme.spacing(4),
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
  width: '100%',
  maxWidth: '1200px',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.08)',
  },
  tableLayout: 'auto',
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '0.95rem',
  padding: theme.spacing(1.5, 2),
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  padding: theme.spacing(1.8, 2),
  fontSize: '0.9rem',
  whiteSpace: 'nowrap',
  overflow: 'hidden',
  textOverflow: 'ellipsis',
}));

const StyledTableRow = styled(TableRow)(({ theme }) => ({
  transition: 'background-color 0.2s',
  '&:nth-of-type(odd)': {
    backgroundColor: 'rgba(255, 255, 255, 0.02)',
  },
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
  },
  '&.empty-row': {
    height: 100,
    '& td': {
      textAlign: 'center',
      color: theme.palette.text.secondary,
    }
  }
}));

const ModelLink = styled(Link)(({ theme }) => ({
  color: theme.palette.primary.main,
  textDecoration: 'none',
  fontWeight: 500,
  transition: 'all 0.2s',
  display: 'inline-block',
  borderBottom: '1px solid transparent',
  '&:hover': {
    color: theme.palette.primary.light,
    borderBottomColor: theme.palette.primary.light,
    transform: 'translateY(-1px)',
  },
}));

const PlatformChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(0, 192, 127, 0.1)',
  color: theme.palette.text.primary,
  fontSize: '0.75rem',
  fontWeight: 500,
}));

const EnvironmentChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(30, 136, 229, 0.1)',
  color: theme.palette.text.primary,
  fontSize: '0.75rem',
  fontWeight: 500,
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const DatasetList = () => {
    const [datasets, setDatasets] = useState([]);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);

    // Format large numbers with abbreviations
    const formatRowCount = (count) => {
        if (!count) return 'N/A';
        
        const num = parseInt(count, 10);
        
        // If less than 1 million, just return the formatted number
        if (num < 1000000) {
            return num.toLocaleString();
        }
        
        // For large numbers, abbreviate with suffix and show full number in parentheses
        let formatted;
        if (num >= 1000000000) {
            formatted = `${(num / 1000000000).toFixed(1)}B`;
        } else if (num >= 1000000) {
            formatted = `${(num / 1000000).toFixed(1)}M`;
        }
        
        return `${formatted} (${num.toLocaleString()})`;
    };

    useEffect(() => {
        const fetchDatasets = async () => {
            setLoading(true);
            try {
                const uri = `${BASE_URL}/catalog/datasets/`;
                const response = await axios.get(uri);
                if (response.status === 200) {
                    setDatasets(response.data);
                } else {
                    throw new Error(`Failed to fetch datasets from ${uri}`);
                }
            } catch (err) {
                setError(err);
            } finally {
                setLoading(false);
            }
        };

        fetchDatasets();
    }, []);

    return (
        <ContentContainer>
            <SectionTitle variant="h4">
                Dataset List
                <Tooltip title="Datasets available in your Kamiwaza catalog">
                    <InfoIcon />
                </Tooltip>
            </SectionTitle>
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={error.message} />}
            
            <StyledTableContainer component={Paper}>
                <StyledTable>
                    <TableHead>
                        <TableRow>
                            <TableHeaderCell>Platform</TableHeaderCell>
                            <TableHeaderCell>ID</TableHeaderCell>
                            <TableHeaderCell>Environment</TableHeaderCell>
                            <TableHeaderCell>Row Count</TableHeaderCell>
                        </TableRow>
                    </TableHead>
                    <TableBody>
                        {datasets.length > 0 ? (
                            datasets.map((dataset, index) => (
                                <StyledTableRow key={dataset.urn || dataset.id || `dataset-${index}`}>
                                    <TableBodyCell>
                                        <PlatformChip 
                                            label={dataset.platform || 'file'} 
                                            size="small" 
                                            variant="outlined"
                                        />
                                    </TableBodyCell>
                                    <TableBodyCell>
                                        <ModelLink to={`/catalog/datasets/${encodeURIComponent(dataset.urn)}`}>
                                            {dataset.name}
                                        </ModelLink>
                                    </TableBodyCell>
                                    <TableBodyCell>
                                        <EnvironmentChip 
                                            label={dataset.environment} 
                                            size="small"
                                            variant="outlined" 
                                        />
                                    </TableBodyCell>
                                    <TableBodyCell>
                                        {formatRowCount(dataset.row_count)}
                                    </TableBodyCell>
                                </StyledTableRow>
                            ))
                        ) : (
                            <StyledTableRow className="empty-row">
                                <TableBodyCell colSpan={4}>
                                    No datasets available.
                                </TableBodyCell>
                            </StyledTableRow>
                        )}
                    </TableBody>
                </StyledTable>
            </StyledTableContainer>
        </ContentContainer>
    );
};

export default DatasetList;
