import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { useParams, Link } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  Button,
  Grid,
  Chip,
  Divider,
  List,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow
} from '@mui/material';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import FolderIcon from '@mui/icons-material/Folder';
import LabelIcon from '@mui/icons-material/Label';
import StorageIcon from '@mui/icons-material/Storage';
import SchemaIcon from '@mui/icons-material/Schema';
import PublicIcon from '@mui/icons-material/Public';
import LinkIcon from '@mui/icons-material/Link';
import AccessTimeIcon from '@mui/icons-material/AccessTime';
import { BASE_URL } from '../../const';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';

// Styled components matching the other updated pages
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const DetailCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  padding: theme.spacing(3),
  marginBottom: theme.spacing(4),
  marginTop: theme.spacing(2),
}));

const DetailSubtitle = styled(Typography)(({ theme }) => ({
  fontSize: '1.3rem',
  fontWeight: 600,
  color: theme.palette.text.primary,
  marginBottom: theme.spacing(2),
}));

const DetailItem = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1.5),
    color: theme.palette.primary.main,
  }
}));

const DetailLabel = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontWeight: 500,
  width: '160px',
  flexShrink: 0,
}));

const DetailValue = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.primary,
  fontWeight: 400,
  flexGrow: 1,
  overflow: 'hidden',
  textOverflow: 'ellipsis',
}));

const TagsContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  flexWrap: 'wrap',
  gap: theme.spacing(1),
  marginTop: theme.spacing(2),
}));

const TagChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(0, 192, 127, 0.1)',
  color: theme.palette.text.primary,
  fontSize: '0.75rem',
  fontWeight: 500,
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.2)',
  }
}));

// eslint-disable-next-line no-unused-vars
const PathsList = styled(List)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.1)',
  borderRadius: theme.shape.borderRadius,
  marginTop: theme.spacing(2),
  '& .MuiListItem-root': {
    borderBottom: '1px solid rgba(255, 255, 255, 0.05)',
    '&:last-child': {
      borderBottom: 'none',
    }
  }
}));

const SchemaTable = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.1)',
  borderRadius: theme.shape.borderRadius,
  marginTop: theme.spacing(2),
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.05)',
    padding: theme.spacing(1, 2),
  },
  '& .MuiTableCell-head': {
    backgroundColor: 'rgba(0, 0, 0, 0.2)',
    color: theme.palette.primary.main,
    fontWeight: 600,
  }
}));

const DatasetDetails = () => {
    const [dataset, setDataset] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);

    let { '*': name } = useParams();

    // Format large numbers with abbreviations - temporarily not used
    // eslint-disable-next-line no-unused-vars
    const formatRowCount = (count) => {
        if (!count) return 'N/A';
        
        const num = parseInt(count, 10);
        
        // If less than 1 million, just return the formatted number
        if (num < 1000000) {
            return num.toLocaleString();
        }
        
        // For large numbers, abbreviate with suffix and show full number in parentheses
        let formatted;
        if (num >= 1000000000) {
            formatted = `${(num / 1000000000).toFixed(1)}B`;
        } else if (num >= 1000000) {
            formatted = `${(num / 1000000).toFixed(1)}M`;
        }
        
        return `${formatted} (${num.toLocaleString()})`;
    };

    useEffect(() => {
        const fetchDatasetDetails = async () => {
            // Do not retrieve if the name is none
            if (!name) {
                setLoading(false);
                return;
            }
            
            try {
                // Decode the URN from the route parameter, then pass it as query parameter
                const decodedUrn = decodeURIComponent(name);
                const uri = `${BASE_URL}/catalog/datasets/by-urn?urn=${encodeURIComponent(decodedUrn)}`;
                const response = await axios.get(uri);
                
                if (response.status === 200) {
                    // If the API returns an array, take the first item
                    const datasetData = Array.isArray(response.data) ? response.data[0] : response.data;
                    setDataset(datasetData);
                } else {
                    throw new Error(`Failed to fetch dataset details from ${uri}`);
                }
            } catch (err) {
                setError(err);
            } finally {
                setLoading(false);
            }
        };

        fetchDatasetDetails();
    }, [name]);

    return (
        <ContentContainer>
            <BackButton 
                component={Link} 
                to="/catalog" 
                startIcon={<ArrowBackIcon />}
                size="small"
            >
                Back to Dataset List
            </BackButton>
            
            <SectionTitle variant="h4">Dataset Details</SectionTitle>
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={error.message} />}
            
            {!loading && !error && dataset && (
                <DetailCard>
                    <DetailSubtitle variant="h5">
                        {dataset.name}
                    </DetailSubtitle>
                    
                    {dataset.description && (
                        <Typography variant="body1" sx={{ color: 'text.secondary', mb: 2, fontStyle: 'italic' }}>
                            {dataset.description}
                        </Typography>
                    )}
                    
                    <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
                    
                    <Grid container spacing={3}>
                        <Grid item xs={12} md={6}>
                            <DetailItem>
                                <StorageIcon />
                                <DetailLabel>Platform:</DetailLabel>
                                <DetailValue>{dataset.platform || 'N/A'}</DetailValue>
                            </DetailItem>
                            
                            <DetailItem>
                                <PublicIcon />
                                <DetailLabel>Environment:</DetailLabel>
                                <DetailValue>{dataset.environment || 'N/A'}</DetailValue>
                            </DetailItem>
                            
                            <DetailItem>
                                <LinkIcon />
                                <DetailLabel>URN:</DetailLabel>
                                <DetailValue sx={{ fontSize: '0.8rem', wordBreak: 'break-all' }}>
                                    {dataset.urn || 'N/A'}
                                </DetailValue>
                            </DetailItem>
                            
                            {dataset.container_urn && (
                                <DetailItem>
                                    <FolderIcon />
                                    <DetailLabel>Container URN:</DetailLabel>
                                    <DetailValue sx={{ fontSize: '0.8rem', wordBreak: 'break-all' }}>
                                        {dataset.container_urn}
                                    </DetailValue>
                                </DetailItem>
                            )}
                            
                            {dataset.created_at && (
                                <DetailItem>
                                    <AccessTimeIcon />
                                    <DetailLabel>Created:</DetailLabel>
                                    <DetailValue>{new Date(dataset.created_at).toLocaleString()}</DetailValue>
                                </DetailItem>
                            )}
                            
                            {dataset.updated_at && (
                                <DetailItem>
                                    <AccessTimeIcon />
                                    <DetailLabel>Updated:</DetailLabel>
                                    <DetailValue>{new Date(dataset.updated_at).toLocaleString()}</DetailValue>
                                </DetailItem>
                            )}
                        </Grid>
                        
                        <Grid item xs={12} md={6}>
                            <Box sx={{ mb: 3 }}>
                                <Typography variant="subtitle1" sx={{ fontWeight: 600, mb: 1 }}>
                                    <LabelIcon sx={{ verticalAlign: 'middle', mr: 1 }} />
                                    Tags
                                </Typography>
                                
                                {Array.isArray(dataset.tags) && dataset.tags.length > 0 ? (
                                    <TagsContainer>
                                        {dataset.tags.map((tag, index) => (
                                            <TagChip key={index} label={tag} size="small" />
                                        ))}
                                    </TagsContainer>
                                ) : (
                                    <Typography sx={{ color: 'text.secondary', fontStyle: 'italic' }}>
                                        No tags available
                                    </Typography>
                                )}
                            </Box>
                            
                            {dataset.properties && Object.keys(dataset.properties).length > 0 && (
                                <Box>
                                    <Typography variant="subtitle1" sx={{ fontWeight: 600, mb: 1 }}>
                                        <FolderIcon sx={{ verticalAlign: 'middle', mr: 1 }} />
                                        Properties
                                    </Typography>
                                    
                                    <Box sx={{ backgroundColor: 'rgba(0, 0, 0, 0.1)', borderRadius: 1, p: 2 }}>
                                        {Object.entries(dataset.properties).map(([key, value]) => (
                                            <DetailItem key={key} sx={{ mb: 1 }}>
                                                <DetailLabel sx={{ width: 'auto', mr: 2 }}>{key}:</DetailLabel>
                                                <DetailValue>{String(value)}</DetailValue>
                                            </DetailItem>
                                        ))}
                                    </Box>
                                </Box>
                            )}
                        </Grid>
                    </Grid>
                    
                    {dataset.dataset_schema && (
                        <Box sx={{ mt: 4 }}>
                            <Typography variant="subtitle1" sx={{ fontWeight: 600, mb: 1 }}>
                                <SchemaIcon sx={{ verticalAlign: 'middle', mr: 1, color: theme => theme.palette.primary.main }} />
                                Dataset Schema
                            </Typography>
                            
                            <SchemaTable>
                                <Table size="small">
                                    <TableHead>
                                        <TableRow>
                                            <TableCell>Field Name</TableCell>
                                            <TableCell>Data Type</TableCell>
                                        </TableRow>
                                    </TableHead>
                                    <TableBody>
                                        {Object.entries(dataset.dataset_schema).map(([field, type]) => (
                                            <TableRow key={field}>
                                                <TableCell>{field}</TableCell>
                                                <TableCell>
                                                    {typeof type === 'object' 
                                                        ? `${type.type}<${type.value_type}>` 
                                                        : type}
                                                </TableCell>
                                            </TableRow>
                                        ))}
                                    </TableBody>
                                </Table>
                            </SchemaTable>
                        </Box>
                    )}
                </DetailCard>
            )}

            {!loading && !error && !dataset && (
                <Typography variant="body1" sx={{ color: 'text.secondary', fontStyle: 'italic', mt: 2 }}>
                    No dataset found with the specified identifier.
                </Typography>
            )}
        </ContentContainer>
    );
};

export default DatasetDetails;
