// biome-ignore lint/correctness/noUnusedImports: React is required for JSX at runtime
import React from "react";
import { renderWithProviders, testData, userEvent } from "../../../test-utils";
import Sidebar from "../Sidebar";

// Mock constants
// LAB_URL includes /lab to match production behavior (see frontend/src/const.js)
jest.mock("../../../const", () => ({
	BASE_URL: "http://localhost:8000",
	LAB_URL: "http://localhost:8888/lab",
}));

describe("Sidebar", () => {
	const defaultProps = {
		open: true,
		onToggle: jest.fn(),
	};

	// Admin user for testing admin-only menu items
	const adminUser = { ...testData.user, roles: ["admin"], is_superuser: true };

	beforeEach(() => {
		jest.clearAllMocks();
	});

	describe("Rendering", () => {
		test("renders sidebar with logo and menu items for non-admin user", () => {
			const { getByAltText, getByText, queryByText } = renderWithProviders(
				<Sidebar {...defaultProps} />,
				{
					authValue: {
						user: testData.user,
						loading: false,
					},
				},
			);

			// Check logo
			expect(getByAltText("Kamiwaza Logo")).toBeInTheDocument();
			expect(getByText("KAMIWAZA")).toBeInTheDocument();

			// Check non-admin menu items (includes Notebooks, Catalog, News but not admin-only items)
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Notebooks")).toBeInTheDocument();
			expect(getByText("Models")).toBeInTheDocument();
			expect(getByText("Apps")).toBeInTheDocument();
			expect(getByText("Tools")).toBeInTheDocument();
			expect(getByText("Catalog")).toBeInTheDocument();
			expect(getByText("News")).toBeInTheDocument();

			// Admin-only items should NOT be visible
			expect(queryByText("Activity")).not.toBeInTheDocument();
			expect(queryByText("Cluster")).not.toBeInTheDocument();
			expect(queryByText("VectorDBs")).not.toBeInTheDocument();
			expect(queryByText("API Docs")).not.toBeInTheDocument();
		});

		test("renders sidebar with full menu for admin user", () => {
			const { getByAltText, getByText } = renderWithProviders(
				<Sidebar {...defaultProps} />,
				{
					authValue: {
						user: adminUser,
						loading: false,
					},
				},
			);

			// Check logo
			expect(getByAltText("Kamiwaza Logo")).toBeInTheDocument();
			expect(getByText("KAMIWAZA")).toBeInTheDocument();

			// Check admin menu items (full menu)
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Notebooks")).toBeInTheDocument();
			expect(getByText("Models")).toBeInTheDocument();
			expect(getByText("App Garden")).toBeInTheDocument();
			expect(getByText("Tool Shed")).toBeInTheDocument();
			expect(getByText("Catalog")).toBeInTheDocument();
			expect(getByText("Activity")).toBeInTheDocument();
			expect(getByText("Cluster")).toBeInTheDocument();
			expect(getByText("VectorDBs")).toBeInTheDocument();
			expect(getByText("News")).toBeInTheDocument();
			expect(getByText("API Docs")).toBeInTheDocument();
		});

		test("renders in collapsed state when open is false", () => {
			const { queryByText, getByAltText, container } = renderWithProviders(
				<Sidebar open={false} onToggle={defaultProps.onToggle} />,
				{
					authValue: {
						user: testData.user,
						loading: false,
					},
				},
			);

			// Logo should still be visible
			expect(getByAltText("Kamiwaza Logo")).toBeInTheDocument();

			// Logo text should be hidden when collapsed
			expect(queryByText("KAMIWAZA")).not.toBeInTheDocument();

			// Menu items should still be rendered but text should be visually hidden
			const listItems = container.querySelectorAll(".MuiListItem-root");
			expect(listItems.length).toBeGreaterThan(0);

			// Check that sidebar drawer has collapsed styling
			const drawer = container.querySelector(".MuiDrawer-root");
			expect(drawer).toBeInTheDocument();
		});

		test("admin user sees admin-only menu items", () => {
			const { getByText, queryByText } = renderWithProviders(
				<Sidebar {...defaultProps} />,
				{
					authValue: {
						user: adminUser,
						loading: false,
					},
				},
			);

			// Admin-only items should be visible
			expect(getByText("Catalog")).toBeInTheDocument();
			expect(getByText("Cluster")).toBeInTheDocument();
			expect(getByText("VectorDBs")).toBeInTheDocument();
			expect(getByText("Notebooks")).toBeInTheDocument();
			expect(getByText("API Docs")).toBeInTheDocument();

			// Auth & Users has been moved to profile dropdown
			expect(queryByText("Auth & Users")).not.toBeInTheDocument();
		});

		test("non-admin user does not see admin-only menu items", () => {
			const regularUser = {
				...testData.user,
				roles: ["user"],
				is_superuser: false,
			};

			const { queryByText, getByText } = renderWithProviders(
				<Sidebar {...defaultProps} />,
				{
					authValue: {
						user: regularUser,
						loading: false,
					},
				},
			);

			// Non-admin users should see user menu (includes Notebooks, Catalog, News)
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Notebooks")).toBeInTheDocument();
			expect(getByText("Models")).toBeInTheDocument();
			expect(getByText("Apps")).toBeInTheDocument();
			expect(getByText("Tools")).toBeInTheDocument();
			expect(getByText("Catalog")).toBeInTheDocument();
			expect(getByText("News")).toBeInTheDocument();

			// Admin-only items should NOT be visible
			expect(queryByText("Activity")).not.toBeInTheDocument();
			expect(queryByText("Cluster")).not.toBeInTheDocument();
			expect(queryByText("VectorDBs")).not.toBeInTheDocument();
			expect(queryByText("API Docs")).not.toBeInTheDocument();
		});
	});

	describe("Toggle Functionality", () => {
		test("calls onToggle when toggle button is clicked", async () => {
			const user = userEvent.setup();
			const mockToggle = jest.fn();

			const { container } = renderWithProviders(
				<Sidebar open={true} onToggle={mockToggle} />,
				{
					authValue: {
						user: testData.user,
						loading: false,
					},
				},
			);

			// Find toggle button (should contain ChevronLeft when open)
			const toggleButton =
				container.querySelector("button[aria-label]") ||
				container.querySelector("button");

			await user.click(toggleButton);
			expect(mockToggle).toHaveBeenCalledTimes(1);
		});

		test("shows correct chevron icons based on open state", () => {
			const { container, rerender } = renderWithProviders(
				<Sidebar open={true} onToggle={defaultProps.onToggle} />,
				{
					authValue: {
						user: testData.user,
						loading: false,
					},
				},
			);

			// When open, should show ChevronLeft (collapse icon)
			const chevronLeft = container.querySelector(
				'[data-testid="ChevronLeftIcon"]',
			);
			expect(chevronLeft).toBeInTheDocument();

			// Re-render with open=false
			rerender(<Sidebar open={false} onToggle={defaultProps.onToggle} />);

			// When closed, should show ChevronRight (expand icon)
			const chevronRight = container.querySelector(
				'[data-testid="ChevronRightIcon"]',
			);
			expect(chevronRight).toBeInTheDocument();
		});
	});

	describe("Navigation Links", () => {
		test("renders internal navigation links correctly for non-admin user", () => {
			const { getByText } = renderWithProviders(<Sidebar {...defaultProps} />, {
				authValue: {
					user: testData.user,
					loading: false,
				},
				initialEntries: ["/"],
			});

			// Check that internal links are rendered as React Router Links
			const homeLink = getByText("Home").closest("a");
			const modelsLink = getByText("Models").closest("a");
			const appsLink = getByText("Apps").closest("a");
			const toolsLink = getByText("Tools").closest("a");

			expect(homeLink).toHaveAttribute("href", "/");
			expect(modelsLink).toHaveAttribute("href", "/models");
			expect(appsLink).toHaveAttribute("href", "/apps");
			expect(toolsLink).toHaveAttribute("href", "/tool-shed");
		});

		test("renders external links with correct attributes for admin user", () => {
			const { getByText } = renderWithProviders(<Sidebar {...defaultProps} />, {
				authValue: {
					user: adminUser,
					loading: false,
				},
			});

			// Check external links have proper attributes (admin-only items)
			const notebooksLink = getByText("Notebooks").closest("a");
			const apiDocsLink = getByText("API Docs").closest("a");

			expect(notebooksLink).toHaveAttribute(
				"href",
				"http://localhost:8888/lab",
			);
			expect(notebooksLink).toHaveAttribute("target", "_blank");
			expect(notebooksLink).toHaveAttribute("rel", "noopener noreferrer");

			expect(apiDocsLink).toHaveAttribute("href", "http://localhost:8000/docs");
			expect(apiDocsLink).toHaveAttribute("target", "_blank");
			expect(apiDocsLink).toHaveAttribute("rel", "noopener noreferrer");
		});
	});

	describe("Active State Highlighting", () => {
		test("renders menu items correctly on home page for non-admin user", () => {
			const { getByText } = renderWithProviders(<Sidebar {...defaultProps} />, {
				authValue: {
					user: testData.user,
					loading: false,
				},
				initialEntries: ["/"],
			});

			// Non-admin menu items should be present
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Models")).toBeInTheDocument();
			expect(getByText("Apps")).toBeInTheDocument();
			expect(getByText("Tools")).toBeInTheDocument();

			// The home item should be rendered (active state styling tested via visual regression)
			const homeItem = getByText("Home").closest(".MuiListItem-root");
			expect(homeItem).toBeInTheDocument();
		});

		test("renders menu items correctly on models page", () => {
			const { getByText } = renderWithProviders(<Sidebar {...defaultProps} />, {
				authValue: {
					user: testData.user,
					loading: false,
				},
				initialEntries: ["/models/123"],
			});

			const modelsItem = getByText("Models").closest(".MuiListItem-root");
			expect(modelsItem).toBeInTheDocument();

			// Models link should be rendered and accessible
			const modelsLink = getByText("Models").closest("a");
			expect(modelsLink).toHaveAttribute("href", "/models");
		});

		test("navigation structure remains consistent across routes for non-admin", () => {
			const { getByText, queryByText } = renderWithProviders(
				<Sidebar {...defaultProps} />,
				{
					authValue: {
						user: testData.user,
						loading: false,
					},
					initialEntries: ["/apps"],
				},
			);

			// Non-admin navigation items should be present regardless of current route
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Models")).toBeInTheDocument();
			expect(getByText("Apps")).toBeInTheDocument();
			expect(getByText("Tools")).toBeInTheDocument();
			expect(getByText("Catalog")).toBeInTheDocument();
			expect(getByText("News")).toBeInTheDocument();

			// Admin-only items should NOT be present for non-admin user
			expect(queryByText("Cluster")).not.toBeInTheDocument();
			expect(queryByText("VectorDBs")).not.toBeInTheDocument();
		});

		test("navigation structure remains consistent across routes for admin", () => {
			const { getByText } = renderWithProviders(<Sidebar {...defaultProps} />, {
				authValue: {
					user: adminUser,
					loading: false,
				},
				initialEntries: ["/catalog"],
			});

			// All admin navigation items should be present regardless of current route
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Models")).toBeInTheDocument();
			expect(getByText("Catalog")).toBeInTheDocument();
			expect(getByText("App Garden")).toBeInTheDocument();
		});
	});

	describe("User Authentication States", () => {
		test("renders correctly when user is null", () => {
			const { getByText, queryByText } = renderWithProviders(
				<Sidebar {...defaultProps} />,
				{
					authValue: {
						user: null,
						loading: false,
					},
				},
			);

			// Should show user menu items when no user (includes Notebooks, Catalog, News)
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Notebooks")).toBeInTheDocument();
			expect(getByText("Models")).toBeInTheDocument();
			expect(getByText("Apps")).toBeInTheDocument();
			expect(getByText("Tools")).toBeInTheDocument();
			expect(getByText("Catalog")).toBeInTheDocument();
			expect(getByText("News")).toBeInTheDocument();

			// Admin-only items should NOT be visible
			expect(queryByText("Cluster")).not.toBeInTheDocument();
			expect(queryByText("VectorDBs")).not.toBeInTheDocument();
		});

		test("renders correctly when user is undefined", () => {
			const { getByText, queryByText } = renderWithProviders(
				<Sidebar {...defaultProps} />,
				{
					authValue: {
						user: undefined,
						loading: false,
					},
				},
			);

			// Should show user menu items when no user (includes Notebooks, Catalog, News)
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Notebooks")).toBeInTheDocument();
			expect(getByText("Catalog")).toBeInTheDocument();

			// Admin-only items should NOT be visible
			expect(queryByText("Cluster")).not.toBeInTheDocument();
			expect(queryByText("VectorDBs")).not.toBeInTheDocument();
		});
	});

	describe("Responsive Behavior", () => {
		test("renders all menu items when sidebar is collapsed", () => {
			const { container } = renderWithProviders(
				<Sidebar open={false} onToggle={defaultProps.onToggle} />,
				{
					authValue: {
						user: testData.user,
						loading: false,
					},
				},
			);

			// Menu items should still be rendered when collapsed (7 items for non-admin)
			const listItems = container.querySelectorAll(".MuiListItem-root");
			expect(listItems.length).toBe(7); // Home, Notebooks, Models, Apps, Tools, Catalog, News

			// Icons should be present for navigation
			const icons = container.querySelectorAll(".MuiListItemIcon-root");
			expect(icons.length).toBe(7);

			// Drawer should be present
			const drawer = container.querySelector(".MuiDrawer-root");
			expect(drawer).toBeInTheDocument();
		});

		test("renders all menu items when sidebar is open for non-admin", () => {
			const { container, getByText } = renderWithProviders(
				<Sidebar open={true} onToggle={defaultProps.onToggle} />,
				{
					authValue: {
						user: testData.user,
						loading: false,
					},
				},
			);

			// Non-admin menu text should be visible when open
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Notebooks")).toBeInTheDocument();
			expect(getByText("Models")).toBeInTheDocument();
			expect(getByText("Apps")).toBeInTheDocument();
			expect(getByText("Tools")).toBeInTheDocument();
			expect(getByText("Catalog")).toBeInTheDocument();
			expect(getByText("News")).toBeInTheDocument();

			// Menu items should be rendered (7 for non-admin)
			const listItems = container.querySelectorAll(".MuiListItem-root");
			expect(listItems.length).toBe(7);
		});

		test("renders all menu items when sidebar is open for admin", () => {
			const { container, getByText } = renderWithProviders(
				<Sidebar open={true} onToggle={defaultProps.onToggle} />,
				{
					authValue: {
						user: adminUser,
						loading: false,
					},
				},
			);

			// Admin menu text should be visible when open
			expect(getByText("Home")).toBeInTheDocument();
			expect(getByText("Models")).toBeInTheDocument();
			expect(getByText("App Garden")).toBeInTheDocument();
			expect(getByText("Catalog")).toBeInTheDocument();

			// Menu items should be rendered (11 for admin)
			const listItems = container.querySelectorAll(".MuiListItem-root");
			expect(listItems.length).toBe(11);
		});
	});

	describe("Accessibility", () => {
		test("logo has proper alt text", () => {
			const { getByAltText } = renderWithProviders(
				<Sidebar {...defaultProps} />,
				{
					authValue: {
						user: testData.user,
						loading: false,
					},
				},
			);

			expect(getByAltText("Kamiwaza Logo")).toBeInTheDocument();
		});

		test("navigation items are keyboard accessible", () => {
			const { getByText } = renderWithProviders(<Sidebar {...defaultProps} />, {
				authValue: {
					user: testData.user,
					loading: false,
				},
			});

			// Check that links are focusable (they should have proper tabIndex)
			const homeLink = getByText("Home").closest("a");
			const modelsLink = getByText("Models").closest("a");

			expect(homeLink).toBeInTheDocument();
			expect(modelsLink).toBeInTheDocument();

			// Links should be focusable by default
			expect(homeLink.tabIndex).not.toBe(-1);
			expect(modelsLink.tabIndex).not.toBe(-1);
		});

		test("maintains proper semantic structure", () => {
			const { container } = renderWithProviders(<Sidebar {...defaultProps} />, {
				authValue: {
					user: testData.user,
					loading: false,
				},
			});

			// Should have proper list structure
			const list = container.querySelector('ul[role="list"], .MuiList-root');
			expect(list).toBeInTheDocument();

			const listItems = container.querySelectorAll("li, .MuiListItem-root");
			expect(listItems.length).toBeGreaterThan(0);
		});
	});

	describe("Edge Cases", () => {
		test("handles missing onToggle prop gracefully", () => {
			const { container } = renderWithProviders(<Sidebar open={true} />, {
				authValue: {
					user: testData.user,
					loading: false,
				},
			});

			// Should render without crashing
			expect(container).toBeInTheDocument();
		});

		test("handles missing open prop gracefully", () => {
			const { container } = renderWithProviders(
				<Sidebar onToggle={defaultProps.onToggle} />,
				{
					authValue: {
						user: testData.user,
						loading: false,
					},
				},
			);

			expect(container).toBeInTheDocument();
		});

		test("renders without crashing with minimal props", () => {
			const { container } = renderWithProviders(<Sidebar />, {
				authValue: {
					user: testData.user,
					loading: false,
				},
			});

			expect(container).toBeInTheDocument();
		});
	});

	describe("Menu Item Configuration", () => {
		test("displays expected menu items for non-admin user", () => {
			const { container } = renderWithProviders(<Sidebar {...defaultProps} />, {
				authValue: {
					user: testData.user,
					loading: false,
				},
			});

			const menuTexts = Array.from(
				container.querySelectorAll(".MuiListItemText-primary"),
			).map((el) => el.textContent);

			// Non-admin users see user menu (includes Notebooks, Catalog, News)
			const expectedItems = [
				"Home",
				"Notebooks",
				"Models",
				"Apps",
				"Tools",
				"Catalog",
				"News",
			];

			expect(menuTexts).toEqual(expectedItems);
		});

		test("displays all expected menu items for admin user", () => {
			const { container } = renderWithProviders(<Sidebar {...defaultProps} />, {
				authValue: {
					user: adminUser,
					loading: false,
				},
			});

			const menuTexts = Array.from(
				container.querySelectorAll(".MuiListItemText-primary"),
			).map((el) => el.textContent);

			// Admin users see full menu
			const expectedItems = [
				"Home",
				"Notebooks",
				"Models",
				"App Garden",
				"Tool Shed",
				"Catalog",
				"Activity",
				"Cluster",
				"VectorDBs",
				"News",
				"API Docs",
			];

			expect(menuTexts).toEqual(expectedItems);
		});

		test("does not include auth & users in sidebar menu", () => {
			// Auth & Users has been moved to profile dropdown for all users
			const { container, queryByText } = renderWithProviders(
				<Sidebar {...defaultProps} />,
				{
					authValue: {
						user: adminUser,
						loading: false,
					},
				},
			);

			const menuTexts = Array.from(
				container.querySelectorAll(".MuiListItemText-primary"),
			).map((el) => el.textContent);

			// Auth & Users should NOT be in the sidebar (it's in profile dropdown)
			expect(menuTexts).not.toContain("Auth & Users");
			expect(queryByText("Auth & Users")).not.toBeInTheDocument();
		});
	});
});
