import React from 'react';
import { renderWithProviders, userEvent, testData } from '../../../test-utils';
import Header from '../Header';

// Mock useNavigate
const mockNavigate = jest.fn();
jest.mock('react-router-dom', () => ({
  ...jest.requireActual('react-router-dom'),
  useNavigate: () => mockNavigate,
}));

describe('Header', () => {
  const defaultProps = {
    onToggleSidebar: jest.fn(),
    sidebarOpen: false
  };

  beforeEach(() => {
    jest.clearAllMocks();
  });

  describe('Logo and Branding', () => {
    test('displays Kamiwaza logo', () => {
      const { getByAltText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            loading: false
          }
        }
      );

      const logo = getByAltText('Kamiwaza Logo');
      expect(logo).toBeInTheDocument();
      expect(logo).toHaveAttribute('src', '/kamiwaza-logo-with-text.png');
    });

    test('displays Kamiwaza brand text', () => {
      const { getByAltText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            loading: false
          }
        }
      );

      // The logo now includes the text, so we verify the logo is present
      expect(getByAltText('Kamiwaza Logo')).toBeInTheDocument();
      
      // NOTE: If the logo is changed back to icon-only, restore these individual letter tests:
      // expect(getByText('K')).toBeInTheDocument();
      // expect(getByText('A')).toBeInTheDocument();
      // expect(getByText('M')).toBeInTheDocument();
      // expect(getByText('I')).toBeInTheDocument();
      // expect(getByText(/WAZA/)).toBeInTheDocument();
    });
  });

  describe('Action Buttons', () => {
    test('displays notification button with badge', () => {
      const { getByLabelText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            loading: false
          }
        }
      );

      const notificationButton = getByLabelText('Notifications');
      expect(notificationButton).toBeInTheDocument();
    });

    test('displays help button', () => {
      const { getByLabelText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            loading: false
          }
        }
      );

      const helpButton = getByLabelText('Help');
      expect(helpButton).toBeInTheDocument();
    });
  });

  describe('User Authentication States', () => {
    test('displays welcome message and avatar for logged-in user', () => {
      const { getByText, getByLabelText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            loading: false
          }
        }
      );

      expect(getByText(`Welcome, ${testData.user.username}`)).toBeInTheDocument();
      expect(getByLabelText('Account settings')).toBeInTheDocument();
    });

    test('displays login button when user is not authenticated', () => {
      const { getByRole, queryByText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: null,
            loading: false
          }
        }
      );

      expect(getByRole('button', { name: 'Login' })).toBeInTheDocument();
      expect(queryByText(/Welcome/)).not.toBeInTheDocument();
    });

    test('login button navigates to login page', async () => {
      const user = userEvent.setup();
      const { getByRole } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: null,
            loading: false
          }
        }
      );

      await user.click(getByRole('button', { name: 'Login' }));
      expect(mockNavigate).toHaveBeenCalledWith('/login');
    });
  });

  describe('User Avatar and Menu', () => {
    test('displays user initials in avatar', () => {
      const userWithName = {
        ...testData.user,
        username: 'John Doe'
      };

      const { getByText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: userWithName,
            loading: false
          }
        }
      );

      // Avatar should show initials 'JD'
      expect(getByText('JD')).toBeInTheDocument();
    });

    test('handles single name for initials', () => {
      const userWithSingleName = {
        ...testData.user,
        username: 'John'
      };

      const { getByText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: userWithSingleName,
            loading: false
          }
        }
      );

      expect(getByText('J')).toBeInTheDocument();
    });

    test('shows default initial for empty username', () => {
      const userWithEmptyName = {
        ...testData.user,
        username: ''
      };

      const { getByText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: userWithEmptyName,
            loading: false
          }
        }
      );

      expect(getByText('U')).toBeInTheDocument();
    });

    test('opens user menu when avatar is clicked', async () => {
      const user = userEvent.setup();
      const { getByLabelText, getByText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            loading: false
          }
        }
      );

      const avatar = getByLabelText('Account settings');
      await user.click(avatar);

      // Menu items should be visible
      expect(getByText('Settings')).toBeInTheDocument();
      expect(getByText('Logout')).toBeInTheDocument();
    });

    test('navigates to settings and closes menu when settings item is clicked', async () => {
      const user = userEvent.setup();
      const { getByLabelText, getByText, queryByText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            loading: false
          }
        }
      );

      // Open menu
      await user.click(getByLabelText('Account settings'));
      expect(getByText('Settings')).toBeInTheDocument();

      // Click settings
      await user.click(getByText('Settings'));
      expect(mockNavigate).toHaveBeenCalledWith('/settings?tab=personal');

      // Menu should close (items no longer visible)
      expect(queryByText('Settings')).not.toBeInTheDocument();
    });
  });

  describe('Logout Functionality', () => {
    test('handles logout when logout menu item is clicked', async () => {
      const mockLogout = jest.fn();
      const user = userEvent.setup();

      const { getByLabelText, getByText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            logout: mockLogout,
            loading: false
          }
        }
      );

      // Open menu and click logout
      await user.click(getByLabelText('Account settings'));
      await user.click(getByText('Logout'));

      expect(mockLogout).toHaveBeenCalledTimes(1);
      expect(mockNavigate).toHaveBeenCalledWith('/login');
    });
  });

  describe('Accessibility', () => {
    test('action buttons have proper labels', () => {
      const { getByLabelText } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            loading: false
          }
        }
      );

      expect(getByLabelText('Notifications')).toBeInTheDocument();
      expect(getByLabelText('Help')).toBeInTheDocument();
      expect(getByLabelText('Account settings')).toBeInTheDocument();
    });

    test('header has proper semantic structure', () => {
      const { container } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: testData.user,
            loading: false
          }
        }
      );

      // AppBar should render as header
      const header = container.querySelector('header');
      expect(header).toBeInTheDocument();
    });
  });

  describe('Edge Cases', () => {
    test('handles undefined user gracefully', () => {
      const { getByRole } = renderWithProviders(
        <Header {...defaultProps} />,
        {
          authValue: {
            user: undefined,
            loading: false
          }
        }
      );

      // Should show login button
      expect(getByRole('button', { name: 'Login' })).toBeInTheDocument();
    });

    test('renders without crashing with minimal props', () => {
      renderWithProviders(<Header />, {
        authValue: {
          user: testData.user,
          loading: false
        }
      });
    });
  });
});
