import React from 'react';
import { renderWithTheme } from '../../../test-utils';
import ErrorComponent from '../ErrorComponent';

describe('ErrorComponent', () => {
  test('renders without crashing', () => {
    renderWithTheme(<ErrorComponent message="Test error" />);
  });

  test('displays the error message', () => {
    const errorMessage = 'Something went wrong!';
    const { getByText } = renderWithTheme(<ErrorComponent message={errorMessage} />);
    
    expect(getByText(`Error: ${errorMessage}`)).toBeInTheDocument();
  });

  test('error is announced to screen readers', () => {
    const { getByRole } = renderWithTheme(<ErrorComponent message="Test error" />);
    
    // Test accessibility - users with screen readers should be alerted
    const errorAlert = getByRole('alert');
    expect(errorAlert).toBeInTheDocument();
    expect(errorAlert).toHaveTextContent('Error: Test error');
  });

  test('handles empty message gracefully', () => {
    const { getByText } = renderWithTheme(<ErrorComponent message="" />);
    
    expect(getByText('Error:')).toBeInTheDocument();
  });

  test('handles undefined message gracefully', () => {
    const { getByText } = renderWithTheme(<ErrorComponent />);
    
    expect(getByText('Error:')).toBeInTheDocument();
  });

  test('handles special characters in message (React auto-escapes)', () => {
    const specialMessage = 'Error with <script>alert("xss")</script> & symbols';
    const { getByText } = renderWithTheme(<ErrorComponent message={specialMessage} />);
    
    // React automatically escapes HTML, so the literal text should be displayed
    expect(getByText(`Error: ${specialMessage}`)).toBeInTheDocument();
    
    // Verify that the script tag is not executed (remains as text)
    const { container } = renderWithTheme(<ErrorComponent message={specialMessage} />);
    expect(container.querySelector('script')).not.toBeInTheDocument();
  });
});