import React from 'react';
import { renderWithTheme, userEvent, fireEvent } from '../../../test-utils';
import ConfirmModal from '../ConfirmModal';

describe('ConfirmModal', () => {
  const defaultProps = {
    title: 'Confirm Action',
    message: 'Are you sure you want to proceed?',
    onConfirm: jest.fn(),
    onCancel: jest.fn(),
    showModal: true
  };

  beforeEach(() => {
    jest.clearAllMocks();
  });

  test('renders without crashing when modal is shown', () => {
    renderWithTheme(<ConfirmModal {...defaultProps} />);
  });

  test('does not render modal content when showModal is false', () => {
    const { queryByText } = renderWithTheme(
      <ConfirmModal {...defaultProps} showModal={false} />
    );
    
    // When modal is closed, content should not be visible
    expect(queryByText(defaultProps.title)).not.toBeInTheDocument();
    expect(queryByText(defaultProps.message)).not.toBeInTheDocument();
  });

  test('displays the correct title and message', () => {
    const { getByText } = renderWithTheme(<ConfirmModal {...defaultProps} />);
    
    expect(getByText(defaultProps.title)).toBeInTheDocument();
    expect(getByText(defaultProps.message)).toBeInTheDocument();
  });

  test('shows confirm and cancel options to user', () => {
    const { getByRole } = renderWithTheme(<ConfirmModal {...defaultProps} />);
    
    expect(getByRole('button', { name: 'Yes' })).toBeInTheDocument();
    expect(getByRole('button', { name: 'No' })).toBeInTheDocument();
  });

  test('calls onConfirm when user confirms action', async () => {
    const user = userEvent.setup();
    const { getByRole } = renderWithTheme(<ConfirmModal {...defaultProps} />);
    
    await user.click(getByRole('button', { name: 'Yes' }));
    
    expect(defaultProps.onConfirm).toHaveBeenCalledTimes(1);
  });

  test('calls onCancel when user cancels action', async () => {
    const user = userEvent.setup();
    const { getByRole } = renderWithTheme(<ConfirmModal {...defaultProps} />);
    
    await user.click(getByRole('button', { name: 'No' }));
    
    expect(defaultProps.onCancel).toHaveBeenCalledTimes(1);
  });

  test('calls onCancel when close button is clicked', async () => {
    const user = userEvent.setup();
    const { getByLabelText } = renderWithTheme(<ConfirmModal {...defaultProps} />);
    const closeButton = getByLabelText('close');
    
    await user.click(closeButton);
    
    expect(defaultProps.onCancel).toHaveBeenCalledTimes(1);
  });

  test('user can cancel with Escape key', () => {
    renderWithTheme(<ConfirmModal {...defaultProps} />);
    
    // For window-level keyboard events, fireEvent is more reliable than userEvent
    // This component listens to window keydown events, so we dispatch directly to window
    fireEvent.keyDown(window, { key: 'Escape', code: 'Escape', keyCode: 27 });
    
    expect(defaultProps.onCancel).toHaveBeenCalledTimes(1);
  });

  test('handles missing props gracefully', () => {
    const minimalProps = {
      onConfirm: jest.fn(),
      onCancel: jest.fn(),
      showModal: true
    };
    
    const { container, getByRole } = renderWithTheme(<ConfirmModal {...minimalProps} />);
    
    // Should render without crashing even with minimal props
    expect(container).toBeInTheDocument();
    
    // Should still provide user with action buttons
    expect(getByRole('button', { name: 'Yes' })).toBeInTheDocument();
    expect(getByRole('button', { name: 'No' })).toBeInTheDocument();
    
    // Modal should function even without title/message
  });

  test('has proper accessibility attributes', () => {
    const { getByLabelText } = renderWithTheme(<ConfirmModal {...defaultProps} />);
    
    // Check for close button accessibility
    const closeButton = getByLabelText('close');
    expect(closeButton).toHaveAttribute('aria-label', 'close');
  });

  test('modal styling and structure', () => {
    const { getByText } = renderWithTheme(<ConfirmModal {...defaultProps} />);
    
    // Check that modal content is rendered by looking for title
    expect(getByText(defaultProps.title)).toBeInTheDocument();
    expect(getByText(defaultProps.message)).toBeInTheDocument();
  });

  test('modal shows and hides based on showModal prop', () => {
    const { rerender, queryByText } = renderWithTheme(
      <ConfirmModal {...defaultProps} showModal={false} />
    );
    
    // Initially hidden
    expect(queryByText(defaultProps.title)).not.toBeInTheDocument();
    
    // Show modal
    rerender(<ConfirmModal {...defaultProps} showModal={true} />);
    expect(queryByText(defaultProps.title)).toBeInTheDocument();
    
    // Hide modal again
    rerender(<ConfirmModal {...defaultProps} showModal={false} />);
    expect(queryByText(defaultProps.title)).not.toBeInTheDocument();
  });
});