import AppsIcon from "@mui/icons-material/Apps";
import CategoryIcon from "@mui/icons-material/Category";
import ChevronLeftIcon from "@mui/icons-material/ChevronLeft";
import ChevronRightIcon from "@mui/icons-material/ChevronRight";
import CodeIcon from "@mui/icons-material/Code";
import DnsIcon from "@mui/icons-material/Dns";
import HandymanIcon from "@mui/icons-material/Handyman";
import HomeIcon from "@mui/icons-material/Home";
import LayersIcon from "@mui/icons-material/Layers";
import MenuBookIcon from "@mui/icons-material/MenuBook";
import NewspaperIcon from "@mui/icons-material/Newspaper";
import NotificationsIcon from "@mui/icons-material/Notifications";
import StorageIcon from "@mui/icons-material/Storage";
import {
	Box,
	Divider,
	Drawer,
	IconButton,
	List,
	ListItem,
	ListItemButton,
	ListItemIcon,
	ListItemText,
	Tooltip,
	Typography,
} from "@mui/material";
import { styled } from "@mui/material/styles";
import PropTypes from "prop-types";
import React from "react";
import { Link, useLocation } from "react-router-dom";
import { BASE_URL, LAB_URL } from "../../const";
import { useSecurity } from "../../context/SecurityContext";
import { useIsAdmin } from "../../hooks/useIsAdmin";
import { BANNER_HEIGHT } from "../security";

const DRAWER_WIDTH = 240;

const StyledDrawer = styled(Drawer, {
	shouldForwardProp: (prop) => prop !== "open",
})(({ theme, open }) => ({
	width: open ? DRAWER_WIDTH : theme.spacing(9),
	flexShrink: 0,
	whiteSpace: "nowrap",
	boxSizing: "border-box",
	overflowX: "hidden",
	transition: theme.transitions.create(["width", "margin"], {
		easing: theme.transitions.easing.sharp,
		duration: theme.transitions.duration.enteringScreen,
	}),
	"& .MuiDrawer-paper": {
		width: open ? DRAWER_WIDTH : theme.spacing(9),
		overflowX: "hidden",
		transition: theme.transitions.create(["width", "margin"], {
			easing: theme.transitions.easing.sharp,
			duration: theme.transitions.duration.enteringScreen,
		}),
		backgroundImage:
			"linear-gradient(rgba(0, 192, 127, 0.03), rgba(0, 0, 0, 0.03))",
		boxShadow: "0 0 20px rgba(0, 0, 0, 0.15)",
	},
}));

const StyledLogo = styled(Box)(({ theme }) => ({
	display: "flex",
	alignItems: "center",
	justifyContent: "flex-start",
	padding: theme.spacing(2),
	paddingLeft: theme.spacing(3),
	height: "64px",
	borderBottom: `1px solid ${theme.palette.divider}`,
}));

const Logo = styled("img")(({ theme }) => ({
	height: "36px",
	marginRight: theme.spacing(2),
	transition: theme.transitions.create("all", {
		easing: theme.transitions.easing.sharp,
		duration: theme.transitions.duration.enteringScreen,
	}),
}));

const LogoText = styled(Typography)(({ theme }) => ({
	fontSize: "1.5rem",
	fontWeight: 600,
	backgroundImage: "linear-gradient(90deg, #00c07f, #00c0a0)",
	backgroundClip: "text",
	textFillColor: "transparent",
	WebkitBackgroundClip: "text",
	WebkitTextFillColor: "transparent",
}));

const DrawerToggleButton = styled(IconButton)(({ theme }) => ({
	position: "absolute",
	right: -15,
	top: 20,
	backgroundColor: theme.palette.background.paper,
	width: 30,
	height: 30,
	border: `1px solid ${theme.palette.divider}`,
	borderRadius: "50%",
	display: "flex",
	justifyContent: "center",
	alignItems: "center",
	"&:hover": {
		backgroundColor: theme.palette.background.default,
	},
	zIndex: 1200,
}));

// Simplified menu for regular (non-admin) users
const USER_MENU_ITEMS = [
	{ text: "Home", icon: <HomeIcon />, path: "/", external: false },
	{ text: "Notebooks", icon: <MenuBookIcon />, path: LAB_URL, external: true },
	{ text: "Models", icon: <LayersIcon />, path: "/models", external: false },
	{ text: "Apps", icon: <AppsIcon />, path: "/apps", external: false },
	{
		text: "Tools",
		icon: <HandymanIcon />,
		path: "/tool-shed",
		external: false,
	},
	{
		text: "Catalog",
		icon: <CategoryIcon />,
		path: "/catalog",
		external: false,
	},
	{ text: "News", icon: <NewspaperIcon />, path: "/news", external: false },
];

// Full menu for admin users
const ADMIN_MENU_ITEMS = [
	{ text: "Home", icon: <HomeIcon />, path: "/", external: false },
	{ text: "Notebooks", icon: <MenuBookIcon />, path: LAB_URL, external: true },
	{ text: "Models", icon: <LayersIcon />, path: "/models", external: false },
	{ text: "App Garden", icon: <AppsIcon />, path: "/apps", external: false },
	{
		text: "Tool Shed",
		icon: <HandymanIcon />,
		path: "/tool-shed",
		external: false,
	},
	{
		text: "Catalog",
		icon: <CategoryIcon />,
		path: "/catalog",
		external: false,
	},
	{
		text: "Activity",
		icon: <NotificationsIcon />,
		path: "/activity",
		external: false,
	},
	{
		text: "Cluster",
		icon: <StorageIcon />,
		path: "/cluster/home",
		external: false,
	},
	{
		text: "VectorDBs",
		icon: <DnsIcon />,
		path: "/vectordbs/home",
		external: false,
	},
	{ text: "News", icon: <NewspaperIcon />, path: "/news", external: false },
	{
		text: "API Docs",
		icon: <CodeIcon />,
		path: `${BASE_URL}/docs`,
		external: true,
	},
];

const Sidebar = ({ open, onToggle }) => {
	const isAdmin = useIsAdmin();
	const { shouldShowBanners } = useSecurity();
	const location = useLocation();
	const menuItems = isAdmin ? ADMIN_MENU_ITEMS : USER_MENU_ITEMS;

	const isActive = (item) => {
		if (item.external) {
			return false;
		}
		if (item.search) {
			const current = new URLSearchParams(location.search);
			const target = new URLSearchParams(item.search);
			return (
				location.pathname === item.path &&
				current.get("tab") === target.get("tab")
			);
		}
		if (item.path === "/") {
			return location.pathname === "/";
		}
		return location.pathname.startsWith(item.path);
	};

	return (
		<StyledDrawer
			variant="permanent"
			open={open}
			sx={{
				"& .MuiDrawer-paper": {
					top: shouldShowBanners ? BANNER_HEIGHT : 0,
					height: shouldShowBanners
						? `calc(100% - ${BANNER_HEIGHT * 2}px)`
						: "100%",
				},
			}}
		>
			<StyledLogo>
				<Logo src="/logo_icon.png" alt="Kamiwaza Logo" />
				{open && <LogoText variant="h6">KAMIWAZA</LogoText>}
			</StyledLogo>

			<DrawerToggleButton onClick={onToggle}>
				{open ? (
					<ChevronLeftIcon fontSize="small" />
				) : (
					<ChevronRightIcon fontSize="small" />
				)}
			</DrawerToggleButton>

			<Divider sx={{ my: 1 }} />

			<List>
				{menuItems.map((item) => {
					const listItem = (
						<ListItem
							key={item.text}
							disablePadding
							sx={{
								display: "block",
								py: 0.5,
								backgroundColor: isActive(item)
									? "rgba(0, 192, 127, 0.15)"
									: "transparent",
								borderLeft: isActive(item)
									? "4px solid #00c07f"
									: "4px solid transparent",
								"&:hover": {
									backgroundColor: "rgba(180, 192, 188, 0.1)",
								},
							}}
						>
							<ListItemButton
								sx={{
									minHeight: 48,
									justifyContent: open ? "initial" : "center",
									px: 2.5,
									backgroundColor: "transparent",
									"&:hover": {
										backgroundColor: "transparent",
									},
								}}
								component={item.external ? "a" : Link}
								to={
									!item.external
										? item.search
											? { pathname: item.path, search: item.search }
											: item.path
										: undefined
								}
								href={item.external ? item.path : undefined}
								target={item.external ? "_blank" : undefined}
								rel={item.external ? "noopener noreferrer" : undefined}
							>
								<ListItemIcon
									sx={{
										minWidth: 0,
										mr: open ? 3 : "auto",
										justifyContent: "center",
										color: "#00c07f", // Always green
									}}
								>
									{item.icon}
								</ListItemIcon>
								<ListItemText
									primary={item.text}
									sx={{
										opacity: open ? 1 : 0,
										display: open ? "block" : "none",
										"& .MuiTypography-root": {
											fontWeight: 700,
										},
									}}
								/>
							</ListItemButton>
						</ListItem>
					);

					return open ? (
						listItem
					) : (
						<Tooltip title={item.text} placement="right" key={item.text}>
							{listItem}
						</Tooltip>
					);
				})}
			</List>
		</StyledDrawer>
	);
};

Sidebar.propTypes = {
	open: PropTypes.bool.isRequired,
	onToggle: PropTypes.func.isRequired,
};

export default Sidebar;
