import React, { useState, useEffect } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import { useNavigate } from 'react-router-dom';
import { useAuth } from '../../context/AuthContext';
import { useIsAdmin } from '../../hooks/useIsAdmin';
import { useSecurity } from '../../context/SecurityContext';
import { BANNER_HEIGHT } from '../security';
import { 
  AppBar, 
  Toolbar, 
  Typography, 
  Button, 
  Box, 
  IconButton,
  Avatar,
  Menu,
  MenuItem,
  Tooltip,
  Badge,
  Chip,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  DialogContentText
} from '@mui/material';
import MenuIcon from '@mui/icons-material/Menu';
import NotificationsIcon from '@mui/icons-material/Notifications';
import HelpOutlineIcon from '@mui/icons-material/HelpOutline';
import DeveloperModeIcon from '@mui/icons-material/DeveloperMode';

const StyledAppBar = styled(AppBar)(({ theme }) => ({
  backgroundColor: theme.palette.background.paper,
  color: theme.palette.text.primary,
  boxShadow: 'none',
  borderBottom: `1px solid ${theme.palette.divider}`,
  position: 'fixed',
  zIndex: theme.zIndex.drawer + 1,
  transition: theme.transitions.create(['margin', 'width'], {
    easing: theme.transitions.easing.sharp,
    duration: theme.transitions.duration.leavingScreen,
  }),
}));

const StyledToolbar = styled(Toolbar)(({ theme }) => ({
  paddingLeft: theme.spacing(2),
  paddingRight: theme.spacing(2),
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'center',
  height: 64,
}));

const ActionButtonsContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(1),
}));

const UserDisplayName = styled(Typography)(({ theme }) => ({
  marginRight: theme.spacing(1),
  fontWeight: 500,
  fontSize: '0.9rem',
}));

const StyledIconButton = styled(IconButton)(({ theme }) => ({
  width: 40,
  height: 40,
  borderRadius: 20,
  color: theme.palette.text.primary,
  '&:hover': {
    backgroundColor: 'rgba(255,255,255,0.1)',
  }
}));

const UserAvatar = styled(Avatar)(({ theme }) => ({
  width: 40,
  height: 40,
  backgroundColor: theme.palette.primary.main,
  color: theme.palette.primary.contrastText,
  cursor: 'pointer',
  '&:hover': {
    boxShadow: '0 0 0 2px rgba(0,192,127,0.5)',
  }
}));

const StyledButton = styled(Button)(({ theme }) => ({
  borderRadius: theme.shape.borderRadius,
  padding: theme.spacing(1, 2),
  transition: 'all 0.2s',
  fontWeight: 500,
  textTransform: 'none',
  backgroundColor: theme.palette.primary.main,
  color: theme.palette.primary.contrastText,
  '&:hover': {
    backgroundColor: theme.palette.primary.dark,
    transform: 'translateY(-2px)',
  },
}));

const Header = ({ onToggleSidebar }) => {
  const { user, logout } = useAuth();
  const isAdmin = useIsAdmin();
  const { shouldShowBanners } = useSecurity();
  const navigate = useNavigate();
  const [anchorEl, setAnchorEl] = useState(null);
  const [advancedMode, setAdvancedMode] = useState(false);
  const [showDisableDialog, setShowDisableDialog] = useState(false);

  useEffect(() => {
    // Check if advanced mode is enabled in localStorage
    const savedMode = localStorage.getItem('advancedDeveloperMode');
    setAdvancedMode(savedMode === 'true');
  }, []);
  
  const handleLogout = () => {
    logout();
    navigate('/login');
    handleCloseMenu();
  };

  const handleOpenMenu = (event) => {
    setAnchorEl(event.currentTarget);
  };

  const handleCloseMenu = () => {
    setAnchorEl(null);
  };

  const getInitials = (name) => {
    if (!name) return 'U';
    return name.split(' ').map(n => n[0]).join('').toUpperCase();
  };

  const handleAdvancedModeClick = () => {
    setShowDisableDialog(true);
  };

  const handleDisableConfirm = () => {
    setAdvancedMode(false);
    localStorage.setItem('advancedDeveloperMode', 'false');
    setShowDisableDialog(false);
    // Reload page to apply changes
    window.location.reload();
  };

  return (
    <StyledAppBar position="fixed" sx={{ top: shouldShowBanners ? BANNER_HEIGHT : 0 }}>
      <StyledToolbar>
        {/* Logo and Page Title */}
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          {!advancedMode && (
            <IconButton
              color="inherit"
              aria-label="open drawer"
              onClick={onToggleSidebar}
              edge="start"
              sx={{ mr: 1 }}
              data-tour="hamburger-menu"
            >
              <MenuIcon />
            </IconButton>
          )}
          <img 
            src="/kamiwaza-logo-with-text.png" 
            alt="Kamiwaza Logo" 
            style={{ height: '50px' }} 
          />
          
          {/* Advanced Developer Mode chip - only shown to admin users */}
          {isAdmin && advancedMode && (
            <Chip
              icon={<DeveloperModeIcon />}
              label="Advanced Developer Mode: On"
              onClick={handleAdvancedModeClick}
              color="primary"
              variant="outlined"
              size="small"
              sx={{
                cursor: 'pointer',
                '&:hover': {
                  backgroundColor: 'rgba(0, 192, 127, 0.1)',
                }
              }}
            />
          )}
        </Box>

        {/* Right-aligned action buttons */}
        <ActionButtonsContainer>
          <Tooltip title="Notifications">
            <StyledIconButton color="inherit">
              <Badge badgeContent={0} color="primary">
                <NotificationsIcon />
              </Badge>
            </StyledIconButton>
          </Tooltip>
          
          <Tooltip title="Help">
            <StyledIconButton color="inherit">
              <HelpOutlineIcon />
            </StyledIconButton>
          </Tooltip>

          {user ? (
            <>
              <UserDisplayName variant="body2">
                {`Welcome, ${user.username}`}
              </UserDisplayName>
              
              <Tooltip title="Account settings">
                <UserAvatar onClick={handleOpenMenu}>
                  {getInitials(user.username)}
                </UserAvatar>
              </Tooltip>
              
              <Menu
                anchorEl={anchorEl}
                open={Boolean(anchorEl)}
                onClose={handleCloseMenu}
                PaperProps={{
                  elevation: 3,
                  sx: {
                    mt: 1.5,
                    backgroundColor: 'background.paper',
                    boxShadow: '0 8px 16px rgba(0,0,0,0.3)',
                    borderRadius: 2,
                    minWidth: 180,
                  },
                }}
                transformOrigin={{ horizontal: 'right', vertical: 'top' }}
                anchorOrigin={{ horizontal: 'right', vertical: 'bottom' }}
              >
                <MenuItem
                  onClick={() => {
                    handleCloseMenu();
                    navigate('/settings?tab=personal');
                  }}
                >
                  Settings
                </MenuItem>
                <MenuItem onClick={handleLogout}>Logout</MenuItem>
              </Menu>
            </>
          ) : (
            <StyledButton onClick={() => navigate('/login')}>
              Login
            </StyledButton>
          )}
        </ActionButtonsContainer>
      </StyledToolbar>

      {/* Dialog for disabling advanced mode - only rendered for admin users */}
      {isAdmin && (
        <Dialog
          open={showDisableDialog}
          onClose={() => setShowDisableDialog(false)}
        >
          <DialogTitle>Disable Advanced Developer Mode?</DialogTitle>
          <DialogContent>
            <DialogContentText>
              This will switch to the simplified interface designed for novice users.
              Some advanced features will be hidden. Do you want to continue?
            </DialogContentText>
          </DialogContent>
          <DialogActions>
            <Button onClick={() => setShowDisableDialog(false)}>Cancel</Button>
            <Button onClick={handleDisableConfirm} color="primary" variant="contained">
              Yes, Disable
            </Button>
          </DialogActions>
        </Dialog>
      )}
    </StyledAppBar>
  );
};

Header.propTypes = {
  onToggleSidebar: PropTypes.func.isRequired,
};

export default Header;
