import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { BASE_URL } from '../../const';
import { useParams, Link, useNavigate } from 'react-router-dom';
import { StyledMainContent, StyledHeader, StyledParagraph, StyledButton } from '../../StyledComponents';
import ArrowBackIosIcon from '@mui/icons-material/ArrowBackIos';

const UserDetails = () => {
    const [user, setUser] = useState(null);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);

    const { id } = useParams();
    const navigate = useNavigate();

    useEffect(() => {
        const fetchUserDetails = async () => {
            if (!id) {
                console.error('User ID is undefined:', id);
                setError(new Error('User ID is undefined'));
                return;
            }

            setLoading(true);
            try {
                const uri = `${BASE_URL}/auth/users/${id}`;
                console.log('Fetching user details from:', uri);
                const response = await axios.get(uri);
                if (response.status === 200) {
                    setUser(response.data);
                } else {
                    throw new Error(`Failed to fetch user details from ${uri}`);
                }
            } catch (err) {
                console.error('Error fetching user details:', err);
                setError(err);
            } finally {
                setLoading(false);
            }
        };

        fetchUserDetails();
    }, [id]);

    if (loading) {
        return (
            <StyledMainContent>
                <StyledHeader>Loading User Details</StyledHeader>
                <StyledParagraph>Please wait...</StyledParagraph>
            </StyledMainContent>
        );
    }

    if (error) {
        return (
            <StyledMainContent>
                <StyledHeader>Error</StyledHeader>
                <StyledParagraph>Error: {error.message}</StyledParagraph>
                <StyledButton component={Link} to="/admin" variant="outlined" startIcon={<ArrowBackIosIcon />}>
                    {"<< Back to Admin"}
                </StyledButton>
            </StyledMainContent>
        );
    }

    return (
        <StyledMainContent>
            <StyledButton component={Link} to="/admin" variant="outlined" startIcon={<ArrowBackIosIcon />}>
                {"<< Back to Admin"}
            </StyledButton>
            <StyledHeader>User Details</StyledHeader>
            {user ? (
                <div className="user-list">
                    <div key={user.id} className="user-card">
                        <h2 className="user-name">{user.name}</h2>
                        <p className="user-info"><strong>ID:</strong> {user.id}</p>
                        <p className="user-info"><strong>Username:</strong> {user.username}</p>
                        <p className="user-info"><strong>Email:</strong> {user.email}</p>
                        <p className="user-info"><strong>Is Admin:</strong> {user.is_superuser ? 'Yes' : 'No'}</p>
                        <p className="user-info"><strong>Created At:</strong> {user.created_at}</p>
                    </div>
                </div>
            ) : (
                <StyledParagraph>No user data available.</StyledParagraph>
            )}
        </StyledMainContent>
    );
};

export default UserDetails;
