import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Link, useLocation } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  List,
  ListItem,
  ListItemText,
  ListItemIcon,
  Button,
  Chip,
  Tooltip,
  IconButton
} from '@mui/material';
import ComputerIcon from '@mui/icons-material/Computer';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import RouterIcon from '@mui/icons-material/Router';
import VisibilityIcon from '@mui/icons-material/Visibility';
import SignalCellularAltIcon from '@mui/icons-material/SignalCellularAlt';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 0),
}));

const SectionTitle = styled(Typography)(({ theme, embedded }) => ({
  color: theme.palette.primary.main,
  fontSize: embedded ? '1.5rem' : '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const NodeCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
  marginBottom: theme.spacing(4),
}));

const NodesListContainer = styled(List)(({ theme }) => ({
  padding: 0,
}));

const NodeItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(2, 3),
  borderBottom: '1px solid rgba(255, 255, 255, 0.05)',
  transition: 'background-color 0.2s',
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.03)',
  },
  '&:last-child': {
    borderBottom: 'none',
  }
}));

const NodeLink = styled(Link)(({ theme }) => ({
  color: theme.palette.primary.main,
  textDecoration: 'none',
  fontWeight: 500,
  fontSize: '1rem',
  transition: 'all 0.2s',
  flex: 1,
  '&:hover': {
    color: theme.palette.primary.light,
    textDecoration: 'underline',
  },
}));

const ComputerIconStyled = styled(ComputerIcon)(({ theme }) => ({
  color: theme.palette.primary.main,
  marginRight: theme.spacing(1),
  fontSize: '1.2rem',
}));

const EmptyState = styled(Box)(({ theme }) => ({
  padding: theme.spacing(4),
  textAlign: 'center',
  color: theme.palette.text.secondary,
  fontStyle: 'italic',
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const IPChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.05)',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(2),
  fontSize: '0.75rem',
  height: 24,
}));

const StatusChip = styled(Chip)(({ theme, alive }) => ({
  backgroundColor: alive ? 'rgba(0, 192, 127, 0.1)' : 'rgba(244, 67, 54, 0.1)',
  color: alive ? theme.palette.success.main : theme.palette.error.main,
  marginLeft: theme.spacing(2),
  fontSize: '0.75rem',
  height: 24,
}));

const ViewButton = styled(IconButton)(({ theme }) => ({
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  padding: theme.spacing(0.75),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const RunningNodeList = ({ isEmbedded = false }) => {
    const [nodes, setNodes] = useState([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const locationPath = useLocation();

    useEffect(() => {
        const fetchNodes = async () => {
            try {
                const uri = `${BASE_URL}/cluster/get_running_nodes`;
                const response = await axios.get(uri);
                if (response.status === 200) {
                    setNodes(response.data);
                } else {
                    throw new Error(`Failed to fetch nodes from ${uri}`);
                }
            } catch (err) {
                setError(err.response && err.response.data && err.response.data.detail 
                    ? err.response.data.detail 
                    : err);
            } finally {
                setLoading(false);
            }
        };

        fetchNodes();
    }, []);

    // Function to truncate node ID for display
    const formatNodeId = (id) => {
        if (!id) return 'Unknown ID';
        if (id.length <= 20) return id;
        return `${id.substring(0, 8)}...${id.substring(id.length - 8)}`;
    };

    return (
        <ContentContainer>
            {!isEmbedded && locationPath.pathname !== "/cluster/home" && (
                <BackButton 
                    component={Link} 
                    to="/cluster/home" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to Clusters Home
                </BackButton>
            )}
            
            <SectionTitle variant="h4" embedded={isEmbedded}>
                Running Nodes
                <Tooltip title="List of currently active compute nodes in your cluster">
                    <InfoIcon />
                </Tooltip>
            </SectionTitle>
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={typeof error === 'string' ? error : error.message} />}
            
            {!loading && !error && (
                <NodeCard>
                    {nodes.length === 0 ? (
                        <EmptyState>
                            <Typography variant="body1">No running nodes available.</Typography>
                        </EmptyState>
                    ) : (
                        <NodesListContainer>
                            {nodes.map(node => (
                                <NodeItem key={node.node_id}>
                                    <ComputerIconStyled />
                                    <ListItemText 
                                        primary={
                                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                                <NodeLink to={`/cluster/runningnodes/${node.node_id}`}>
                                                    {formatNodeId(node.node_id)}
                                                </NodeLink>
                                                <IPChip 
                                                    icon={<RouterIcon fontSize="small" />}
                                                    label={node.node_manager_address}
                                                    size="small"
                                                />
                                                <StatusChip 
                                                    icon={<SignalCellularAltIcon fontSize="small" />}
                                                    label={node.alive ? "Active" : "Inactive"}
                                                    size="small"
                                                    alive={node.alive}
                                                />
                                                <Tooltip title="View Node Details">
                                                    <ViewButton 
                                                        component={Link} 
                                                        to={`/cluster/runningnodes/${node.node_id}`}
                                                        size="small"
                                                    >
                                                        <VisibilityIcon fontSize="small" />
                                                    </ViewButton>
                                                </Tooltip>
                                            </Box>
                                        }
                                    />
                                </NodeItem>
                            ))}
                        </NodesListContainer>
                    )}
                </NodeCard>
            )}
        </ContentContainer>
    );
};

export default RunningNodeList;
