import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { useParams, Link } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  Button,
  Grid,
  Divider,
  Chip,
  Tooltip,
  List,
  ListItem,
  ListItemText,
  ListItemIcon
} from '@mui/material';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import ComputerIcon from '@mui/icons-material/Computer';
import RouterIcon from '@mui/icons-material/Router';
import DnsIcon from '@mui/icons-material/Dns';
import MemoryIcon from '@mui/icons-material/Memory';
import StorageIcon from '@mui/icons-material/Storage';
import SettingsEthernetIcon from '@mui/icons-material/SettingsEthernet';
import SignalCellularAltIcon from '@mui/icons-material/SignalCellularAlt';
import CheckCircleIcon from '@mui/icons-material/CheckCircle';
import CancelIcon from '@mui/icons-material/Cancel';
import { BASE_URL } from '../../const';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const DetailCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  padding: theme.spacing(3),
  marginBottom: theme.spacing(4),
  marginTop: theme.spacing(2),
}));

const DetailSubtitle = styled(Typography)(({ theme }) => ({
  fontSize: '1.3rem',
  fontWeight: 600,
  color: theme.palette.text.primary,
  marginBottom: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1),
    color: theme.palette.primary.main,
  }
}));

const InfoSection = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(3),
}));

const SectionLabel = styled(Typography)(({ theme }) => ({
  fontSize: '1.1rem',
  fontWeight: 600,
  color: theme.palette.primary.main,
  marginBottom: theme.spacing(1),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1),
  }
}));

const StyledList = styled(List)(({ theme }) => ({
  padding: 0,
}));

const DetailItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(1, 0),
  '& svg': {
    color: theme.palette.secondary.light,
  }
}));

const DetailLabel = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontWeight: 500,
  minWidth: '180px',
  flexShrink: 0,
}));

const DetailValue = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.primary,
  fontWeight: 400,
}));

const StatusChip = styled(Chip)(({ theme, alive }) => ({
  backgroundColor: alive ? 'rgba(0, 192, 127, 0.1)' : 'rgba(244, 67, 54, 0.1)',
  color: alive ? theme.palette.success.main : theme.palette.error.main,
  marginLeft: theme.spacing(1),
}));

const ResourceCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  padding: theme.spacing(2),
  height: '100%',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  transition: 'all 0.2s',
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
    boxShadow: '0 6px 16px rgba(0, 0, 0, 0.2)',
  }
}));

const ResourceValue = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontWeight: 700,
  fontSize: '1.8rem',
  marginTop: theme.spacing(1),
}));

const ResourceLabel = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontSize: '0.9rem',
  marginTop: theme.spacing(0.5),
}));

const ButtonContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  gap: theme.spacing(2),
  marginBottom: theme.spacing(3),
}));

const EmptyState = styled(Typography)(({ theme }) => ({
  fontStyle: 'italic',
  color: theme.palette.text.secondary,
}));

const formatMemory = (bytes) => {
  if (!bytes) return 'N/A';
  const gb = bytes / 1024 / 1024 / 1024;
  return `${gb.toFixed(2)} GiB`;
};

const RunningNodeDetail = () => {
    const [node, setNode] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const { node_id } = useParams();

    useEffect(() => {
        const fetchRunningNodeDetail = async () => {
            try {
                const response = await axios.get(`${BASE_URL}/cluster/get_running_nodes`);
                const nodeDetail = response.data.find(node => node.node_id === node_id);
                setNode(nodeDetail);
            } catch (err) {
                setError(err);
            } finally {
                setLoading(false);
            }
        };

        fetchRunningNodeDetail();
    }, [node_id]);

    return (
        <ContentContainer>
            <ButtonContainer>
                <BackButton 
                    component={Link} 
                    to="/cluster/home" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to Cluster Home
                </BackButton>
                
                <BackButton 
                    component={Link} 
                    to="/cluster/runningnodes" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to Running Nodes
                </BackButton>
            </ButtonContainer>
            
            <SectionTitle variant="h4">Running Node Details</SectionTitle>
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={error?.message || 'An error occurred'} />}
            
            {!loading && !error && !node && (
                <EmptyState>Node not found</EmptyState>
            )}
            
            {!loading && !error && node && (
                <>
                    <DetailCard>
                        <DetailSubtitle variant="h5">
                            <ComputerIcon />
                            Node Information
                            <StatusChip 
                                icon={node.alive ? <CheckCircleIcon /> : <CancelIcon />}
                                label={node.alive ? "Active" : "Inactive"}
                                alive={node.alive}
                            />
                        </DetailSubtitle>
                        
                        <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
                        
                        <Grid container spacing={3}>
                            <Grid item xs={12} md={6}>
                                <InfoSection>
                                    <SectionLabel>
                                        <DnsIcon />
                                        Node Details
                                    </SectionLabel>
                                    <StyledList>
                                        <DetailItem>
                                            <DetailLabel>Node ID:</DetailLabel>
                                            <DetailValue>{node.node_id}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Node Address:</DetailLabel>
                                            <DetailValue>{node.node_manager_address}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Node Hostname:</DetailLabel>
                                            <DetailValue>{node.node_manager_hostname}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Node Port:</DetailLabel>
                                            <DetailValue>{node.node_manager_port}</DetailValue>
                                        </DetailItem>
                                    </StyledList>
                                </InfoSection>
                            </Grid>
                            
                            <Grid item xs={12} md={6}>
                                <InfoSection>
                                    <SectionLabel>
                                        <SettingsEthernetIcon />
                                        System Details
                                    </SectionLabel>
                                    <StyledList>
                                        <DetailItem>
                                            <DetailLabel>Object Manager Port:</DetailLabel>
                                            <DetailValue>{node.object_manager_port}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Object Store Socket:</DetailLabel>
                                            <DetailValue>{node.object_store_socket_name}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Raylet Socket:</DetailLabel>
                                            <DetailValue>{node.raylet_socket_name}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Metrics Export Port:</DetailLabel>
                                            <DetailValue>{node.metrics_export_port}</DetailValue>
                                        </DetailItem>
                                    </StyledList>
                                </InfoSection>
                            </Grid>
                        </Grid>
                    </DetailCard>
                    
                    {/* Resources Card */}
                    <DetailCard>
                        <DetailSubtitle variant="h5">
                            <StorageIcon />
                            Resources
                        </DetailSubtitle>
                        
                        <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
                        
                        <Grid container spacing={3}>
                            <Grid item xs={12} sm={6} md={3}>
                                <ResourceCard>
                                    <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                        <MemoryIcon sx={{ color: 'primary.main', mr: 1 }} />
                                        <Typography variant="h6">CPU</Typography>
                                    </Box>
                                    <ResourceValue>
                                        {node.resources?.CPU ? Math.round(node.resources.CPU) : 'N/A'}
                                    </ResourceValue>
                                    <ResourceLabel>Cores</ResourceLabel>
                                </ResourceCard>
                            </Grid>
                            
                            <Grid item xs={12} sm={6} md={3}>
                                <ResourceCard>
                                    <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                        <SignalCellularAltIcon sx={{ color: 'primary.main', mr: 1 }} />
                                        <Typography variant="h6">GPU</Typography>
                                    </Box>
                                    <ResourceValue>
                                        {node.resources?.GPU ? Math.round(node.resources.GPU) : 'N/A'}
                                    </ResourceValue>
                                    <ResourceLabel>Units</ResourceLabel>
                                </ResourceCard>
                            </Grid>
                            
                            <Grid item xs={12} sm={6} md={3}>
                                <ResourceCard>
                                    <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                        <MemoryIcon sx={{ color: 'primary.main', mr: 1 }} />
                                        <Typography variant="h6">Memory</Typography>
                                    </Box>
                                    <ResourceValue>
                                        {formatMemory(node.resources?.memory)}
                                    </ResourceValue>
                                    <ResourceLabel>RAM</ResourceLabel>
                                </ResourceCard>
                            </Grid>
                            
                            <Grid item xs={12} sm={6} md={3}>
                                <ResourceCard>
                                    <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                        <StorageIcon sx={{ color: 'primary.main', mr: 1 }} />
                                        <Typography variant="h6">Object Store</Typography>
                                    </Box>
                                    <ResourceValue>
                                        {formatMemory(node.resources?.object_store_memory)}
                                    </ResourceValue>
                                    <ResourceLabel>Storage</ResourceLabel>
                                </ResourceCard>
                            </Grid>
                        </Grid>
                    </DetailCard>
                </>
            )}
        </ContentContainer>
    );
};

export default RunningNodeDetail;
