import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Link, useLocation } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  List,
  ListItem,
  ListItemText,
  ListItemIcon,
  Button,
  Chip,
  Switch,
  FormControlLabel,
  Tooltip,
  IconButton
} from '@mui/material';
import DeviceHubIcon from '@mui/icons-material/DeviceHub';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import VisibilityIcon from '@mui/icons-material/Visibility';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 0),
}));

const SectionTitle = styled(Typography)(({ theme, embedded }) => ({
  color: theme.palette.primary.main,
  fontSize: embedded ? '1.5rem' : '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const NodeCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
  marginBottom: theme.spacing(4),
}));

const NodesList = styled(List)(({ theme }) => ({
  padding: 0,
}));

const NodeItem = styled(ListItem)(({ theme, inactive }) => ({
  padding: theme.spacing(2, 3),
  borderBottom: '1px solid rgba(255, 255, 255, 0.05)',
  transition: 'background-color 0.2s',
  opacity: inactive ? 0.6 : 1,
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.03)',
  },
  '&:last-child': {
    borderBottom: 'none',
  }
}));

const NodeLink = styled(Link)(({ theme, inactive }) => ({
  color: inactive ? theme.palette.text.secondary : theme.palette.primary.main,
  textDecoration: 'none',
  fontWeight: 500,
  fontSize: '1rem',
  transition: 'all 0.2s',
  flex: 1,
  '&:hover': {
    color: inactive ? theme.palette.text.primary : theme.palette.primary.light,
    textDecoration: 'underline',
  },
}));

const NodeIcon = styled(DeviceHubIcon)(({ theme }) => ({
  color: theme.palette.primary.main,
  marginRight: theme.spacing(1),
  fontSize: '1.2rem',
}));

const EmptyState = styled(Box)(({ theme }) => ({
  padding: theme.spacing(4),
  textAlign: 'center',
  color: theme.palette.text.secondary,
  fontStyle: 'italic',
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const ShowInactiveSwitch = styled(FormControlLabel)(({ theme }) => ({
  marginTop: theme.spacing(2),
  marginBottom: theme.spacing(2),
  '& .MuiFormControlLabel-label': {
    color: theme.palette.text.secondary,
    fontSize: '0.9rem',
  }
}));

const NodeIPChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.05)',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(2),
  fontSize: '0.75rem',
  height: 24,
}));

const ViewButton = styled(IconButton)(({ theme }) => ({
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  padding: theme.spacing(0.75),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const NodeList = ({ isEmbedded = false, nodes: propNodes }) => {
    const [nodes, setNodes] = useState(propNodes || []);
    const [showInactive, setShowInactive] = useState(false);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    const locationPath = useLocation();

    useEffect(() => {
        if (!propNodes) {
            const fetchNodes = async () => {
                setLoading(true);
                try {
                    const uri = `${BASE_URL}/cluster/nodes`;
                    const response = await axios.get(uri);
                    if (response.status === 200) {
                        setNodes(response.data);
                    } else {
                        throw new Error(`Failed to fetch nodes from ${uri}`);
                    }
                } catch (err) {
                    setError(err);
                } finally {
                    setLoading(false);
                }
            };

            fetchNodes();
        }
    }, [propNodes]);

    const getNodeIP = (node) => {
        // Prefer hardware IP if available, fallback to last_config, otherwise 'IP not available'
        return node.hardware?.cluster_ip || node.last_config?.cluster_ip || 'IP not available';
    };

    // Function to truncate node ID for display
    const formatNodeId = (id) => {
        if (!id) return 'Unknown ID';
        if (id.length <= 20) return id;
        return `${id.substring(0, 8)}...${id.substring(id.length - 8)}`;
    };

    const activeNodes = nodes.filter(node => node.active);
    const inactiveNodes = nodes.filter(node => !node.active);
    
    // Decide which nodes to display based on the showInactive state
    const displayNodes = [...activeNodes, ...(showInactive ? inactiveNodes : [])];

    return (
        <ContentContainer>
            {!isEmbedded && locationPath.pathname !== "/cluster/home" && (
                <BackButton 
                    component={Link} 
                    to="/cluster/home" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to Clusters Home
                </BackButton>
            )}
            
            <SectionTitle variant="h4" embedded={isEmbedded}>
                Node List
                <Tooltip title="List of compute nodes in your cluster">
                    <InfoIcon />
                </Tooltip>
            </SectionTitle>
            
            {inactiveNodes.length > 0 && (
                <ShowInactiveSwitch
                    control={
                        <Switch
                            checked={showInactive}
                            onChange={() => setShowInactive(!showInactive)}
                            color="primary"
                            size="small"
                        />
                    }
                    label="Show Inactive Nodes"
                />
            )}
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={error.message} />}
            
            {!loading && !error && (
                <NodeCard>
                    {displayNodes.length === 0 ? (
                        <EmptyState>
                            <Typography variant="body1">No nodes available.</Typography>
                        </EmptyState>
                    ) : (
                        <NodesList>
                            {displayNodes.map(node => (
                                <NodeItem key={node.id} inactive={!node.active}>
                                    <NodeIcon />
                                    <ListItemText 
                                        primary={
                                            <Box sx={{ display: 'flex', alignItems: 'center' }}>
                                                <NodeLink to={`/cluster/node/${node.id}`} inactive={!node.active}>
                                                    {formatNodeId(node.id)}
                                                </NodeLink>
                                                <NodeIPChip 
                                                    label={getNodeIP(node)}
                                                    size="small"
                                                />
                                                <Tooltip title="View Node Details">
                                                    <ViewButton 
                                                        component={Link} 
                                                        to={`/cluster/node/${node.id}`}
                                                        size="small"
                                                    >
                                                        <VisibilityIcon fontSize="small" />
                                                    </ViewButton>
                                                </Tooltip>
                                            </Box>
                                        }
                                    />
                                </NodeItem>
                            ))}
                        </NodesList>
                    )}
                </NodeCard>
            )}
        </ContentContainer>
    );
};

export default NodeList;
