/* global Intl */
import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { useParams, Link } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  Button,
  Grid,
  Divider,
  Chip,
  List,
  ListItem,
  ListItemIcon
} from '@mui/material';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import DeviceHubIcon from '@mui/icons-material/DeviceHub';
import LocationOnIcon from '@mui/icons-material/LocationOn';
import MemoryIcon from '@mui/icons-material/Memory';
import DnsIcon from '@mui/icons-material/Dns';
import RouterIcon from '@mui/icons-material/Router';
import CheckCircleIcon from '@mui/icons-material/CheckCircle';
import CancelIcon from '@mui/icons-material/Cancel';
import SettingsIcon from '@mui/icons-material/Settings';
import { BASE_URL } from '../../const';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const DetailCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  padding: theme.spacing(3),
  marginBottom: theme.spacing(4),
  marginTop: theme.spacing(2),
}));

const DetailSubtitle = styled(Typography)(({ theme }) => ({
  fontSize: '1.3rem',
  fontWeight: 600,
  color: theme.palette.text.primary,
  marginBottom: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1),
    color: theme.palette.primary.main,
  }
}));

const InfoSection = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(3),
}));

const SectionLabel = styled(Typography)(({ theme }) => ({
  fontSize: '1.1rem',
  fontWeight: 600,
  color: theme.palette.primary.main,
  marginBottom: theme.spacing(1),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1),
  }
}));

const StyledList = styled(List)(({ theme }) => ({
  padding: 0,
}));

const DetailItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(1, 0),
  '& svg': {
    color: theme.palette.secondary.light,
  }
}));

const DetailLabel = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontWeight: 500,
  minWidth: '150px',
  flexShrink: 0,
}));

const DetailValue = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.primary,
  fontWeight: 400,
}));

const StyledLink = styled(Link)(({ theme }) => ({
  color: theme.palette.primary.main,
  textDecoration: 'none',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.light,
    textDecoration: 'underline',
  },
}));

const StatusChip = styled(Chip)(({ theme, active }) => ({
  backgroundColor: active ? 'rgba(0, 192, 127, 0.1)' : 'rgba(244, 67, 54, 0.1)',
  color: active ? theme.palette.success.main : theme.palette.error.main,
  marginLeft: theme.spacing(1),
}));

const ButtonContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  gap: theme.spacing(2),
  marginBottom: theme.spacing(3),
}));

const EmptyState = styled(Typography)(({ theme }) => ({
  fontStyle: 'italic',
  color: theme.palette.text.secondary,
}));

const formatDate = (dateString) => {
  if (!dateString) return 'N/A';
  
  const date = new Date(dateString);
  if (isNaN(date.getTime())) return dateString;
  
  return new Intl.DateTimeFormat('en-US', {
    year: 'numeric',
    month: 'long',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit'
  }).format(date);
};

const formatVram = (vram) => {
  if (!vram) return 'N/A';
  const vramStr = String(vram);
  // Return as-is if already has units, is "unknown", or non-numeric
  if (vramStr.toLowerCase() === 'unknown' ||
      vramStr.toLowerCase() === 'n/a' ||
      /[a-zA-Z]/.test(vramStr)) {
    return vramStr;
  }
  // Check if it's a valid number (including decimals)
  if (!isNaN(parseFloat(vramStr))) {
    return `${vramStr} GB`;
  }
  return vramStr;
};

const NodeDetails = () => {
    const [nodeDetails, setNodeDetails] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const { node_id } = useParams();

    useEffect(() => {
        const fetchNodeDetails = async () => {
            try {
                const uri = `${BASE_URL}/cluster/node/${node_id}`;
                const response = await axios.get(uri);
                if (response.status === 200) {
                    setNodeDetails(response.data);
                } else {
                    throw new Error(`Failed to fetch node details from ${uri}`);
                }
            } catch (err) {
                setError(err);
            } finally {
                setLoading(false);
            }
        };

        fetchNodeDetails();
    }, [node_id]);

    const formatList = (items) => {
        if (!items || !Array.isArray(items) || items.length === 0) return 'N/A';
        
        // Special handling for GPUs - detect if this is a GPU array
        if (items.length > 0 && typeof items[0] === 'object' && items[0] !== null && items[0].fullname && items[0].vram) {
            // Return the array directly to be rendered as bullet points
            return items;
        }
        
        // Default behavior for non-GPU arrays
        return items.map(item => {
            if (typeof item === 'object' && item !== null) {
                return item.name || JSON.stringify(item);
            }
            return item;
        }).join(', ');
    };

    return (
        <ContentContainer>
            <ButtonContainer>
                <BackButton 
                    component={Link} 
                    to="/cluster/home" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to Cluster Home
                </BackButton>
                
                <BackButton 
                    component={Link} 
                    to="/cluster/nodes" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to Node List
                </BackButton>
            </ButtonContainer>
            
            <SectionTitle variant="h4">Node Details</SectionTitle>
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={error?.message || 'An error occurred'} />}
            
            {!loading && !error && !nodeDetails && (
                <EmptyState>Node not found</EmptyState>
            )}
            
            {!loading && !error && nodeDetails && (
                <>
                    {/* General Information Card */}
                    <DetailCard>
                        <DetailSubtitle variant="h5">
                            <DeviceHubIcon />
                            {nodeDetails.node.last_config?.name || 'Node ' + node_id.substring(0, 8)}
                            <StatusChip 
                                icon={nodeDetails.node.active ? <CheckCircleIcon /> : <CancelIcon />}
                                label={nodeDetails.node.active ? "Active" : "Inactive"}
                                active={nodeDetails.node.active}
                            />
                        </DetailSubtitle>
                        
                        <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
                        
                        <Grid container spacing={3}>
                            <Grid item xs={12} md={6}>
                                <InfoSection>
                                    <SectionLabel>
                                        <DeviceHubIcon />
                                        Node Information
                                    </SectionLabel>
                                    <StyledList>
                                        <DetailItem>
                                            <DetailLabel>Node ID:</DetailLabel>
                                            <DetailValue>{nodeDetails.node.id}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Last Seen:</DetailLabel>
                                            <DetailValue>{formatDate(nodeDetails.node.last_seen)}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Ray ID:</DetailLabel>
                                            <DetailValue>{nodeDetails.node.ray_id || 'N/A'}</DetailValue>
                                        </DetailItem>
                                        
                                        {nodeDetails.location && (
                                            <DetailItem>
                                                <ListItemIcon>
                                                    <LocationOnIcon />
                                                </ListItemIcon>
                                                <DetailLabel>Location:</DetailLabel>
                                                <DetailValue>
                                                    <StyledLink to={`/cluster/locations/${nodeDetails.location.id}`}>
                                                        {nodeDetails.location.name}
                                                    </StyledLink>
                                                </DetailValue>
                                            </DetailItem>
                                        )}
                                    </StyledList>
                                </InfoSection>
                            </Grid>
                            
                            {nodeDetails.node.last_config && (
                                <Grid item xs={12} md={6}>
                                    <InfoSection>
                                        <SectionLabel>
                                            <RouterIcon />
                                            Network Information
                                        </SectionLabel>
                                        <StyledList>
                                            <DetailItem>
                                                <DetailLabel>Cluster IP:</DetailLabel>
                                                <DetailValue>{nodeDetails.node.last_config.cluster_ip || 'N/A'}</DetailValue>
                                            </DetailItem>
                                            <DetailItem>
                                                <DetailLabel>Local Node ID:</DetailLabel>
                                                <DetailValue>{nodeDetails.node.last_config.local_node_id || 'N/A'}</DetailValue>
                                            </DetailItem>
                                        </StyledList>
                                    </InfoSection>
                                </Grid>
                            )}
                        </Grid>
                    </DetailCard>
                    
                    {/* Hardware Details Card */}
                    {nodeDetails.hardware && (
                        <DetailCard>
                            <DetailSubtitle variant="h5">
                                <MemoryIcon />
                                Hardware Information
                            </DetailSubtitle>
                            
                            <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
                            
                            <Grid container spacing={3}>
                                <Grid item xs={12} md={6}>
                                    <InfoSection>
                                        <SectionLabel>
                                            <MemoryIcon />
                                            Hardware Details
                                        </SectionLabel>
                                        <StyledList>
                                            <DetailItem>
                                                <DetailLabel>Name:</DetailLabel>
                                                <DetailValue>
                                                    <StyledLink to={`/cluster/hardware/${nodeDetails.hardware.id}`}>
                                                        {nodeDetails.hardware.name || `Hardware ${nodeDetails.hardware.id}`}
                                                    </StyledLink>
                                                </DetailValue>
                                            </DetailItem>
                                            
                                            <DetailItem>
                                                <DetailLabel>Processors:</DetailLabel>
                                                <DetailValue>
                                                    {Array.isArray(nodeDetails.hardware.processors) 
                                                        ? formatList(nodeDetails.hardware.processors)
                                                        : 'N/A'}
                                                </DetailValue>
                                            </DetailItem>

                                            <DetailItem>
                                                <DetailLabel>GPUs:</DetailLabel>
                                                <DetailValue>
                                                    {Array.isArray(nodeDetails.hardware.gpus) 
                                                        ? (Array.isArray(formatList(nodeDetails.hardware.gpus)) 
                                                            ? (
                                                                // Check if there's only one GPU
                                                                formatList(nodeDetails.hardware.gpus).length === 1 
                                                                ? `${formatList(nodeDetails.hardware.gpus)[0].fullname} (${formatVram(formatList(nodeDetails.hardware.gpus)[0].vram)})`
                                                                : (
                                                                    // Multiple GPUs - render as bullet list
                                                                    <Box component="ul" sx={{ m: 0, pl: 2 }}>
                                                                        {formatList(nodeDetails.hardware.gpus).map((gpu, index) => (
                                                                            <Box component="li" key={index} sx={{ mb: 0.5 }}>
                                                                                {`${gpu.fullname} (${formatVram(gpu.vram)})`}
                                                                            </Box>
                                                                        ))}
                                                                    </Box>
                                                                )
                                                            )
                                                            : formatList(nodeDetails.hardware.gpus)
                                                        )
                                                        : 'N/A'}
                                                </DetailValue>
                                            </DetailItem>
                                            
                                        </StyledList>
                                    </InfoSection>
                                </Grid>
                                
                                <Grid item xs={12} md={6}>
                                    <InfoSection>
                                        <SectionLabel>
                                            <DnsIcon />
                                            System Information
                                        </SectionLabel>
                                        <StyledList>
                                            <DetailItem>
                                                <DetailLabel>OS:</DetailLabel>
                                                <DetailValue>{nodeDetails.hardware.os || 'N/A'}</DetailValue>
                                            </DetailItem>
                                            
                                            <DetailItem>
                                                <DetailLabel>Platform:</DetailLabel>
                                                <DetailValue>{nodeDetails.hardware.platform || 'N/A'}</DetailValue>
                                            </DetailItem>
                                        </StyledList>
                                    </InfoSection>
                                </Grid>
                            </Grid>
                        </DetailCard>
                    )}
                    
                    {/* Location Details Card */}
                    {nodeDetails.location && (
                        <DetailCard>
                            <DetailSubtitle variant="h5">
                                <LocationOnIcon />
                                Location Information
                            </DetailSubtitle>
                            
                            <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
                            
                            <Grid container spacing={3}>
                                <Grid item xs={12} md={6}>
                                    <InfoSection>
                                        <SectionLabel>
                                            <LocationOnIcon />
                                            Location Details
                                        </SectionLabel>
                                        <StyledList>
                                            <DetailItem>
                                                <DetailLabel>Name:</DetailLabel>
                                                <DetailValue>
                                                    <StyledLink to={`/cluster/locations/${nodeDetails.location.id}`}>
                                                        {nodeDetails.location.name}
                                                    </StyledLink>
                                                </DetailValue>
                                            </DetailItem>
                                            
                                            <DetailItem>
                                                <DetailLabel>Region:</DetailLabel>
                                                <DetailValue>{nodeDetails.location.region || 'N/A'}</DetailValue>
                                            </DetailItem>
                                            
                                            <DetailItem>
                                                <DetailLabel>Zone:</DetailLabel>
                                                <DetailValue>{nodeDetails.location.zone || 'N/A'}</DetailValue>
                                            </DetailItem>
                                            
                                            <DetailItem>
                                                <DetailLabel>Datacenter:</DetailLabel>
                                                <DetailValue>{nodeDetails.location.datacenter || 'N/A'}</DetailValue>
                                            </DetailItem>
                                            
                                            <DetailItem>
                                                <DetailLabel>Building:</DetailLabel>
                                                <DetailValue>{nodeDetails.location.building || 'N/A'}</DetailValue>
                                            </DetailItem>
                                            
                                            <DetailItem>
                                                <DetailLabel>Address:</DetailLabel>
                                                <DetailValue>{nodeDetails.location.address || 'N/A'}</DetailValue>
                                            </DetailItem>
                                        </StyledList>
                                    </InfoSection>
                                </Grid>
                                
                                <Grid item xs={12} md={6}>
                                    <InfoSection>
                                        <SectionLabel>
                                            <DnsIcon />
                                            Contact Information
                                        </SectionLabel>
                                        <StyledList>
                                            <DetailItem>
                                                <DetailLabel>Contact Name:</DetailLabel>
                                                <DetailValue>{nodeDetails.location.contact_name || 'N/A'}</DetailValue>
                                            </DetailItem>
                                            
                                            <DetailItem>
                                                <DetailLabel>Contact Email:</DetailLabel>
                                                <DetailValue>{nodeDetails.location.contact_email || 'N/A'}</DetailValue>
                                            </DetailItem>
                                            
                                            <DetailItem>
                                                <DetailLabel>Contact Phone:</DetailLabel>
                                                <DetailValue>{nodeDetails.location.contact_phone || 'N/A'}</DetailValue>
                                            </DetailItem>
                                        </StyledList>
                                    </InfoSection>
                                </Grid>
                            </Grid>
                        </DetailCard>
                    )}
                    
                    {/* Last Configuration Card */}
                    {nodeDetails.node.last_config && (
                        <DetailCard>
                            <DetailSubtitle variant="h5">
                                <SettingsIcon />
                                Last Configuration
                            </DetailSubtitle>
                            
                            <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
                            
                            <Grid container spacing={3}>
                                <Grid item xs={12} md={6}>
                                    <StyledList>
                                        <DetailItem>
                                            <DetailLabel>OS:</DetailLabel>
                                            <DetailValue>{nodeDetails.node.last_config.os || 'N/A'}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Platform:</DetailLabel>
                                            <DetailValue>{nodeDetails.node.last_config.platform || 'N/A'}</DetailValue>
                                        </DetailItem>
                                        
                                        <DetailItem>
                                            <DetailLabel>Processors:</DetailLabel>
                                            <DetailValue>
                                                {Array.isArray(nodeDetails.node.last_config.processors) 
                                                    ? formatList(nodeDetails.node.last_config.processors)
                                                    : 'N/A'}
                                            </DetailValue>
                                        </DetailItem>
                                    </StyledList>
                                </Grid>
                                
                                <Grid item xs={12} md={6}>
                                    <StyledList>
                                        <DetailItem>
                                            <DetailLabel>GPUs:</DetailLabel>
                                            <DetailValue>
                                                {Array.isArray(nodeDetails.node.last_config.gpus) 
                                                    ? (Array.isArray(formatList(nodeDetails.node.last_config.gpus)) 
                                                        ? (
                                                            <Box component="ul" sx={{ m: 0, pl: 2 }}>
                                                                {formatList(nodeDetails.node.last_config.gpus).map((gpu, index) => (
                                                                    <Box component="li" key={index} sx={{ mb: 0.5 }}>
                                                                        {`${gpu.fullname} (${formatVram(gpu.vram)})`}
                                                                    </Box>
                                                                ))}
                                                            </Box>
                                                        )
                                                        : formatList(nodeDetails.node.last_config.gpus)
                                                    )
                                                    : 'N/A'}
                                            </DetailValue>
                                        </DetailItem>
                                        
                                        {nodeDetails.node.last_config.configuration && (
                                            <DetailItem>
                                                <DetailLabel>Configuration:</DetailLabel>
                                                <DetailValue>
                                                    {typeof nodeDetails.node.last_config.configuration === 'object'
                                                        ? JSON.stringify(nodeDetails.node.last_config.configuration, null, 2)
                                                        : nodeDetails.node.last_config.configuration || 'N/A'}
                                                </DetailValue>
                                            </DetailItem>
                                        )}
                                    </StyledList>
                                </Grid>
                            </Grid>
                        </DetailCard>
                    )}
                </>
            )}
        </ContentContainer>
    );
};

export default NodeDetails;
