import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { BASE_URL } from '../../const';
import { useParams, Link } from 'react-router-dom';
import { StyledMainContent, StyledHeader, StyledParagraph, StyledButton } from '../../StyledComponents';
import { useCluster } from '../../contexts/ClusterContext';
import ArrowBackIosIcon from '@mui/icons-material/ArrowBackIos';

const NodeDetail = () => {
    const [node, setNode] = useState([]);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    const { twoNodeInfo } = useCluster();

    let { '*': id } = useParams();

    useEffect(() => {
        const fetchNodeDetails = async () => {
            setLoading(true);
            try {
                const uri = `${BASE_URL}/cluster/node/${id}`;
                const response = await axios.get(uri);
                if (response.status === 200) {
                    setNode(response.data);
                } else {
                    throw new Error(`Failed to fetch node details from ${uri}`);
                }
                setLoading(false);
            } catch (err) {
                setError(err);
                setLoading(false);
            }
        };

        fetchNodeDetails();
    }, [id]);

    return (
        <StyledMainContent>
            <StyledButton component={Link} to="/cluster/home" variant="outlined" startIcon={<ArrowBackIosIcon />}>
                Clusters Home
            </StyledButton>
            <StyledHeader>Node Details</StyledHeader>
            {loading && <StyledParagraph>Loading...</StyledParagraph>}
            {error && <StyledParagraph>Error: {error.message}</StyledParagraph>}
            {twoNodeInfo && (
                <div className="node-card" style={{ marginBottom: '16px', border: '1px solid rgba(255,255,255,0.1)', borderRadius: 12, padding: 16 }}>
                    <StyledHeader>Paired GPU Node</StyledHeader>
                    <StyledParagraph>
                        <strong>Host:</strong> {twoNodeInfo.remote_host || 'Unknown'}
                    </StyledParagraph>
                    {twoNodeInfo.remote_user && (
                        <StyledParagraph>
                            <strong>User:</strong> {twoNodeInfo.remote_user}
                        </StyledParagraph>
                    )}
                    <StyledParagraph>
                        <strong>Status:</strong> Remote execution enabled
                    </StyledParagraph>
                </div>
            )}
            <div className="node-list">
                {node && (
                    <div key={node.id} className="node-card">
                        <StyledHeader>{node.name}</StyledHeader>
                        <StyledParagraph><strong>ID:</strong> {node.id}</StyledParagraph>
                        <StyledParagraph><strong>Created At:</strong> {node.created_at}</StyledParagraph>
                    </div>
                )}
            </div>
        </StyledMainContent>
    );
};

export default NodeDetail;
