import React, { useState, useEffect } from 'react';
import { Link, useLocation } from 'react-router-dom';
import axios from 'axios';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  List,
  ListItem,
  ListItemText,
  ListItemIcon,
  Button,
  Divider,
  Tooltip
} from '@mui/material';
import LocationOnIcon from '@mui/icons-material/LocationOn';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import AddIcon from '@mui/icons-material/Add';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 0),
}));

const SectionTitle = styled(Typography)(({ theme, embedded }) => ({
  color: theme.palette.primary.main,
  fontSize: embedded ? '1.5rem' : '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const ActionButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  marginLeft: theme.spacing(2),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: '#fff',
  backgroundColor: theme.palette.primary.main,
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: theme.palette.primary.dark,
  }
}));

const LocationCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
  marginBottom: theme.spacing(4),
}));

const LocationsList = styled(List)(({ theme }) => ({
  padding: 0,
}));

const LocationItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(2, 3),
  borderBottom: '1px solid rgba(255, 255, 255, 0.05)',
  transition: 'background-color 0.2s',
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.03)',
  },
  '&:last-child': {
    borderBottom: 'none',
  }
}));

const LocationLink = styled(Link)(({ theme }) => ({
  color: theme.palette.primary.main,
  textDecoration: 'none',
  fontWeight: 500,
  fontSize: '1rem',
  transition: 'all 0.2s',
  display: 'flex',
  alignItems: 'center',
  '&:hover': {
    color: theme.palette.primary.light,
    textDecoration: 'underline',
  },
}));

const LocationIcon = styled(LocationOnIcon)(({ theme }) => ({
  color: theme.palette.primary.main,
  marginRight: theme.spacing(1),
  fontSize: '1.2rem',
}));

const EmptyState = styled(Box)(({ theme }) => ({
  padding: theme.spacing(4),
  textAlign: 'center',
  color: theme.palette.text.secondary,
  fontStyle: 'italic',
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const HeaderContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'flex-start',
  marginBottom: theme.spacing(2),
}));

const LocationList = ({ isEmbedded = false }) => {
    const [locations, setLocations] = useState([]);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState(null);
    const locationPath = useLocation();

    useEffect(() => {
        const fetchLocations = async () => {
            setLoading(true);
            try {
                const uri = `${BASE_URL}/cluster/locations`;
                const response = await axios.get(uri);
                if (response.status === 200) {
                    setLocations(response.data);
                } else {
                    throw new Error(`Failed to fetch locations from ${uri}`);
                }
            } catch (err) {
                setError(err);
            } finally {
                setLoading(false);
            }
        };

        fetchLocations();
    }, []);

    return (
        <ContentContainer>
            {!isEmbedded && locationPath.pathname !== "/cluster/home" && (
                <BackButton 
                    component={Link} 
                    to="/cluster/home" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to Clusters Home
                </BackButton>
            )}
            
            <HeaderContainer>
                <SectionTitle variant="h4" embedded={isEmbedded}>
                    Location List
                    <Tooltip title="List of locations for your computing resources">
                        <InfoIcon />
                    </Tooltip>
                </SectionTitle>
                
                {/* Optional: Add Location button would go here */}
                {/* <ActionButton
                    startIcon={<AddIcon />}
                    size="small"
                >
                    Add Location
                </ActionButton> */}
            </HeaderContainer>
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={error.message} />}
            
            {!loading && !error && (
                <LocationCard>
                    {locations.length === 0 ? (
                        <EmptyState>
                            <Typography variant="body1">No locations available.</Typography>
                        </EmptyState>
                    ) : (
                        <LocationsList>
                            {locations.map(location => (
                                <LocationItem key={location.id}>
                                    <ListItemIcon style={{ minWidth: 'auto', marginRight: '8px' }}>
                                        <LocationIcon />
                                    </ListItemIcon>
                                    <ListItemText 
                                        primary={
                                            <LocationLink to={`/cluster/locations/${location.id}`}>
                                                {location.name || `Default Location ${location.id}`}
                                            </LocationLink>
                                        }
                                    />
                                </LocationItem>
                            ))}
                        </LocationsList>
                    )}
                </LocationCard>
            )}
        </ContentContainer>
    );
};

export default LocationList;
