import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { useParams, Link } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  Button,
  Grid,
  Divider,
  Chip,
  Tooltip,
  List,
  ListItem,
  ListItemText,
  ListItemIcon
} from '@mui/material';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import LocationOnIcon from '@mui/icons-material/LocationOn';
import BusinessIcon from '@mui/icons-material/Business';
import LanguageIcon from '@mui/icons-material/Language';
import HomeIcon from '@mui/icons-material/Home';
import PhoneIcon from '@mui/icons-material/Phone';
import PersonIcon from '@mui/icons-material/Person';
import EmailIcon from '@mui/icons-material/Email';
import EditIcon from '@mui/icons-material/Edit';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import AddLocationModal from './AddLocationForm';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const EditButton = styled(Button)(({ theme }) => ({
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: '#fff',
  backgroundColor: theme.palette.primary.main,
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  marginTop: theme.spacing(2),
  '&:hover': {
    backgroundColor: theme.palette.primary.dark,
  }
}));

const DetailCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  padding: theme.spacing(3),
  marginBottom: theme.spacing(4),
  marginTop: theme.spacing(2),
}));

const DetailSubtitle = styled(Typography)(({ theme }) => ({
  fontSize: '1.3rem',
  fontWeight: 600,
  color: theme.palette.text.primary,
  marginBottom: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1),
    color: theme.palette.primary.main,
  }
}));

const InfoSection = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(3),
}));

const SectionLabel = styled(Typography)(({ theme }) => ({
  fontSize: '1.1rem',
  fontWeight: 600,
  color: theme.palette.primary.main,
  marginBottom: theme.spacing(1),
}));

const DetailItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(1, 0),
  '& svg': {
    color: theme.palette.primary.main
  }
}));

const DetailLabel = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontWeight: 500,
  minWidth: '120px',
  flexShrink: 0,
}));

const DetailValue = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.primary,
  fontWeight: 400,
}));

const EmptyState = styled(Typography)(({ theme }) => ({
  fontStyle: 'italic',
  color: theme.palette.text.secondary,
}));

const ButtonContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  gap: theme.spacing(2),
  marginBottom: theme.spacing(3),
}));

const LocationDetails = () => {
    const [location, setLocation] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [showModal, setShowModal] = useState(false);
    const { location_id } = useParams();

    const fetchLocationDetails = async () => {
        try {
            const response = await axios.get(`${BASE_URL}/cluster/location/${location_id}`);
            setLocation(response.data);
        } catch (err) {
            setError(err);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchLocationDetails();
    }, [location_id]);

    const handleUpdateLocation = async (updatedLocation) => {
        try {
            await axios.put(`${BASE_URL}/cluster/location/${location_id}`, updatedLocation);
            fetchLocationDetails(); // Refresh the location details after update
            setShowModal(false);
        } catch (err) {
            console.error('Update location error:', err);
            setError(err);
        }
    };

    return (
        <ContentContainer>
            <ButtonContainer>
                <BackButton 
                    component={Link} 
                    to="/cluster/home" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to Cluster Home
                </BackButton>
                
                <BackButton 
                    component={Link} 
                    to="/cluster/locations" 
                    startIcon={<ArrowBackIcon />}
                    size="small"
                >
                    Back to Location List
                </BackButton>
            </ButtonContainer>
            
            <SectionTitle variant="h4">Location Details</SectionTitle>
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={error?.message || 'An error occurred'} />}
            
            {!loading && !error && !location && (
                <EmptyState>Location not found</EmptyState>
            )}
            
            {!loading && !error && location && (
                <DetailCard>
                    <DetailSubtitle variant="h5">
                        <LocationOnIcon />
                        {location.name || `Location ${location_id}`}
                    </DetailSubtitle>
                    
                    <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
                    
                    <Grid container spacing={4}>
                        <Grid item xs={12} md={6}>
                            <InfoSection>
                                <SectionLabel>Location Information</SectionLabel>
                                <List dense disablePadding>
                                    <DetailItem>
                                        <ListItemIcon>
                                            <LanguageIcon />
                                        </ListItemIcon>
                                        <DetailLabel>Region:</DetailLabel>
                                        <DetailValue>{location.region || 'N/A'}</DetailValue>
                                    </DetailItem>
                                    
                                    <DetailItem>
                                        <ListItemIcon>
                                            <LocationOnIcon />
                                        </ListItemIcon>
                                        <DetailLabel>Zone:</DetailLabel>
                                        <DetailValue>{location.zone || 'N/A'}</DetailValue>
                                    </DetailItem>
                                    
                                    <DetailItem>
                                        <ListItemIcon>
                                            <BusinessIcon />
                                        </ListItemIcon>
                                        <DetailLabel>Datacenter:</DetailLabel>
                                        <DetailValue>{location.datacenter || 'N/A'}</DetailValue>
                                    </DetailItem>
                                    
                                    <DetailItem>
                                        <ListItemIcon>
                                            <HomeIcon />
                                        </ListItemIcon>
                                        <DetailLabel>Building:</DetailLabel>
                                        <DetailValue>{location.building || 'N/A'}</DetailValue>
                                    </DetailItem>
                                    
                                    <DetailItem>
                                        <ListItemIcon>
                                            <HomeIcon />
                                        </ListItemIcon>
                                        <DetailLabel>Address:</DetailLabel>
                                        <DetailValue>{location.address || 'N/A'}</DetailValue>
                                    </DetailItem>
                                </List>
                            </InfoSection>
                        </Grid>
                        
                        <Grid item xs={12} md={6}>
                            <InfoSection>
                                <SectionLabel>Contact Information</SectionLabel>
                                <List dense disablePadding>
                                    <DetailItem>
                                        <ListItemIcon>
                                            <PersonIcon />
                                        </ListItemIcon>
                                        <DetailLabel>Contact Name:</DetailLabel>
                                        <DetailValue>{location.contact_name || 'N/A'}</DetailValue>
                                    </DetailItem>
                                    
                                    <DetailItem>
                                        <ListItemIcon>
                                            <EmailIcon />
                                        </ListItemIcon>
                                        <DetailLabel>Contact Email:</DetailLabel>
                                        <DetailValue>{location.contact_email || 'N/A'}</DetailValue>
                                    </DetailItem>
                                    
                                    <DetailItem>
                                        <ListItemIcon>
                                            <PhoneIcon />
                                        </ListItemIcon>
                                        <DetailLabel>Contact Phone:</DetailLabel>
                                        <DetailValue>{location.contact_phone || 'N/A'}</DetailValue>
                                    </DetailItem>
                                </List>
                            </InfoSection>
                        </Grid>
                    </Grid>
                    
                    <EditButton 
                        startIcon={<EditIcon />}
                        onClick={() => setShowModal(true)}
                    >
                        Update Location
                    </EditButton>
                </DetailCard>
            )}
            
            {showModal && (
                <AddLocationModal
                    closeModal={() => setShowModal(false)}
                    showModal={showModal}
                    addLocation={handleUpdateLocation}
                    locationDetails={location}
                />
            )}
        </ContentContainer>
    );
};

export default LocationDetails;
