import React, { useState, useEffect } from 'react';
import PropTypes from 'prop-types';
import axios from 'axios';
import { Link, useLocation } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import { useCluster } from '../../contexts/ClusterContext';
import {
  Typography,
  Box,
  Paper,
  List,
  ListItem,
  ListItemText,
  Button,
  Chip,
  Switch,
  FormControlLabel,
  Tooltip,
  IconButton,
  Snackbar,
  Alert
} from '@mui/material';
import {
  DeviceHub as DeviceHubIcon,
  Memory as MemoryIcon,
  Refresh as RefreshIcon
} from '@mui/icons-material';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import VisibilityIcon from '@mui/icons-material/Visibility';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 0),
}));

const SectionTitle = styled(Typography, {
  shouldForwardProp: prop => prop !== 'embedded',
})(({ theme, embedded }) => ({
  color: theme.palette.primary.main,
  fontSize: embedded ? '1.5rem' : '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  },
}));

const HardwareCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
  marginBottom: theme.spacing(4),
}));

const HardwareListContainer = styled(List)(({ theme }) => ({
  padding: 0,
}));

const HardwareItem = styled(ListItem, {
  shouldForwardProp: prop => prop !== 'inactive',
})(({ theme, inactive }) => ({
  padding: theme.spacing(2, 3),
  borderBottom: '1px solid rgba(255, 255, 255, 0.05)',
  transition: 'background-color 0.2s',
  opacity: inactive ? 0.6 : 1,
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.03)',
  },
  '&:last-child': {
    borderBottom: 'none',
  },
}));

const HardwareLink = styled(Link, {
  shouldForwardProp: prop => prop !== 'inactive',
})(({ theme, inactive }) => ({
  color: inactive ? theme.palette.text.secondary : theme.palette.primary.main,
  textDecoration: 'none',
  fontWeight: 500,
  fontSize: '1rem',
  transition: 'all 0.2s',
  flex: 1,
  '&:hover': {
    color: inactive ? theme.palette.text.primary : theme.palette.primary.light,
    textDecoration: 'underline',
  },
}));

const HardwareIcon = styled(MemoryIcon, {
  shouldForwardProp: prop => prop !== 'inactive',
})(({ theme, inactive }) => ({
  color: inactive ? theme.palette.text.disabled : theme.palette.primary.main,
  marginRight: theme.spacing(1),
  fontSize: '1.2rem',
}));

const EmptyState = styled(Box)(({ theme }) => ({
  padding: theme.spacing(4),
  textAlign: 'center',
  color: theme.palette.text.secondary,
  fontStyle: 'italic',
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const ShowInactiveSwitch = styled(FormControlLabel)(({ theme }) => ({
  marginTop: theme.spacing(2),
  marginBottom: theme.spacing(2),
  '& .MuiFormControlLabel-label': {
    color: theme.palette.text.secondary,
    fontSize: '0.9rem',
  },
}));

const NodeIdChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.05)',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(2),
  fontSize: '0.75rem',
  height: 24,
}));

const ViewButton = styled(IconButton)(({ theme }) => ({
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  padding: theme.spacing(0.75),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  },
}));

const GpuChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(0, 192, 127, 0.15)',
  color: theme.palette.primary.main,
  marginLeft: theme.spacing(1),
  fontSize: '0.75rem',
  height: 24,
  '& .MuiChip-icon': {
    color: theme.palette.primary.main,
  },
}));

const GpuInfo = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  marginTop: theme.spacing(1),
  fontSize: '0.8rem',
  color: theme.palette.text.secondary,
}));

const SummaryCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  padding: theme.spacing(2, 3),
  marginBottom: theme.spacing(3),
  display: 'flex',
  justifyContent: 'space-between',
  alignItems: 'center',
}));

const SummaryItem = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  '& .MuiSvgIcon-root': {
    marginRight: theme.spacing(1),
    color: theme.palette.primary.main,
  },
}));

const RefreshButton = styled(Button)(({ theme }) => ({
  background: 'rgba(255, 255, 255, 0.2)',
  color: 'white',
  border: '1px solid rgba(255, 255, 255, 0.3)',
  marginLeft: theme.spacing(2),
  '&:hover': {
    background: 'rgba(255, 255, 255, 0.3)',
    border: '1px solid rgba(255, 255, 255, 0.5)',
  },
}));

const HardwareList = ({ isEmbedded = false }) => {
  const [hardware, setHardware] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [refreshing, setRefreshing] = useState(false);
  const [snackbar, setSnackbar] = useState({ open: false, message: '', severity: 'success' });
  const [showInactive, setShowInactive] = useState(false);
  const locationPath = useLocation();
  
  // Use cluster context to get capabilities instead of making duplicate API call
  const { refetchCapabilities } = useCluster();

  const fetchHardwareData = async () => {
    try {
      setLoading(true);
      
      // Only fetch hardware list, use cluster capabilities from context
      const hardwareResponse = await axios.get(`${BASE_URL}/cluster/hardware`);
      
      if (hardwareResponse.status === 200) {
        console.log('Hardware data received:', hardwareResponse.data);
        setHardware(hardwareResponse.data);
      } else {
        throw new Error(`Failed to fetch hardware from ${BASE_URL}/cluster/hardware`);
      }
      
    } catch (err) {
      setError(err);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchHardwareData();
  }, []);

  const handleRefresh = async () => {
    try {
      setRefreshing(true);
      
      // Call the refresh hardware endpoint
      const refreshResponse = await axios.post(`${BASE_URL}/cluster/refresh_hardware`);
      
      if (refreshResponse.status === 200) {
        // Show success message
        setSnackbar({
          open: true,
          message: 'Hardware information refreshed successfully!',
          severity: 'success'
        });
        
        // Refresh both hardware data and cluster capabilities
        await Promise.all([
          fetchHardwareData(),
          refetchCapabilities()
        ]);
      }
    } catch (err) {
      console.error('Failed to refresh hardware:', err);
      setSnackbar({
        open: true,
        message: 'Failed to refresh hardware information',
        severity: 'error'
      });
    } finally {
      setRefreshing(false);
    }
  };

  const handleCloseSnackbar = () => {
    setSnackbar({ ...snackbar, open: false });
  };

  // Function to truncate node ID for display
  const formatNodeId = id => {
    if (!id) return 'N/A';
    if (id.length <= 20) return id;
    return `${id.substring(0, 8)}...${id.substring(id.length - 8)}`;
  };

  const activeHardware = hardware.filter(hw => hw.active !== false);
  const inactiveHardware = hardware.filter(hw => hw.active === false);

  // Decide which hardware to display based on the showInactive state
  const displayHardware = [...activeHardware, ...(showInactive ? inactiveHardware : [])];

  // Aggregate totals from the displayed hardware list
  const totalGpuCount = displayHardware.reduce((sum, hw) => {
    const gpus = Array.isArray(hw?.gpus) ? hw.gpus : [];
    return sum + gpus.length;
  }, 0);
  const totalVramGb = displayHardware.reduce((sum, hw) => {
    const gpus = Array.isArray(hw?.gpus) ? hw.gpus : [];
    const hwVram = gpus.reduce((s, g) => s + (parseFloat(g?.vram) || 0), 0);
    return sum + hwVram;
  }, 0);

  return (
    <ContentContainer>
      {!isEmbedded && locationPath.pathname !== '/cluster/home' && (
        <BackButton component={Link} to="/cluster/home" startIcon={<ArrowBackIcon />} size="small">
          Back to Clusters Home
        </BackButton>
      )}

      <SectionTitle variant="h4" embedded={isEmbedded}>
        Hardware List
        <Tooltip title="List of physical hardware in your cluster">
          <InfoIcon />
        </Tooltip>
      </SectionTitle>

      {inactiveHardware.length > 0 && (
        <ShowInactiveSwitch
          control={
            <Switch
              checked={showInactive}
              onChange={() => setShowInactive(!showInactive)}
              color="primary"
              size="small"
            />
          }
          label="Show Inactive Hardware"
        />
      )}

      {loading && <Spinner />}
      {error && <ErrorComponent message={error.message} />}

      {!loading && !error && (
        <>
          {/* Summary Card */}
          <SummaryCard>
            <SummaryItem>
              <DeviceHubIcon />
              <Typography variant="h6" color="text.primary">
                Total Hardware: {hardware.length}
              </Typography>
            </SummaryItem>
            <SummaryItem>
              <MemoryIcon />
              <Typography variant="h6" color="text.primary">
                Total GPUs: {totalGpuCount}
              </Typography>
            </SummaryItem>
            <SummaryItem>
              <MemoryIcon />
              <Typography variant="h6" color="text.primary">
                Total VRAM: {totalVramGb.toFixed(2)} GB
              </Typography>
            </SummaryItem>
            <RefreshButton
              variant="outlined"
              startIcon={<RefreshIcon />}
              onClick={handleRefresh}
              disabled={refreshing}
            >
              {refreshing ? 'Refreshing...' : 'Refresh'}
            </RefreshButton>
          </SummaryCard>

          <HardwareCard>
            {displayHardware.length === 0 ? (
              <EmptyState>
                <Typography variant="body1">No hardware available.</Typography>
              </EmptyState>
            ) : (
              <HardwareListContainer data-tour="hardware-list">
                {displayHardware.map(hw => (
                  <HardwareItem key={hw.id} inactive={hw.active === false}>
                    <HardwareIcon inactive={hw.active === false} />
                    <ListItemText
                      primary={
                        <Box sx={{ display: 'flex', alignItems: 'center' }}>
                          <HardwareLink
                            to={`/cluster/hardware/${hw.id}`}
                            inactive={hw.active === false}
                          >
                            {hw.name || `Hardware ${hw.id}`}
                          </HardwareLink>
                          {hw.node_id && (
                            <NodeIdChip
                              icon={<DeviceHubIcon fontSize="small" />}
                              label={formatNodeId(hw.node_id)}
                              size="small"
                            />
                          )}
                          {hw.gpus && hw.gpus.length > 0 && (
                            <GpuChip
                              icon={<MemoryIcon fontSize="small" />}
                              label={`${hw.gpus.length} GPU${hw.gpus.length > 1 ? 's' : ''}`}
                              size="small"
                            />
                          )}
                          <Tooltip title="View Hardware Details">
                            <ViewButton
                              component={Link}
                              to={`/cluster/hardware/${hw.id}`}
                              size="small"
                            >
                              <VisibilityIcon fontSize="small" />
                            </ViewButton>
                          </Tooltip>
                        </Box>
                      }
                      secondary={
                        hw.gpus && hw.gpus.length > 0 ? (
                          <GpuInfo>
                            {hw.gpus.map((gpu, index) => (
                              <span key={index}>
                                {gpu.fullname || gpu.name} ({gpu.vram}GB)
                                {index < hw.gpus.length - 1 ? ', ' : ''}
                              </span>
                            ))}
                          </GpuInfo>
                        ) : null
                      }
                    />
                  </HardwareItem>
                ))}
              </HardwareListContainer>
            )}
          </HardwareCard>
        </>
      )}

      <Snackbar
        open={snackbar.open}
        autoHideDuration={6000}
        onClose={handleCloseSnackbar}
        anchorOrigin={{ vertical: 'top', horizontal: 'right' }}
      >
        <Alert onClose={handleCloseSnackbar} severity={snackbar.severity}>
          {snackbar.message}
        </Alert>
      </Snackbar>
    </ContentContainer>
  );
};

HardwareList.propTypes = {
  isEmbedded: PropTypes.bool,
};

export default HardwareList;
