import React, { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import axios from 'axios';
import { useParams } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  Button,
  Grid,
  Divider,
  Chip,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  List,
  ListItem,
} from '@mui/material';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import MemoryIcon from '@mui/icons-material/Memory';
import DesktopWindowsIcon from '@mui/icons-material/DesktopWindows';
import DeviceHubIcon from '@mui/icons-material/DeviceHub';
import CheckCircleIcon from '@mui/icons-material/CheckCircle';
import CancelIcon from '@mui/icons-material/Cancel';
import SettingsIcon from '@mui/icons-material/Settings';
import { BASE_URL } from '../../const';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { useCluster } from '../../contexts/ClusterContext';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  },
}));

const DetailCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  padding: theme.spacing(3),
  marginBottom: theme.spacing(4),
  marginTop: theme.spacing(2),
}));

const DetailSubtitle = styled(Typography)(({ theme }) => ({
  fontSize: '1.3rem',
  fontWeight: 600,
  color: theme.palette.text.primary,
  marginBottom: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1),
    color: theme.palette.primary.main,
  },
}));

const InfoSection = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(3),
}));

const SectionLabel = styled(Typography)(({ theme }) => ({
  fontSize: '1.1rem',
  fontWeight: 600,
  color: theme.palette.primary.main,
  marginBottom: theme.spacing(1),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1),
  },
}));

const StyledList = styled(List)(({ theme }) => ({
  padding: 0,
}));

const DetailItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(1, 0),
  '& svg': {
    color: theme.palette.secondary.light,
  },
}));

const DetailLabel = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontWeight: 500,
  minWidth: '180px',
  flexShrink: 0,
}));

const DetailValue = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.primary,
  fontWeight: 400,
}));

const StatusChip = styled(Chip, {
  shouldForwardProp: prop => prop !== 'active',
})(({ theme, active }) => ({
  backgroundColor: active ? 'rgba(0, 192, 127, 0.1)' : 'rgba(244, 67, 54, 0.1)',
  color: active ? theme.palette.success.main : theme.palette.error.main,
  marginLeft: theme.spacing(1),
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.02)',
  borderRadius: theme.shape.borderRadius,
  marginTop: theme.spacing(2),
  marginBottom: theme.spacing(3),
  border: '1px solid rgba(255, 255, 255, 0.05)',
  maxWidth: '100%',
  overflow: 'auto',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.05)',
  },
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '0.9rem',
  paddingTop: theme.spacing(1.5),
  paddingBottom: theme.spacing(1.5),
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  fontSize: '0.9rem',
  color: theme.palette.text.primary,
}));

const ButtonContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  gap: theme.spacing(2),
  marginBottom: theme.spacing(3),
}));

const EmptyState = styled(Typography)(({ theme }) => ({
  fontStyle: 'italic',
  color: theme.palette.text.secondary,
}));

const formatDate = dateString => {
  if (!dateString) return 'N/A';

  const date = new Date(dateString);
  if (isNaN(date.getTime())) return dateString;

  return new window.Intl.DateTimeFormat('en-US', {
    year: 'numeric',
    month: 'long',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit',
  }).format(date);
};

const HardwareDetails = () => {
  const [hardware, setHardware] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const { hardware_id } = useParams();
  const { twoNodeInfo } = useCluster();

  useEffect(() => {
    const fetchHardwareDetails = async () => {
      try {
        const uri = `${BASE_URL}/cluster/hardware/${hardware_id}`;
        const response = await axios.get(uri);
        if (response.status === 200) {
          setHardware(response.data);
        } else {
          throw new Error(`Failed to fetch hardware details from ${uri}`);
        }
      } catch (err) {
        setError(err);
      } finally {
        setLoading(false);
      }
    };

    fetchHardwareDetails();
  }, [hardware_id]);

  const renderGPUTable = gpus => {
    if (!gpus || gpus.length === 0) return <EmptyState>No GPUs available</EmptyState>;

    const totalVRAM = gpus.reduce((sum, gpu) => sum + parseFloat(gpu.vram || 0), 0);

    return (
      <>
        {gpus.length > 1 && (
          <Box sx={{ mb: 2 }}>
            <Chip
              icon={<MemoryIcon />}
              label={`Total VRAM: ${totalVRAM.toFixed(2)} GB`}
              color="primary"
              variant="outlined"
            />
          </Box>
        )}
        <StyledTableContainer>
          <StyledTable size="small">
            <TableHead>
              <TableRow>
                <TableHeaderCell>Full Name</TableHeaderCell>
                <TableHeaderCell>Name</TableHeaderCell>
                <TableHeaderCell>Type</TableHeaderCell>
                <TableHeaderCell>VRAM (GB)</TableHeaderCell>
                <TableHeaderCell>Compute Capability</TableHeaderCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {gpus.map((gpu, index) => (
                <TableRow key={index} hover>
                  <TableBodyCell>{gpu.fullname || 'N/A'}</TableBodyCell>
                  <TableBodyCell>{gpu.name || 'N/A'}</TableBodyCell>
                  <TableBodyCell>
                    <Chip
                      label={getGPUTypeLabel(gpu.type)}
                      size="small"
                      color={getGPUTypeColor(gpu.type)}
                      variant="outlined"
                    />
                  </TableBodyCell>
                  <TableBodyCell>{gpu.vram || 'N/A'}</TableBodyCell>
                  <TableBodyCell>
                    {gpu.compute_capability ? `${gpu.compute_capability}` : 'N/A'}
                  </TableBodyCell>
                </TableRow>
              ))}
            </TableBody>
          </StyledTable>
        </StyledTableContainer>
      </>
    );
  };

  const getGPUTypeLabel = (type) => {
    const typeLabels = {
      'cuda': 'CUDA',
      'cuda_rtx': 'NVIDIA RTX',
      'cuda_professional': 'NVIDIA Pro',
      'cuda_consumer': 'NVIDIA GeForce',
      'rocm': 'AMD ROCm',
      'habana': 'Intel Habana Gaudi',
      'opencl': 'OpenCL',
      'opencl_intel': 'Intel OpenCL',
      'opencl_amd': 'AMD OpenCL',
      'opencl_nvidia': 'NVIDIA OpenCL',
      'mps': 'Apple Metal'
    };
    return typeLabels[type] || type || 'Unknown';
  };

  const getGPUTypeColor = (type) => {
    const typeColors = {
      'cuda': 'primary',
      'cuda_rtx': 'success',
      'cuda_professional': 'info',
      'cuda_consumer': 'secondary',
      'rocm': 'warning',
      'habana': 'error',
      'opencl': 'default',
      'opencl_intel': 'info',
      'opencl_amd': 'warning',
      'opencl_nvidia': 'primary',
      'mps': 'default'
    };
    return typeColors[type] || 'default';
  };

  return (
    <ContentContainer>
      <ButtonContainer>
        <BackButton component={Link} to="/cluster/home" startIcon={<ArrowBackIcon />} size="small">
          Back to Cluster Home
        </BackButton>

        <BackButton
          component={Link}
          to="/cluster/hardware"
          startIcon={<ArrowBackIcon />}
          size="small"
        >
          Back to Hardware List
        </BackButton>
      </ButtonContainer>

      <SectionTitle variant="h4">Hardware Details</SectionTitle>

      {twoNodeInfo && (
        <DetailCard>
          <DetailSubtitle variant="h6">
            <DeviceHubIcon /> Paired GPU Node
          </DetailSubtitle>
          <Divider sx={{ mb: 2, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
          <InfoSection>
            <DetailItem disableGutters>
              <DetailLabel>Host</DetailLabel>
              <DetailValue>{twoNodeInfo.remote_host || 'Unknown'}</DetailValue>
            </DetailItem>
            {twoNodeInfo.remote_user && (
              <DetailItem disableGutters>
                <DetailLabel>SSH User</DetailLabel>
                <DetailValue>{twoNodeInfo.remote_user}</DetailValue>
              </DetailItem>
            )}
            <DetailItem disableGutters>
              <DetailLabel>Mode</DetailLabel>
              <DetailValue>Remote container launches enabled</DetailValue>
            </DetailItem>
          </InfoSection>
        </DetailCard>
      )}

      {loading && <Spinner />}
      {error && <ErrorComponent message={error?.message || 'An error occurred'} />}

      {!loading && !error && !hardware && <EmptyState>Hardware not found</EmptyState>}

      {!loading && !error && hardware && (
        <Box data-tour="hardware-details-card">
          {/* General Information Card */}
          <DetailCard>
            <DetailSubtitle variant="h5">
              <MemoryIcon />
              {hardware.name || `Hardware ${hardware.id}`}
              <StatusChip
                icon={hardware.active ? <CheckCircleIcon /> : <CancelIcon />}
                label={hardware.active ? 'Active' : 'Inactive'}
                active={hardware.active}
              />
            </DetailSubtitle>

            <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />

            <Grid container spacing={3}>
              <Grid item xs={12} md={6}>
                <InfoSection>
                  <SectionLabel>
                    <DeviceHubIcon />
                    Node Information
                  </SectionLabel>
                  <StyledList>
                    <DetailItem>
                      <DetailLabel>Hardware ID:</DetailLabel>
                      <DetailValue>{hardware.id}</DetailValue>
                    </DetailItem>

                    <DetailItem>
                      <DetailLabel>Node ID:</DetailLabel>
                      <DetailValue>{hardware.node_id || 'No Node (Historical)'}</DetailValue>
                    </DetailItem>

                    <DetailItem>
                      <DetailLabel>Local Node ID:</DetailLabel>
                      <DetailValue>{hardware.local_node_id || 'N/A'}</DetailValue>
                    </DetailItem>

                    <DetailItem>
                      <DetailLabel>Cluster IP:</DetailLabel>
                      <DetailValue>{hardware.cluster_ip || 'N/A'}</DetailValue>
                    </DetailItem>

                    <DetailItem>
                      <DetailLabel>Ray Node ID:</DetailLabel>
                      <DetailValue>{hardware.ray_node_id || 'N/A'}</DetailValue>
                    </DetailItem>

                    <DetailItem>
                      <DetailLabel>Created At:</DetailLabel>
                      <DetailValue>{formatDate(hardware.created_at)}</DetailValue>
                    </DetailItem>
                  </StyledList>
                </InfoSection>
              </Grid>

              <Grid item xs={12} md={6}>
                <InfoSection>
                  <SectionLabel>
                    <DesktopWindowsIcon />
                    System Information
                  </SectionLabel>
                  <StyledList>
                    <DetailItem>
                      <DetailLabel>Processor Vendor:</DetailLabel>
                      <DetailValue>{hardware.processor_vendor || 'N/A'}</DetailValue>
                    </DetailItem>

                    <DetailItem>
                      <DetailLabel>Processors:</DetailLabel>
                      <DetailValue>
                        {hardware.processors && hardware.processors.length > 0
                          ? hardware.processors.join(', ')
                          : 'N/A'}
                      </DetailValue>
                    </DetailItem>

                    <DetailItem>
                      <DetailLabel>OS:</DetailLabel>
                      <DetailValue>{hardware.os || 'N/A'}</DetailValue>
                    </DetailItem>

                    <DetailItem>
                      <DetailLabel>Platform:</DetailLabel>
                      <DetailValue>{hardware.platform || 'N/A'}</DetailValue>
                    </DetailItem>
                  </StyledList>
                </InfoSection>
              </Grid>
            </Grid>
          </DetailCard>

          {/* GPUs Card */}
          {hardware.gpus && hardware.gpus.length > 0 && (
            <DetailCard>
              <DetailSubtitle variant="h5">
                <MemoryIcon />
                GPU Information
              </DetailSubtitle>

              <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />

              {renderGPUTable(hardware.gpus)}
            </DetailCard>
          )}

          {/* Configuration Card */}
          {hardware.configuration && Object.keys(hardware.configuration).length > 0 && (
            <DetailCard>
              <DetailSubtitle variant="h5">
                <SettingsIcon />
                Configuration
              </DetailSubtitle>

              <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />

              <StyledList>
                {Object.entries(hardware.configuration)
                  .filter(([key]) => key !== 'gpus') // Skip GPUs as we show them separately
                  .map(([key, value]) => (
                    <DetailItem key={key}>
                      <DetailLabel>{key}:</DetailLabel>
                      <DetailValue>
                        {typeof value === 'object'
                          ? JSON.stringify(value, null, 2)
                          : value?.toString() || 'N/A'}
                      </DetailValue>
                    </DetailItem>
                  ))}
              </StyledList>

              {hardware.configuration.gpus && hardware.configuration.gpus.length > 0 && (
                <Box sx={{ mt: 3 }}>
                  <SectionLabel>Configuration GPUs</SectionLabel>
                  {renderGPUTable(hardware.configuration.gpus)}
                </Box>
              )}
            </DetailCard>
          )}
        </Box>
      )}
    </ContentContainer>
  );
};

export default HardwareDetails;
