import React, { useState, useCallback, useMemo } from 'react';
import axios from 'axios';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  TextField,
  Button,
  Alert,
  Stepper,
  Step,
  StepLabel,
  IconButton,
  Chip,
  Grid,
  FormControlLabel,
  Checkbox,
  Collapse,
  Tooltip,
  CircularProgress,
} from '@mui/material';
import AddIcon from '@mui/icons-material/Add';
import DeleteIcon from '@mui/icons-material/Delete';
import ExpandMoreIcon from '@mui/icons-material/ExpandMore';
import ExpandLessIcon from '@mui/icons-material/ExpandLess';
import HelpOutlineIcon from '@mui/icons-material/HelpOutline';
import VpnKeyIcon from '@mui/icons-material/VpnKey';
import FederationsList from './FederationsList';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 0),
}));

const FormCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  padding: theme.spacing(3),
  marginBottom: theme.spacing(3),
}));

const FormSection = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(3),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.2rem',
  fontWeight: 600,
  marginBottom: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(1),
}));

const IPListContainer = styled(Box)(({ theme }) => ({
  marginTop: theme.spacing(2),
  padding: theme.spacing(2),
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  borderRadius: 8,
  border: '1px solid rgba(255, 255, 255, 0.08)',
}));

const IPRow = styled(Box)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(2),
  marginBottom: theme.spacing(1),
  '&:last-child': {
    marginBottom: 0,
  },
}));

const AdvancedSection = styled(Box)(({ theme }) => ({
  marginTop: theme.spacing(2),
  padding: theme.spacing(2),
  backgroundColor: 'rgba(33, 150, 243, 0.05)',
  borderRadius: 8,
  border: '1px solid rgba(33, 150, 243, 0.2)',
}));

const FederationsTab = () => {
  const [showForm, setShowForm] = useState(false);
  const [activeStep, setActiveStep] = useState(0);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [success, setSuccess] = useState(null);
  const [showAdvanced, setShowAdvanced] = useState(false);
  const [federationsKey, setFederationsKey] = useState(0); // For refreshing list
  const [pskGenerated, setPskGenerated] = useState(false);
  
  const [formData, setFormData] = useState({
    remote_cluster_name: '',
    remote_ips: [{ ip: '', primary: true }],
    preshared_key: '',
    callback_hostname: '',
  });

  const steps = ['Configure Remote', 'Set Security', 'Review & Pair'];

  // Validate IP address or hostname
  const validateIpOrHostname = useCallback((value) => {
    if (!value) return false;
    
    // Check if it's a valid IP address
    const ipRegex = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/;
    if (ipRegex.test(value)) return true;
    
    // Check if it's a valid hostname
    const hostnameRegex = /^[a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(\.[a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$/;
    return hostnameRegex.test(value) && value.length <= 255;
  }, []);

  const handleInputChange = useCallback((field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.value,
    }));
  }, []);

  const handleIpChange = useCallback((index, value) => {
    setFormData(prev => {
      const newIps = [...prev.remote_ips];
      newIps[index].ip = value;
      return { ...prev, remote_ips: newIps };
    });
  }, []);

  const handlePrimaryChange = useCallback((index) => {
    setFormData(prev => {
      const newIps = prev.remote_ips.map((ip, i) => ({
        ...ip,
        primary: i === index,
      }));
      return { ...prev, remote_ips: newIps };
    });
  }, []);

  const addIp = useCallback(() => {
    setFormData(prev => ({
      ...prev,
      remote_ips: [...prev.remote_ips, { ip: '', primary: false }],
    }));
  }, []);

  const removeIp = useCallback((index) => {
    setFormData(prev => {
      const newIps = prev.remote_ips.filter((_, i) => i !== index);
      // Ensure at least one is primary
      if (newIps.length > 0 && !newIps.some(ip => ip.primary)) {
        newIps[0].primary = true;
      }
      return { ...prev, remote_ips: newIps };
    });
  }, []);

  const generatePSK = useCallback(() => {
    // Server will generate a secure PSK automatically
    // Clear any existing PSK to let server generate
    setFormData(prev => ({ ...prev, preshared_key: '' }));
    setPskGenerated(true);
  }, []);

  const handleNext = () => {
    setActiveStep((prev) => prev + 1);
  };

  const handleBack = () => {
    setActiveStep((prev) => prev - 1);
  };

  const handleSubmit = async () => {
    setLoading(true);
    setError(null);
    setSuccess(null);

    try {
      // Filter out empty IPs
      const validIps = formData.remote_ips.filter(ip => ip.ip.trim() !== '');
      
      if (validIps.length === 0) {
        throw new Error('At least one remote IP is required');
      }

      const payload = {
        remote_cluster_name: formData.remote_cluster_name,
        remote_ips: validIps,
        preshared_key: formData.preshared_key || undefined,
        callback_hostname: formData.callback_hostname || undefined,
      };

      // Create the federation
      const createResponse = await axios.post(`${BASE_URL}/cluster/federations`, payload);
      const federationId = createResponse.data.id;

      // Initiate pairing
      await axios.post(`${BASE_URL}/cluster/federations/${federationId}/pair`);

      setSuccess('Federation created and pairing initiated successfully!');
      setShowForm(false);
      setFormData({
        remote_cluster_name: '',
        remote_ips: [{ ip: '', primary: true }],
        preshared_key: '',
        callback_hostname: '',
      });
      setActiveStep(0);
      setFederationsKey(prev => prev + 1); // Refresh the list
    } catch (err) {
      console.error('Error creating federation:', err);
      setError(err.response?.data?.detail || err.message || 'Failed to create federation');
    } finally {
      setLoading(false);
    }
  };

  const isStepValid = () => {
    switch (activeStep) {
      case 0:
        return (
          formData.remote_cluster_name.trim() !== '' &&
          formData.remote_ips.some(ip => ip.ip.trim() !== '' && validateIpOrHostname(ip.ip)) &&
          formData.remote_ips.every(ip => !ip.ip || validateIpOrHostname(ip.ip)) &&
          (!formData.callback_hostname || validateIpOrHostname(formData.callback_hostname))
        );
      case 1:
        return true; // PSK is optional
      case 2:
        return true;
      default:
        return false;
    }
  };

  return (
    <ContentContainer>
      {error && (
        <Alert severity="error" onClose={() => setError(null)} sx={{ mb: 2 }}>
          {error}
        </Alert>
      )}
      {success && (
        <Alert severity="success" onClose={() => setSuccess(null)} sx={{ mb: 2 }}>
          {success}
        </Alert>
      )}

      {showForm ? (
        <FormCard>
          <Typography variant="h5" gutterBottom>
            Add Federated Cluster
          </Typography>
          
          <Stepper activeStep={activeStep} sx={{ mt: 3, mb: 4 }}>
            {steps.map((label) => (
              <Step key={label}>
                <StepLabel>{label}</StepLabel>
              </Step>
            ))}
          </Stepper>

          {activeStep === 0 && (
            <FormSection>
              <SectionTitle>Remote Cluster Information</SectionTitle>
              <TextField
                fullWidth
                label="Remote Cluster Name"
                value={formData.remote_cluster_name}
                onChange={handleInputChange('remote_cluster_name')}
                margin="normal"
                required
                helperText="Display name for the remote cluster"
              />
              
              <IPListContainer>
                <Typography variant="subtitle2" gutterBottom>
                  Remote IP Addresses
                </Typography>
                {formData.remote_ips.map((ip, index) => (
                  <IPRow key={index}>
                    <TextField
                      label={`IP Address ${index + 1}`}
                      value={ip.ip}
                      onChange={(e) => handleIpChange(index, e.target.value)}
                      size="small"
                      sx={{ flex: 1 }}
                      placeholder="e.g., 10.0.0.1 or cluster.example.com"
                      error={ip.ip && !validateIpOrHostname(ip.ip)}
                      helperText={ip.ip && !validateIpOrHostname(ip.ip) ? "Invalid IP address or hostname" : ""}
                    />
                    <FormControlLabel
                      control={
                        <Checkbox
                          checked={ip.primary}
                          onChange={() => handlePrimaryChange(index)}
                        />
                      }
                      label="Primary"
                    />
                    {formData.remote_ips.length > 1 && (
                      <IconButton
                        size="small"
                        onClick={() => removeIp(index)}
                        color="error"
                      >
                        <DeleteIcon />
                      </IconButton>
                    )}
                  </IPRow>
                ))}
                <Button
                  startIcon={<AddIcon />}
                  onClick={addIp}
                  size="small"
                  sx={{ mt: 1 }}
                >
                  Add IP
                </Button>
              </IPListContainer>
            </FormSection>
          )}

          {activeStep === 1 && (
            <FormSection>
              <SectionTitle>
                Security Configuration
                <Tooltip title="Pre-shared key for secure communication">
                  <HelpOutlineIcon fontSize="small" />
                </Tooltip>
              </SectionTitle>
              
              <Box sx={{ display: 'flex', gap: 2, alignItems: 'flex-start' }}>
                <TextField
                  fullWidth
                  label="Pre-Shared Key (PSK)"
                  value={pskGenerated ? '(Auto-generated on server)' : formData.preshared_key}
                  onChange={handleInputChange('preshared_key')}
                  margin="normal"
                  type="password"
                  disabled={pskGenerated}
                  helperText={pskGenerated ? "A secure key will be generated on the server" : "Leave empty to auto-generate a secure key"}
                />
                <Button
                  startIcon={<VpnKeyIcon />}
                  onClick={generatePSK}
                  variant="outlined"
                  sx={{ mt: 2 }}
                  disabled={pskGenerated}
                >
                  {pskGenerated ? 'Key will be auto-generated' : 'Use Auto-Generated Key'}
                </Button>
              </Box>

              <Button
                onClick={() => setShowAdvanced(!showAdvanced)}
                endIcon={showAdvanced ? <ExpandLessIcon /> : <ExpandMoreIcon />}
                sx={{ mt: 2 }}
              >
                Advanced Options
              </Button>
              
              <Collapse in={showAdvanced}>
                <AdvancedSection>
                  <TextField
                    fullWidth
                    label="Callback Hostname"
                    value={formData.callback_hostname}
                    onChange={handleInputChange('callback_hostname')}
                    margin="normal"
                    error={formData.callback_hostname && !validateIpOrHostname(formData.callback_hostname)}
                    helperText={
                      formData.callback_hostname && !validateIpOrHostname(formData.callback_hostname)
                        ? "Invalid IP address or hostname"
                        : "Optional: Specify the hostname/IP the remote cluster should use to reach this cluster"
                    }
                    placeholder="e.g., my-cluster.example.com or 203.0.113.1"
                  />
                </AdvancedSection>
              </Collapse>
            </FormSection>
          )}

          {activeStep === 2 && (
            <FormSection>
              <SectionTitle>Review Configuration</SectionTitle>
              <Box sx={{ p: 2, backgroundColor: 'rgba(0, 0, 0, 0.2)', borderRadius: 2 }}>
                <Grid container spacing={2}>
                  <Grid item xs={12}>
                    <Typography variant="subtitle2" color="text.secondary">
                      Remote Cluster Name
                    </Typography>
                    <Typography variant="body1">
                      {formData.remote_cluster_name}
                    </Typography>
                  </Grid>
                  <Grid item xs={12}>
                    <Typography variant="subtitle2" color="text.secondary">
                      Remote IPs
                    </Typography>
                    {formData.remote_ips.filter(ip => ip.ip).map((ip, index) => (
                      <Box key={index}>
                        <Typography variant="body1">
                          {ip.ip} {ip.primary && <Chip label="Primary" size="small" />}
                        </Typography>
                      </Box>
                    ))}
                  </Grid>
                  <Grid item xs={12}>
                    <Typography variant="subtitle2" color="text.secondary">
                      Security
                    </Typography>
                    <Typography variant="body1">
                      PSK: {formData.preshared_key ? '••••••••' : 'Will be auto-generated'}
                    </Typography>
                  </Grid>
                  {formData.callback_hostname && (
                    <Grid item xs={12}>
                      <Typography variant="subtitle2" color="text.secondary">
                        Callback Hostname
                      </Typography>
                      <Typography variant="body1">
                        {formData.callback_hostname}
                      </Typography>
                    </Grid>
                  )}
                </Grid>
              </Box>
            </FormSection>
          )}

          <Box sx={{ display: 'flex', justifyContent: 'space-between', mt: 3 }}>
            <Button
              onClick={() => setShowForm(false)}
              variant="outlined"
            >
              Cancel
            </Button>
            <Box sx={{ display: 'flex', gap: 2 }}>
              {activeStep > 0 && (
                <Button onClick={handleBack}>
                  Back
                </Button>
              )}
              {activeStep < steps.length - 1 ? (
                <Button
                  variant="contained"
                  onClick={handleNext}
                  disabled={!isStepValid()}
                >
                  Next
                </Button>
              ) : (
                <Button
                  variant="contained"
                  onClick={handleSubmit}
                  disabled={loading || !isStepValid()}
                  startIcon={loading && <CircularProgress size={20} />}
                >
                  {loading ? 'Pairing...' : 'Create & Pair'}
                </Button>
              )}
            </Box>
          </Box>
        </FormCard>
      ) : (
        <FederationsList 
          key={federationsKey}
          onAddNew={() => setShowForm(true)} 
        />
      )}
    </ContentContainer>
  );
};

export default FederationsTab;