import React, { useState, useEffect, useCallback, useMemo } from 'react';
import axios from 'axios';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  List,
  ListItem,
  ListItemText,
  ListItemIcon,
  Chip,
  IconButton,
  Tooltip,
  CircularProgress,
  Alert,
  Button,
} from '@mui/material';
import HubIcon from '@mui/icons-material/Hub';
import RefreshIcon from '@mui/icons-material/Refresh';
import LinkOffIcon from '@mui/icons-material/LinkOff';
import CheckCircleIcon from '@mui/icons-material/CheckCircle';
import ErrorIcon from '@mui/icons-material/Error';
import PendingIcon from '@mui/icons-material/Pending';
import WifiTetheringIcon from '@mui/icons-material/WifiTethering';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 0),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.5rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  display: 'flex',
  alignItems: 'center',
  gap: theme.spacing(2),
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const FederationCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
  marginBottom: theme.spacing(2),
}));

const FederationListContainer = styled(List)(({ theme }) => ({
  padding: 0,
}));

const FederationItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(2, 3),
  borderBottom: '1px solid rgba(255, 255, 255, 0.05)',
  '&:last-child': {
    borderBottom: 'none',
  },
}));

const StatusChip = styled(Chip)(({ theme, status }) => ({
  fontWeight: 600,
  fontSize: '0.75rem',
  ...(status === 'PAIRED' && {
    backgroundColor: 'rgba(76, 175, 80, 0.15)',
    color: '#4caf50',
    border: '1px solid rgba(76, 175, 80, 0.3)',
  }),
  ...(status === 'PAIRING' && {
    backgroundColor: 'rgba(255, 193, 7, 0.15)',
    color: '#ffc107',
    border: '1px solid rgba(255, 193, 7, 0.3)',
  }),
  ...(status === 'DISCONNECTED' && {
    backgroundColor: 'rgba(244, 67, 54, 0.15)',
    color: '#f44336',
    border: '1px solid rgba(244, 67, 54, 0.3)',
  }),
  ...(status === 'FAILED' && {
    backgroundColor: 'rgba(244, 67, 54, 0.15)',
    color: '#f44336',
    border: '1px solid rgba(244, 67, 54, 0.3)',
  }),
}));

const EmptyState = styled(Box)(({ theme }) => ({
  textAlign: 'center',
  padding: theme.spacing(6, 2),
  color: theme.palette.text.secondary,
}));

const ActionButtons = styled(Box)(({ theme }) => ({
  display: 'flex',
  gap: theme.spacing(1),
  marginLeft: 'auto',
}));

const FederationsList = ({ isEmbedded = false, onAddNew }) => {
  const [federations, setFederations] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [actionLoading, setActionLoading] = useState({});

  const fetchFederations = useCallback(async () => {
    try {
      const response = await axios.get(`${BASE_URL}/cluster/federations`);
      if (response.status === 200) {
        setFederations(response.data);
      }
    } catch (err) {
      console.error('Error fetching federations:', err);
      setError(err);
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchFederations();
    // Poll every 30 seconds for updates
    const interval = setInterval(fetchFederations, 30000);
    return () => clearInterval(interval);
  }, []);

  const handlePing = useCallback(async (federationId) => {
    setActionLoading(prev => ({ ...prev, [federationId]: 'ping' }));
    try {
      const response = await axios.post(`${BASE_URL}/cluster/federations/${federationId}/ping`);
      if (response.data.reachable) {
        // Refresh the list to show updated ping time
        await fetchFederations();
      }
    } catch (err) {
      console.error('Error pinging federation:', err);
    } finally {
      setActionLoading(prev => ({ ...prev, [federationId]: null }));
    }
  }, [fetchFederations]);

  const handleDisconnect = useCallback(async (federationId, force = false) => {
    if (!window.confirm(`Are you sure you want to disconnect this federation?${force ? ' (forced)' : ''}`)) {
      return;
    }
    
    setActionLoading(prev => ({ ...prev, [federationId]: 'disconnect' }));
    try {
      await axios.post(`${BASE_URL}/cluster/federations/${federationId}/disconnect?force=${force}`);
      await fetchFederations();
    } catch (err) {
      console.error('Error disconnecting federation:', err);
      if (!force) {
        // If graceful disconnect failed, offer force disconnect
        if (window.confirm('Graceful disconnect failed. Force disconnect?')) {
          await handleDisconnect(federationId, true);
        }
      }
    } finally {
      setActionLoading(prev => ({ ...prev, [federationId]: null }));
    }
  }, [fetchFederations]);

  const getStatusIcon = (status) => {
    switch (status) {
      case 'PAIRED':
        return <CheckCircleIcon sx={{ color: '#4caf50' }} />;
      case 'PAIRING':
        return <PendingIcon sx={{ color: '#ffc107' }} />;
      case 'DISCONNECTED':
      case 'FAILED':
        return <ErrorIcon sx={{ color: '#f44336' }} />;
      default:
        return <HubIcon />;
    }
  };

  const formatDateTime = (dateString) => {
    if (!dateString) return 'Never';
    return new Date(dateString).toLocaleString();
  };

  const getPrimaryIp = (ips) => {
    if (!ips || !ips.length) return 'Unknown';
    const primary = ips.find(ip => ip.primary);
    return primary ? primary.ip : ips[0].ip;
  };

  if (loading) return <Spinner />;
  if (error) return <ErrorComponent message={error.message} />;

  return (
    <ContentContainer>
      <SectionTitle variant="h5">
        <HubIcon />
        Federated Clusters
        {onAddNew && (
          <Button
            variant="contained"
            color="primary"
            size="small"
            onClick={onAddNew}
            sx={{ marginLeft: 'auto' }}
          >
            Add Federation
          </Button>
        )}
      </SectionTitle>

      {federations.length === 0 ? (
        <EmptyState>
          <HubIcon sx={{ fontSize: 48, mb: 2, opacity: 0.5 }} />
          <Typography variant="h6" gutterBottom>
            No Federations Yet
          </Typography>
          <Typography variant="body2">
            {isEmbedded 
              ? 'Switch to the Federations tab to add your first federated cluster.'
              : 'Add a federated cluster to enable cross-cluster communication.'}
          </Typography>
        </EmptyState>
      ) : (
        <FederationCard>
          <FederationListContainer>
            {federations.map((federation) => (
              <FederationItem key={federation.id}>
                <ListItemIcon>
                  {getStatusIcon(federation.status)}
                </ListItemIcon>
                <ListItemText
                  primary={
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                      <Typography variant="subtitle1" fontWeight={600}>
                        {federation.remote_cluster_name}
                      </Typography>
                      <StatusChip 
                        label={federation.status} 
                        size="small" 
                        status={federation.status}
                      />
                    </Box>
                  }
                  secondary={
                    <Box sx={{ mt: 1 }}>
                      <Typography variant="caption" display="block">
                        Remote: {getPrimaryIp(federation.remote_ips)}
                        {federation.callback_hostname && ` • Callback: ${federation.callback_hostname}`}
                      </Typography>
                      <Typography variant="caption" display="block">
                        Last Ping: {formatDateTime(federation.last_ping)} • 
                        Last Activity: {formatDateTime(federation.last_activity)}
                      </Typography>
                      {federation.created_by && (
                        <Typography variant="caption" display="block">
                          Created by: {federation.created_by} on {formatDateTime(federation.created_at)}
                        </Typography>
                      )}
                    </Box>
                  }
                />
                <ActionButtons>
                  {federation.status === 'PAIRED' && (
                    <>
                      <Tooltip title="Ping Remote Cluster">
                        <IconButton
                          size="small"
                          onClick={() => handlePing(federation.id)}
                          disabled={actionLoading[federation.id] === 'ping'}
                        >
                          {actionLoading[federation.id] === 'ping' ? (
                            <CircularProgress size={20} />
                          ) : (
                            <WifiTetheringIcon />
                          )}
                        </IconButton>
                      </Tooltip>
                      <Tooltip title="Disconnect Federation">
                        <IconButton
                          size="small"
                          onClick={() => handleDisconnect(federation.id)}
                          disabled={actionLoading[federation.id] === 'disconnect'}
                        >
                          {actionLoading[federation.id] === 'disconnect' ? (
                            <CircularProgress size={20} />
                          ) : (
                            <LinkOffIcon />
                          )}
                        </IconButton>
                      </Tooltip>
                    </>
                  )}
                  {federation.status === 'PAIRING' && (
                    <Tooltip title="Pairing in progress...">
                      <CircularProgress size={20} />
                    </Tooltip>
                  )}
                </ActionButtons>
              </FederationItem>
            ))}
          </FederationListContainer>
        </FederationCard>
      )}
    </ContentContainer>
  );
};

export default FederationsList;