import React, { useState, useEffect } from 'react';
import { Link, useLocation } from 'react-router-dom';
import axios from 'axios';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  List,
  ListItem,
  ListItemText,
  ListItemIcon,
  Button,
  Divider,
  Tooltip,
  Chip,
} from '@mui/material';
import StorageIcon from '@mui/icons-material/Storage';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import { BASE_URL } from '../../const';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 0),
}));

const SectionTitle = styled(Typography, {
  shouldForwardProp: prop => prop !== 'embedded',
})(({ theme, embedded }) => ({
  color: theme.palette.primary.main,
  fontSize: embedded ? '1.5rem' : '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  },
}));

const ClusterCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'hidden',
  marginBottom: theme.spacing(4),
}));

const ClusterListContainer = styled(List)(({ theme }) => ({
  padding: 0,
}));

const ClusterItem = styled(ListItem)(({ theme }) => ({
  padding: theme.spacing(2, 3),
  borderBottom: '1px solid rgba(255, 255, 255, 0.05)',
  transition: 'background-color 0.2s',
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.03)',
  },
  '&:last-child': {
    borderBottom: 'none',
  },
}));

const ClusterLink = styled(Link)(({ theme }) => ({
  color: theme.palette.primary.main,
  textDecoration: 'none',
  fontWeight: 500,
  fontSize: '1rem',
  transition: 'all 0.2s',
  display: 'flex',
  alignItems: 'center',
  '&:hover': {
    color: theme.palette.primary.light,
    textDecoration: 'underline',
  },
}));

const ClusterIcon = styled(StorageIcon)(({ theme }) => ({
  color: theme.palette.primary.main,
  marginRight: theme.spacing(1),
  fontSize: '1.2rem',
}));

const EmptyState = styled(Box)(({ theme }) => ({
  padding: theme.spacing(4),
  textAlign: 'center',
  color: theme.palette.text.secondary,
  fontStyle: 'italic',
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const ClusterList = ({ isEmbedded = false }) => {
  const [clusters, setClusters] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const location = useLocation();

  useEffect(() => {
    const fetchClusters = async () => {
      setLoading(true);
      try {
        const uri = `${BASE_URL}/cluster/clusters`;
        const response = await axios.get(uri);
        if (response.status === 200) {
          setClusters(response.data);
        } else {
          throw new Error(`Failed to fetch clusters from ${uri}`);
        }
      } catch (err) {
        setError(err);
      } finally {
        setLoading(false);
      }
    };

    fetchClusters();
  }, []);

  return (
    <ContentContainer>
      {!isEmbedded && location.pathname !== '/cluster/home' && (
        <BackButton component={Link} to="/cluster/home" startIcon={<ArrowBackIcon />} size="small">
          Back to Clusters Home
        </BackButton>
      )}

      <SectionTitle variant="h4" embedded={isEmbedded}>
        Cluster List
        <Tooltip title="List of compute clusters available in your Kamiwaza instance">
          <InfoIcon />
        </Tooltip>
      </SectionTitle>

      {loading && <Spinner />}
      {error && <ErrorComponent message={error.message} />}

      {!loading && !error && (
        <ClusterCard>
          {clusters.length === 0 ? (
            <EmptyState>
              <Typography variant="body1">No clusters available.</Typography>
            </EmptyState>
          ) : (
            <ClusterListContainer>
              {clusters.map(cluster => (
                <ClusterItem key={cluster.id}>
                  <ListItemIcon style={{ minWidth: 'auto', marginRight: '8px' }}>
                    <ClusterIcon />
                  </ListItemIcon>
                  <ListItemText
                    primary={
                      <ClusterLink to={`/cluster/${cluster.id}`}>{cluster.name}</ClusterLink>
                    }
                  />
                </ClusterItem>
              ))}
            </ClusterListContainer>
          )}
        </ClusterCard>
      )}
    </ContentContainer>
  );
};

export default ClusterList;
