import React, { useEffect } from 'react';
import { styled } from '@mui/material/styles';
import { Typography, Box, Tab, Tabs, Paper, IconButton, Tooltip } from '@mui/material';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import StorageIcon from '@mui/icons-material/Storage';
import LocationOnIcon from '@mui/icons-material/LocationOn';
import DeviceHubIcon from '@mui/icons-material/DeviceHub';
import ComputerIcon from '@mui/icons-material/Computer';
import MemoryIcon from '@mui/icons-material/Memory';
import HubIcon from '@mui/icons-material/Hub';
import RefreshIcon from '@mui/icons-material/Refresh';
import Spinner from '../common/Spinner';
import ClusterList from './ClusterList';
import LocationList from './LocationList';
import NodeList from './NodeList';
import HardwareList from './HardwareList';
import ErrorComponent from '../common/ErrorComponent';
import RunningNodeList from './RunningNodeList';
import FederationsList from './FederationsList';
import FederationsTab from './FederationsTab';
import { useTour } from '../../contexts/TourContext';
import { useCluster } from '../../contexts/ClusterContext';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(1),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  display: 'flex',
  alignItems: 'center',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const TabsContainer = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: '12px 12px 0 0',
  boxShadow: 'none',
  borderBottom: '1px solid rgba(255, 255, 255, 0.08)',
  marginBottom: theme.spacing(3),
  marginTop: theme.spacing(3),
}));

const StyledTabs = styled(Tabs)(({ theme }) => ({
  '& .MuiTabs-indicator': {
    backgroundColor: theme.palette.primary.main,
    height: 3,
  },
}));

const StyledTab = styled(Tab)(({ theme }) => ({
  textTransform: 'none',
  fontSize: '0.9rem',
  fontWeight: 500,
  padding: theme.spacing(2, 3),
  color: theme.palette.text.secondary,
  '&.Mui-selected': {
    color: theme.palette.primary.main,
    fontWeight: 600,
  },
  '&:hover': {
    color: theme.palette.text.primary,
    opacity: 1,
  },
  '& .MuiSvgIcon-root': {
    marginRight: theme.spacing(1),
    fontSize: '1.2rem',
  },
}));

const TabPanel = styled(Box)(({ theme }) => ({
  padding: theme.spacing(3, 0),
}));

function TabPanelContent(props) {
  const { children, value, index, ...other } = props;

  return (
    <TabPanel
      role="tabpanel"
      hidden={value !== index}
      id={`cluster-tabpanel-${index}`}
      aria-labelledby={`cluster-tab-${index}`}
      {...other}
    >
      {value === index && children}
    </TabPanel>
  );
}

const ClusterHome = () => {
  const { startTour } = useTour();
  const { refetchCapabilities, clusterCapabilities } = useCluster();
  const [loading, setLoading] = React.useState(true);
  const [error, setError] = React.useState(null);
  const [tabValue, setTabValue] = React.useState(0);
  const [refreshing, setRefreshing] = React.useState(false);

  // Refresh hardware capabilities only when cluster home page first loads
  // Let individual components manage their own refresh logic to avoid excessive calls
  React.useEffect(() => {
    // Only refresh if we don't already have capabilities loaded
    const needsRefresh = !clusterCapabilities;
    if (needsRefresh) {
      refetchCapabilities();
    }
    setLoading(false);
  }, []); // Remove refetchCapabilities dependency to prevent excessive calls

  // Start cluster tour after a delay
  useEffect(() => {
    const timer = setTimeout(() => {
      startTour('cluster');
    }, 1500);

    return () => clearTimeout(timer);
  }, [startTour]);

  const handleTabChange = (_, newValue) => {
    setTabValue(newValue);
  };

  const handleManualRefresh = async () => {
    setRefreshing(true);
    try {
      await refetchCapabilities();
    } catch (err) {
      setError(err);
    } finally {
      setRefreshing(false);
    }
  };

  // Auto-advance tour when tabs are clicked
  React.useEffect(() => {
    // Check if we're in a tour
    const tourOverlay = document.querySelector('.__floater__body');
    if (!tourOverlay) return;

    // When user clicks Hardware tab during tour
    if (tabValue === 4) {
      // Give more time for the hardware list to render
      setTimeout(() => {
        // Wait for the hardware list to be in the DOM
        const checkForHardwareList = () => {
          const hardwareList = document.querySelector('[data-tour="hardware-list"]');
          const hardwareLink = hardwareList?.querySelector('li:first-child a');
          if (hardwareList && hardwareLink) {
            // Hardware list is ready with clickable link, advance tour
            window.dispatchEvent(new CustomEvent('tourTabClicked', { detail: 'hardware' }));
          } else {
            // Try again in 100ms
            setTimeout(checkForHardwareList, 100);
          }
        };
        checkForHardwareList();
      }, 500);
    }
  }, [tabValue]);

  return (
    <ContentContainer data-tour="cluster-overview">
      <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
        <SectionTitle variant="h4" style={{ marginBottom: 0 }}>Cluster Home</SectionTitle>
        <Tooltip title="Refresh hardware information">
          <IconButton 
            onClick={handleManualRefresh} 
            disabled={refreshing}
            color="primary"
          >
            <RefreshIcon />
          </IconButton>
        </Tooltip>
      </Box>

      {loading && <Spinner />}
      {error && <ErrorComponent message={error.message} />}

      <TabsContainer data-tour="cluster-tabs">
        <StyledTabs
          value={tabValue}
          onChange={handleTabChange}
          variant="scrollable"
          scrollButtons="auto"
          aria-label="cluster management tabs"
        >
          <StyledTab icon={<StorageIcon />} label="Clusters" iconPosition="start" />
          <StyledTab icon={<LocationOnIcon />} label="Locations" iconPosition="start" />
          <StyledTab icon={<DeviceHubIcon />} label="Nodes" iconPosition="start" />
          <StyledTab icon={<ComputerIcon />} label="Running Nodes" iconPosition="start" />
          <StyledTab
            icon={<MemoryIcon />}
            label="Hardware"
            iconPosition="start"
            data-tour="hardware-tab"
          />
          <StyledTab icon={<HubIcon />} label="Federations" iconPosition="start" />
        </StyledTabs>
      </TabsContainer>

      <TabPanelContent value={tabValue} index={0}>
        <ClusterList isEmbedded={true} />
      </TabPanelContent>

      <TabPanelContent value={tabValue} index={1}>
        <LocationList isEmbedded={true} />
      </TabPanelContent>

      <TabPanelContent value={tabValue} index={2}>
        <NodeList isEmbedded={true} />
      </TabPanelContent>

      <TabPanelContent value={tabValue} index={3}>
        <RunningNodeList isEmbedded={true} />
      </TabPanelContent>

      <TabPanelContent value={tabValue} index={4}>
        <HardwareList isEmbedded={true} />
      </TabPanelContent>

      <TabPanelContent value={tabValue} index={5}>
        <FederationsTab />
      </TabPanelContent>

      {/* Show Federations List in main view */}
      {tabValue === 0 && (
        <Box sx={{ mt: 4 }}>
          <FederationsList isEmbedded={true} />
        </Box>
      )}
    </ContentContainer>
  );
};

export default ClusterHome;
