import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { useParams, Link } from 'react-router-dom';
import { styled } from '@mui/material/styles';
import { 
  Typography, 
  Box, 
  Paper, 
  Button,
  Grid,
  Divider,
  Chip,
  Tooltip
} from '@mui/material';
import BadgeIcon from '@mui/icons-material/Badge';
import ArrowBackIcon from '@mui/icons-material/ArrowBack';
import StorageIcon from '@mui/icons-material/Storage';
import LocationOnIcon from '@mui/icons-material/LocationOn';
import CalendarTodayIcon from '@mui/icons-material/CalendarToday';
import { BASE_URL } from '../../const';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';

// Styled components
const ContentContainer = styled(Box)(({ theme }) => ({
  maxWidth: '100%',
  margin: '0 auto',
  padding: theme.spacing(3, 2),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.8rem',
  fontWeight: 700,
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1.5),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '80px',
    height: '3px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const BackButton = styled(Button)(({ theme }) => ({
  marginBottom: theme.spacing(3),
  textTransform: 'none',
  fontSize: '0.875rem',
  fontWeight: 500,
  color: theme.palette.primary.main,
  backgroundColor: 'rgba(0, 192, 127, 0.08)',
  border: '1px solid rgba(0, 192, 127, 0.2)',
  borderRadius: '8px',
  padding: theme.spacing(0.75, 2),
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.15)',
  }
}));

const DetailCard = styled(Paper)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  padding: theme.spacing(3),
  marginBottom: theme.spacing(4),
  marginTop: theme.spacing(2),
}));

const DetailSubtitle = styled(Typography)(({ theme }) => ({
  fontSize: '1.3rem',
  fontWeight: 600,
  color: theme.palette.text.primary,
  marginBottom: theme.spacing(2),
}));

const DetailItem = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  '& svg': {
    marginRight: theme.spacing(1.5),
    color: theme.palette.primary.main,
  }
}));

const DetailLabel = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.secondary,
  fontWeight: 500,
  width: '120px',
  flexShrink: 0,
}));

const DetailValue = styled(Typography)(({ theme }) => ({
  color: theme.palette.text.primary,
  fontWeight: 400,
  flexGrow: 1,
  overflow: 'hidden',
  textOverflow: 'ellipsis',
}));

const LocationChip = styled(Chip)(({ theme }) => ({
  backgroundColor: 'rgba(33, 150, 243, 0.1)',
  color: theme.palette.text.primary,
  fontWeight: 500,
  '& .MuiChip-icon': {
    color: theme.palette.secondary.main,
  },
}));

const formatDate = (dateString) => {
  if (!dateString) return 'N/A';
  
  const date = new Date(dateString);
  if (isNaN(date.getTime())) return dateString;
  
  return new Intl.DateTimeFormat('en-US', {
    year: 'numeric',
    month: 'long',
    day: 'numeric',
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit'
  }).format(date);
};

const ClusterDetails = () => {
    const [cluster, setCluster] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);

    let { cluster_id } = useParams();

    useEffect(() => {
        const fetchClusterDetails = async () => {
            try {
                const uri = `${BASE_URL}/cluster/cluster/${cluster_id}`;
                const response = await axios.get(uri);
                if (response.status === 200) {
                    setCluster(response.data);
                } else {
                    throw new Error(`Failed to fetch cluster details from ${uri}`);
                }
            } catch (err) {
                setError(err);
            } finally {
                setLoading(false);
            }
        };

        fetchClusterDetails();
    }, [cluster_id]);

    return (
        <ContentContainer>
            <BackButton 
                component={Link} 
                to="/cluster/home" 
                startIcon={<ArrowBackIcon />}
                size="small"
            >
                Back to Clusters Home
            </BackButton>
            
            <SectionTitle variant="h4">Cluster Details</SectionTitle>
            
            {loading && <Spinner />}
            {error && <ErrorComponent message={error?.message || 'An error occurred'} />}
            
            {!loading && !error && cluster && (
                <DetailCard>
                    <DetailSubtitle variant="h5">
                        <StorageIcon sx={{ mr: 1, verticalAlign: 'middle' }} />
                        {cluster.name || 'Unnamed Cluster'}
                    </DetailSubtitle>
                    
                    <Divider sx={{ mb: 3, borderColor: 'rgba(255, 255, 255, 0.08)' }} />
                    
                    <Grid container spacing={3}>
                        <Grid item xs={12} md={6}>
                            <DetailItem>
                                <BadgeIcon />
                                <DetailLabel>ID:</DetailLabel>
                                <DetailValue>{cluster.id || 'N/A'}</DetailValue>
                            </DetailItem>
                            
                            <DetailItem>
                                <LocationOnIcon />
                                <DetailLabel>Location:</DetailLabel>
                                <DetailValue>
                                    {cluster.location_id ? (
                                        <LocationChip 
                                            icon={<LocationOnIcon />}
                                            label={cluster.location_id}
                                            size="small"
                                            component={Link}
                                            to={`/cluster/locations/${cluster.location_id}`}
                                            clickable
                                        />
                                    ) : (
                                        'N/A'
                                    )}
                                </DetailValue>
                            </DetailItem>
                            
                            <DetailItem>
                                <CalendarTodayIcon />
                                <DetailLabel>Created At:</DetailLabel>
                                <DetailValue>{formatDate(cluster.created_at)}</DetailValue>
                            </DetailItem>
                        </Grid>
                    </Grid>
                </DetailCard>
            )}

            {!loading && !error && !cluster && (
                <Typography variant="body1" sx={{ color: 'text.secondary', fontStyle: 'italic', mt: 2 }}>
                    No cluster found with the specified identifier.
                </Typography>
            )}
        </ContentContainer>
    );
};

export default ClusterDetails;
