import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { BASE_URL } from '../../const';
import { StyledModal, StyledButton, StyledTable, StyledTableCell, StyledTextField, StyledCircularProgress, StyledModalContent, StyledSubHeader } from '../../StyledComponents';

const AddLocationModal = ({ closeModal, showModal, addLocation, locationDetails }) => {
    const [locationData, setLocationData] = useState({
        name: '',
        datacenter: '',
        region: '',
        zone: '',
        building: '',
        address: '',
        contact_phone: '',
        contact_name: '',
        contact_email: ''
    });

    useEffect(() => {
        if (locationDetails) {
            setLocationData(locationDetails);
        }
    }, [locationDetails]);

    const handleAddLocation = async () => {
        try {
            // Filter out empty strings before sending the data
            const filteredLocationData = Object.fromEntries(
                Object.entries(locationData).filter(([_, value]) => value !== '')
            );
    
            const response = await axios.post(`${BASE_URL}/cluster/location`, filteredLocationData);
            if (response.status === 200) {
                addLocation(response.data); // Call addLocation with the new location
                closeModal();
            } else {
                // Handle non-successful response
            }
        } catch (err) {
            console.error('Add location error:', err);
            // Handle add location error
        }
    };

    const handleUpdateLocation = async () => {
        try {
            // Filter out empty strings before sending the data
            const filteredLocationData = Object.fromEntries(
                Object.entries(locationData).filter(([_, value]) => value !== '')
            );

            const response = await axios.put(`${BASE_URL}/cluster/location/${locationDetails.id}`, filteredLocationData);
            if (response.status === 200) {
                addLocation(response.data); // Call addLocation with the updated location
                closeModal();
            } else {
                // Handle non-successful response
            }
        } catch (err) {
            console.error('Update location error:', err);
            // Handle update location error
        }
    };

    const handleInputChange = (event) => {
        setLocationData({
            ...locationData,
            [event.target.name]: event.target.value
        });
    };

    return (
        <StyledModal open={showModal} onClose={closeModal}>
            <StyledModalContent>
                <StyledTable>
                    <thead>
                        <tr>
                            <StyledTableCell>
                                <StyledSubHeader>{locationDetails ? 'Update Location' : 'Add New Location'}</StyledSubHeader>
                            </StyledTableCell>
                        </tr>
                    </thead>
                    <tbody>
                        {Object.keys(locationData).map((key) => (
                            <tr key={key}>
                                <StyledTableCell>{key.split('_').map(word => word.charAt(0).toUpperCase() + word.slice(1)).join(' ')}</StyledTableCell>
                                <StyledTableCell>
                                    <StyledTextField
                                        name={key}
                                        value={locationData[key]}
                                        onChange={handleInputChange}
                                    />
                                </StyledTableCell>
                            </tr>
                        ))}
                    </tbody>
                </StyledTable>
                <StyledButton variant="secondary" onClick={closeModal}>
                    Close
                </StyledButton>
                <StyledButton variant="primary" onClick={locationDetails ? handleUpdateLocation : handleAddLocation}>
                    {locationDetails ? 'Update Location' : 'Add Location'}
                </StyledButton>
            </StyledModalContent>
        </StyledModal>
    );
};

export default AddLocationModal;
