import React from 'react';
import { renderWithProviders } from '../../../test-utils';
import ProtectedRoute from '../ProtectedRoute';

describe('ProtectedRoute', () => {
  const TestComponent = () => <div>Protected Content</div>;

  beforeEach(() => {
    jest.clearAllMocks();
  });

  describe('Authentication States', () => {
    test('displays loading state when authentication is in progress', () => {
      const { getByText } = renderWithProviders(
        <ProtectedRoute>
          <TestComponent />
        </ProtectedRoute>,
        {
          authValue: {
            user: null,
            loading: true
          }
        }
      );

      expect(getByText('Loading...')).toBeInTheDocument();
    });

    test('renders protected content when user is authenticated', () => {
      const { getByText } = renderWithProviders(
        <ProtectedRoute>
          <TestComponent />
        </ProtectedRoute>,
        {
          authValue: {
            user: { id: '1', username: 'testuser' },
            loading: false
          }
        }
      );

      expect(getByText('Protected Content')).toBeInTheDocument();
    });

    test('redirects to login when user is not authenticated', () => {
      // When user is null and not loading, should redirect
      const { queryByText } = renderWithProviders(
        <ProtectedRoute>
          <TestComponent />
        </ProtectedRoute>,
        {
          authValue: {
            user: null,
            loading: false
          }
        }
      );

      // Protected content should not be rendered
      expect(queryByText('Protected Content')).not.toBeInTheDocument();
      // Note: We can't easily test the Navigate redirect in this test setup
      // but we can verify the protected content is not rendered
    });
  });

  describe('Edge Cases', () => {
    test('handles undefined user gracefully', () => {
      const { queryByText } = renderWithProviders(
        <ProtectedRoute>
          <TestComponent />
        </ProtectedRoute>,
        {
          authValue: {
            user: undefined,
            loading: false
          }
        }
      );

      expect(queryByText('Protected Content')).not.toBeInTheDocument();
    });

    test('prioritizes loading state over authentication check', () => {
      // Even with a user, loading state should take precedence
      const { getByText, queryByText } = renderWithProviders(
        <ProtectedRoute>
          <TestComponent />
        </ProtectedRoute>,
        {
          authValue: {
            user: { id: '1', username: 'testuser' },
            loading: true
          }
        }
      );

      expect(getByText('Loading...')).toBeInTheDocument();
      expect(queryByText('Protected Content')).not.toBeInTheDocument();
    });
  });

  describe('Children Rendering', () => {
    test('renders multiple children when authenticated', () => {
      const { getByText } = renderWithProviders(
        <ProtectedRoute>
          <div>First Child</div>
          <div>Second Child</div>
        </ProtectedRoute>,
        {
          authValue: {
            user: { id: '1', username: 'testuser' },
            loading: false
          }
        }
      );

      expect(getByText('First Child')).toBeInTheDocument();
      expect(getByText('Second Child')).toBeInTheDocument();
    });

    test('renders complex child components when authenticated', () => {
      const ComplexChild = () => (
        <div>
          <h1>Dashboard</h1>
          <p>Welcome to the protected area</p>
        </div>
      );

      const { getByText } = renderWithProviders(
        <ProtectedRoute>
          <ComplexChild />
        </ProtectedRoute>,
        {
          authValue: {
            user: { id: '1', username: 'testuser' },
            loading: false
          }
        }
      );

      expect(getByText('Dashboard')).toBeInTheDocument();
      expect(getByText('Welcome to the protected area')).toBeInTheDocument();
    });
  });
});