import React from 'react';
import { renderWithProviders, userEvent, testData } from '../../../test-utils';
import { waitFor } from '@testing-library/react';
import Login from '../Login';

// Mock useNavigate
const mockNavigate = jest.fn();
jest.mock('react-router-dom', () => ({
  ...jest.requireActual('react-router-dom'),
  useNavigate: () => mockNavigate,
}));

// Mock fetch globally
global.fetch = jest.fn();

describe('Login', () => {
  beforeEach(() => {
    jest.clearAllMocks();
    fetch.mockClear();
    window.localStorage.clear();

    // Mock IDP providers fetch that happens on component mount
    fetch.mockResolvedValueOnce({
      ok: true,
      json: async () => ({ providers: [] })
    });
  });

  describe('Rendering', () => {
    test('renders login form when user is not authenticated', () => {
      const { getByRole, container } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      // Look for input fields more directly
      const usernameInput = container.querySelector('input[name="username"]');
      const passwordInput = container.querySelector('input[name="password"]');
      
      expect(usernameInput).toBeInTheDocument();
      expect(passwordInput).toBeInTheDocument();
      expect(getByRole('button', { name: 'Login' })).toBeInTheDocument();
    });

    test('displays Kamiwaza branding', () => {
      const { getByAltText } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      // The logo now includes the text, so we only need to check for the logo image
      expect(getByAltText('Kamiwaza Logo')).toBeInTheDocument();
      
      // NOTE: If the logo is changed back to icon-only, restore these individual letter tests:
      // expect(getByText('K')).toBeInTheDocument();
      // expect(getByText('A')).toBeInTheDocument();
      // expect(getByText('M')).toBeInTheDocument();
      // expect(getByText('I')).toBeInTheDocument();
      // expect(getByText('WAZA')).toBeInTheDocument();
    });

    test('displays "Sign In" heading', () => {
      const { getByText } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      expect(getByText('Sign In')).toBeInTheDocument();
    });

    test('does not render when user is already authenticated', () => {
      const { container } = renderWithProviders(<Login />, {
        authValue: {
          user: testData.user,
          loading: false
        }
      });

      // Should return null when user is authenticated
      expect(container.firstChild).toBeNull();
    });
  });

  describe('Auto-redirect for authenticated users', () => {
    test('redirects to home when user is already logged in', () => {
      renderWithProviders(<Login />, {
        authValue: {
          user: testData.user,
          loading: false
        }
      });

      expect(mockNavigate).toHaveBeenCalledWith('/', { replace: true });
    });
  });

  describe('Form Input Handling', () => {
    test('updates username field when user types', async () => {
      const user = userEvent.setup();
      const { container } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      const usernameInput = container.querySelector('input[name="username"]');
      await user.type(usernameInput, 'testuser');

      expect(usernameInput).toHaveValue('testuser');
    });

    test('updates password field when user types', async () => {
      const user = userEvent.setup();
      const { container } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      const passwordInput = container.querySelector('input[name="password"]');
      await user.type(passwordInput, 'testpassword');

      expect(passwordInput).toHaveValue('testpassword');
    });

    test('username field receives focus on component mount', () => {
      const { container } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      const usernameInput = container.querySelector('input[name="username"]');
      
      // Test the actual behavior - that the username field receives focus
      // This is more reliable than checking HTML attributes with MUI
      expect(document.activeElement).toBe(usernameInput);
    });

    test('form fields are required', () => {
      const { container } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      expect(container.querySelector('input[name="username"]')).toBeRequired();
      expect(container.querySelector('input[name="password"]')).toBeRequired();
    });
  });

  describe('Password Visibility Toggle', () => {
    test('password field starts hidden', () => {
      const { container } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      const passwordInput = container.querySelector('input[name="password"]');
      expect(passwordInput).toHaveAttribute('type', 'password');
    });

    test('toggles password visibility when eye icon is clicked', async () => {
      const user = userEvent.setup();
      const { container } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      const passwordInput = container.querySelector('input[name="password"]');
      
      // Find the visibility toggle button (it should be an IconButton)
      const toggleButton = container.querySelector('button[type="button"]');
      expect(toggleButton).toBeInTheDocument();

      // Initially password type
      expect(passwordInput).toHaveAttribute('type', 'password');

      // Click to show password
      await user.click(toggleButton);
      expect(passwordInput).toHaveAttribute('type', 'text');

      // Click again to hide password
      await user.click(toggleButton);
      expect(passwordInput).toHaveAttribute('type', 'password');
    });
  });

  describe('Form Submission', () => {
    test('calls login function with correct credentials on form submit', async () => {
      const mockFetchUserData = jest.fn().mockResolvedValue({});
      const user = userEvent.setup();

      // Mock successful login response
      fetch.mockResolvedValueOnce({
        ok: true,
        json: async () => ({
          access_token: 'fake-token',
          refresh_token: 'fake-refresh-token'
        })
      });

      const { container, getByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          fetchUserData: mockFetchUserData,
          loading: false
        }
      });

      // Fill out form
      await user.type(container.querySelector('input[name="username"]'), 'testuser');
      await user.type(container.querySelector('input[name="password"]'), 'testpass');

      // Submit form
      await user.click(getByRole('button', { name: 'Login' }));

      expect(fetch).toHaveBeenCalledTimes(2); // 1 for IDP providers, 1 for login
      expect(fetch).toHaveBeenCalledWith(
        expect.stringContaining('/auth/token'),
        expect.objectContaining({
          method: 'POST',
          headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
          },
          body: expect.any(URLSearchParams)
        })
      );

      // Check that URLSearchParams was created with correct data
      const loginCallArgs = fetch.mock.calls.find(call => call[0].includes('/auth/token'));
      expect(loginCallArgs[1].body.get('username')).toBe('testuser');
      expect(loginCallArgs[1].body.get('password')).toBe('testpass');
      expect(loginCallArgs[1].body.get('grant_type')).toBe('password');
    });

    test('navigates to home page after successful login', async () => {
      const mockFetchUserData = jest.fn().mockResolvedValue({});
      const user = userEvent.setup();

      // Mock successful login response (IDP providers already mocked in beforeEach)
      fetch.mockResolvedValueOnce({
        ok: true,
        json: async () => ({ access_token: 'fake-token' })
      });

      const { container, getByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          fetchUserData: mockFetchUserData,
          loading: false
        }
      });

      await user.type(container.querySelector('input[name="username"]'), 'testuser');
      await user.type(container.querySelector('input[name="password"]'), 'testpass');

      // Submit the form
      await user.click(getByRole('button', { name: 'Login' }));

      // Wait for the login API call to complete
      await waitFor(() => {
        expect(fetch).toHaveBeenCalledWith(
          expect.stringContaining('/auth/token'),
          expect.objectContaining({
            method: 'POST',
            credentials: 'include'
          })
        );
      });

      // Wait for navigation
      await waitFor(() => {
        expect(mockNavigate).toHaveBeenCalledWith('/', { replace: true });
      });

      expect(mockFetchUserData).toHaveBeenCalled();
    });

    test('respects redirect query parameter after successful login', async () => {
      const mockFetchUserData = jest.fn().mockResolvedValue({});
      const user = userEvent.setup();

      fetch.mockResolvedValueOnce({
        ok: true,
        json: async () => ({ access_token: 'fake-token' })
      });

      window.history.pushState({}, 'Test', '/login?redirect=/models');

      const { container, getByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          fetchUserData: mockFetchUserData,
          loading: false
        },
      });

      await user.type(container.querySelector('input[name="username"]'), 'testuser');
      await user.type(container.querySelector('input[name="password"]'), 'testpass');
      await user.click(getByRole('button', { name: 'Login' }));

      await waitFor(() => {
        expect(mockNavigate).toHaveBeenCalledWith('/models', { replace: true });
      });

      expect(mockFetchUserData).toHaveBeenCalled();
    });

    test('prevents form submission with empty fields', async () => {
      const user = userEvent.setup();

      const { getByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      // Try to submit without filling fields
      await user.click(getByRole('button', { name: 'Login' }));

      // HTML5 validation should prevent submission
      // Only the IDP providers call should have been made, no login call
      expect(fetch).toHaveBeenCalledTimes(1);
      expect(fetch).toHaveBeenCalledWith(expect.stringContaining('/auth/idp/public/providers'));
    });
  });

  describe('Error Handling', () => {
    test('displays error message for invalid credentials (401)', async () => {
      const user = userEvent.setup();

      // Mock 401 error response
      fetch.mockResolvedValueOnce({
        ok: false,
        status: 401,
        headers: new Headers({ 'content-type': 'text/plain' }),
        text: async () => 'Login failed'
      });

      const { container, getByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      await user.type(container.querySelector('input[name="username"]'), 'wronguser');
      await user.type(container.querySelector('input[name="password"]'), 'wrongpass');
      await user.click(getByRole('button', { name: 'Login' }));

      await waitFor(() => {
        const alert = getByRole('alert');
        expect(alert).toBeInTheDocument();
        expect(alert).toHaveTextContent('Login failed');
      });
    });

    test.skip('displays service unavailable message for network errors', async () => {
      const user = userEvent.setup();

      // Mock network error - component will show default error message
      fetch.mockRejectedValueOnce(new Error('Network Error'));

      const { container, getByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      await user.type(container.querySelector('input[name="username"]'), 'testuser');
      await user.type(container.querySelector('input[name="password"]'), 'testpass');
      await user.click(getByRole('button', { name: 'Login' }));

      await waitFor(() => {
        const alert = getByRole('alert');
        expect(alert).toBeInTheDocument();
        // Component shows the error message or defaults to 'Invalid username or password'
        expect(alert).toHaveTextContent('Invalid username or password');
      });
    });

    test('displays generic error message for unexpected errors', async () => {
      const user = userEvent.setup();

      // Mock 500 error response
      fetch.mockResolvedValueOnce({
        ok: false,
        status: 500,
        headers: new Headers({ 'content-type': 'text/plain' }),
        text: async () => 'Internal Server Error'
      });

      const { container, getByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      await user.type(container.querySelector('input[name="username"]'), 'testuser');
      await user.type(container.querySelector('input[name="password"]'), 'testpass');
      await user.click(getByRole('button', { name: 'Login' }));

      await waitFor(() => {
        const alert = getByRole('alert');
        expect(alert).toBeInTheDocument();
        // Component shows the server error text
        expect(alert).toHaveTextContent('Internal Server Error');
      });
    });

    test('clears error message when form is resubmitted', async () => {
      const mockFetchUserData = jest.fn().mockResolvedValue({});
      const user = userEvent.setup();

      // Mock 401 error for first call, success for second
      fetch
        .mockResolvedValueOnce({
          ok: false,
          status: 401,
          headers: new Headers({ 'content-type': 'text/plain' }),
          text: async () => 'Login failed'
        })
        .mockResolvedValueOnce({
          ok: true,
          json: async () => ({
            access_token: 'fake-token',
            refresh_token: 'fake-refresh-token'
          })
        });

      const { container, getByRole, queryByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          fetchUserData: mockFetchUserData,
          loading: false
        }
      });

      // First submission with error
      await user.type(container.querySelector('input[name="username"]'), 'wronguser');
      await user.type(container.querySelector('input[name="password"]'), 'wrongpass');
      await user.click(getByRole('button', { name: 'Login' }));

      await waitFor(() => {
        expect(getByRole('alert')).toHaveTextContent('Login failed');
      });

      // Clear fields and try again
      const usernameInput = container.querySelector('input[name="username"]');
      const passwordInput = container.querySelector('input[name="password"]');
      await user.clear(usernameInput);
      await user.clear(passwordInput);
      await user.type(usernameInput, 'correctuser');
      await user.type(passwordInput, 'correctpass');
      await user.click(getByRole('button', { name: 'Login' }));

      // Error should be cleared
      await waitFor(() => {
        expect(queryByRole('alert')).not.toBeInTheDocument();
      });
    });
  });

  describe('Accessibility', () => {
    test('form fields have proper labels', () => {
      const { container } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      // Check that form inputs exist and have name attributes for accessibility
      expect(container.querySelector('input[name="username"]')).toBeInTheDocument();
      expect(container.querySelector('input[name="password"]')).toBeInTheDocument();
      
      // Check that labels exist in the document
      expect(container.querySelector('label[for]')).toBeInTheDocument();
    });

    test('submit button is accessible', () => {
      const { getByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      const submitButton = getByRole('button', { name: 'Login' });
      expect(submitButton).toBeInTheDocument();
      expect(submitButton).toHaveAttribute('type', 'submit');
    });

    test('error messages have proper alert role', async () => {
      const user = userEvent.setup();

      // Mock 401 error response
      fetch.mockResolvedValueOnce({
        ok: false,
        status: 401
      });

      const { container, getByRole, findByRole } = renderWithProviders(<Login />, {
        authValue: {
          user: null,
          loading: false
        }
      });

      await user.type(container.querySelector('input[name="username"]'), 'wronguser');
      await user.type(container.querySelector('input[name="password"]'), 'wrongpass');
      await user.click(getByRole('button', { name: 'Login' }));

      await waitFor(() => {
        const errorAlert = getByRole('alert');
        expect(errorAlert).toBeInTheDocument();
        // The component shows this message when it can't parse the error response
        expect(errorAlert).toHaveTextContent('Unable to contact authentication service. Check connectivity and try again.');
      });
    });
  });
});
