import React from 'react';
import { renderWithProviders, testData } from '../../../test-utils';
import AdminRoute from '../AdminRoute';

// Mock Navigate component to capture redirect attempts
const mockNavigate = jest.fn();
jest.mock('react-router-dom', () => ({
  ...jest.requireActual('react-router-dom'),
  Navigate: ({ to, replace, state }) => {
    mockNavigate({ to, replace, state });
    return null;
  },
  useLocation: () => ({
    pathname: '/cluster/home',
    search: '',
    hash: '',
  }),
}));

describe('AdminRoute', () => {
  // Admin user for testing
  const adminUser = { ...testData.user, roles: ['admin'], is_superuser: true };
  // Regular non-admin user
  const regularUser = { ...testData.user, roles: ['user'], is_superuser: false };

  beforeEach(() => {
    jest.clearAllMocks();
  });

  describe('Loading State', () => {
    test('renders loading indicator when auth is loading', () => {
      const { getByText } = renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: null,
            loading: true
          }
        }
      );

      expect(getByText('Loading...')).toBeInTheDocument();
    });

    test('does not render children when loading', () => {
      const { queryByText } = renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: adminUser,
            loading: true
          }
        }
      );

      expect(queryByText('Protected Content')).not.toBeInTheDocument();
      expect(queryByText('Loading...')).toBeInTheDocument();
    });
  });

  describe('Unauthenticated User', () => {
    test('redirects to login when user is null', () => {
      renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: null,
            loading: false
          }
        }
      );

      expect(mockNavigate).toHaveBeenCalledWith({
        to: '/login',
        replace: true,
        state: { from: '/cluster/home' }
      });
    });

    test('redirects to login when user is undefined', () => {
      renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: undefined,
            loading: false
          }
        }
      );

      expect(mockNavigate).toHaveBeenCalledWith({
        to: '/login',
        replace: true,
        state: { from: '/cluster/home' }
      });
    });
  });

  describe('Non-Admin User', () => {
    test('redirects non-admin user to homepage', () => {
      renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: regularUser,
            loading: false
          }
        }
      );

      expect(mockNavigate).toHaveBeenCalledWith({
        to: '/',
        replace: true,
        state: undefined
      });
    });

    test('redirects user with empty roles array to homepage', () => {
      const userWithEmptyRoles = { ...testData.user, roles: [] };

      renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: userWithEmptyRoles,
            loading: false
          }
        }
      );

      expect(mockNavigate).toHaveBeenCalledWith({
        to: '/',
        replace: true,
        state: undefined
      });
    });

    test('redirects user with undefined roles to homepage', () => {
      const userWithNoRoles = { ...testData.user, roles: undefined };

      renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: userWithNoRoles,
            loading: false
          }
        }
      );

      expect(mockNavigate).toHaveBeenCalledWith({
        to: '/',
        replace: true,
        state: undefined
      });
    });
  });

  describe('Admin User', () => {
    test('renders children for admin user', () => {
      const { getByText, queryByText } = renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: adminUser,
            loading: false
          }
        }
      );

      expect(getByText('Protected Content')).toBeInTheDocument();
      expect(queryByText('Loading...')).not.toBeInTheDocument();
      expect(mockNavigate).not.toHaveBeenCalled();
    });

    test('renders complex children for admin user', () => {
      const { getByText, getByRole } = renderWithProviders(
        <AdminRoute>
          <div>
            <h1>Admin Dashboard</h1>
            <button>Admin Action</button>
          </div>
        </AdminRoute>,
        {
          authValue: {
            user: adminUser,
            loading: false
          }
        }
      );

      expect(getByText('Admin Dashboard')).toBeInTheDocument();
      expect(getByRole('button', { name: 'Admin Action' })).toBeInTheDocument();
    });

    test('admin user with multiple roles including admin is allowed', () => {
      const userWithMultipleRoles = { ...testData.user, roles: ['user', 'admin', 'editor'] };

      const { getByText } = renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: userWithMultipleRoles,
            loading: false
          }
        }
      );

      expect(getByText('Protected Content')).toBeInTheDocument();
      expect(mockNavigate).not.toHaveBeenCalled();
    });
  });

  describe('Edge Cases', () => {
    test('handles user with roles as non-array gracefully', () => {
      const userWithStringRole = { ...testData.user, roles: 'admin' };

      renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: userWithStringRole,
            loading: false
          }
        }
      );

      // Should redirect since roles is not an array
      expect(mockNavigate).toHaveBeenCalledWith({
        to: '/',
        replace: true,
        state: undefined
      });
    });

    test('handles user with null roles gracefully', () => {
      const userWithNullRoles = { ...testData.user, roles: null };

      renderWithProviders(
        <AdminRoute>
          <div>Protected Content</div>
        </AdminRoute>,
        {
          authValue: {
            user: userWithNullRoles,
            loading: false
          }
        }
      );

      // Should redirect since roles is null
      expect(mockNavigate).toHaveBeenCalledWith({
        to: '/',
        replace: true,
        state: undefined
      });
    });
  });
});
