import React, { useState, useContext, useEffect, useMemo } from 'react';
import { useNavigate, useLocation } from 'react-router-dom';
import { AuthContext } from '../../context/AuthContext';
import { BASE_URL } from '../../const';
import { styled } from '@mui/material/styles';
import {
  Box,
  TextField,
  Button,
  Paper,
  Typography,
  Container,
  Alert,
  InputAdornment,
  IconButton,
  Stack
} from '@mui/material';
import { Visibility, VisibilityOff, PersonOutline, LockOutlined, Google, Login as LoginIcon } from '@mui/icons-material';

const fontStack = `var(--app-font-stack)`;

const LoginContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  flexDirection: 'column',
  alignItems: 'center',
  justifyContent: 'center',
  minHeight: '100vh',
  padding: theme.spacing(3),
  backgroundColor: theme.palette.background.default,
  fontFamily: fontStack,
}));

const LoginForm = styled(Paper)(({ theme }) => ({
  padding: theme.spacing(4),
  borderRadius: 16,
  maxWidth: 450,
  width: '100%',
  display: 'flex',
  flexDirection: 'column',
  gap: theme.spacing(3),
  backgroundColor: theme.palette.background.paper,
  boxShadow: '0 8px 32px rgba(0, 0, 0, 0.2)',
  position: 'relative',
  overflow: 'hidden',
  fontFamily: fontStack,
  '&::before': {
    content: '""',
    position: 'absolute',
    top: 0,
    left: 0,
    width: '100%',
    height: '5px',
    background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  },
}));

const StyledTextField = styled(TextField)(({ theme }) => ({
  fontFamily: fontStack,
  '& .MuiOutlinedInput-root': {
    borderRadius: 8,
    backgroundColor: 'rgba(255, 255, 255, 0.03)',
    '&:hover .MuiOutlinedInput-notchedOutline': {
      borderColor: theme.palette.primary.main,
    },
    '&.Mui-focused .MuiOutlinedInput-notchedOutline': {
      borderColor: `${theme.palette.primary.main} !important`,
      borderWidth: '1px !important',
    },
    '& input, & input::placeholder': {
      fontFamily: fontStack,
    },
    // Fix for autofill blue background
    '& input:-webkit-autofill, & input:-webkit-autofill:hover, & input:-webkit-autofill:focus, & input:-webkit-autofill:active': {
      WebkitBoxShadow: '0 0 0 30px #121212 inset !important', // Match your background color
      WebkitTextFillColor: '#ffffff !important', // Match your text color
      caretColor: '#ffffff !important',
      borderRadius: 'inherit',
      fontFamily: fontStack,
    },
  },
  '& .MuiInputLabel-root': {
    color: theme.palette.text.secondary,
    fontFamily: fontStack,
  },
  '& .MuiOutlinedInput-notchedOutline': {
    borderColor: 'rgba(255, 255, 255, 0.1)',
  },
  // Fix for focus
  '& .MuiOutlinedInput-input': {
    padding: theme.spacing(1.5),
    '&:focus': {
      boxShadow: 'none',
    },
    fontFamily: fontStack,
  },
}));

const StyledButton = styled(Button)(({ theme }) => ({
  borderRadius: 8,
  padding: theme.spacing(1.2, 2),
  fontSize: '1rem',
  fontWeight: 600,
  textTransform: 'none',
  boxShadow: '0 4px 10px rgba(0, 192, 127, 0.3)',
  background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  transition: 'all 0.3s ease',
  fontFamily: fontStack,
  '&:hover': {
    boxShadow: '0 6px 15px rgba(0, 192, 127, 0.4)',
    transform: 'translateY(-2px)',
    background: 'linear-gradient(90deg, #00c07f, #00c0a0)',
  },
}));

const Logo = styled('img')(({ theme }) => ({
  height: 150,
  marginBottom: theme.spacing(2),
  display: 'block',
  margin: '0 auto',
}));

const sanitizeRedirectTarget = (candidate) => {
  const defaultTarget = '/';
  if (!candidate || typeof candidate !== 'string') {
    return defaultTarget;
  }
  const trimmed = candidate.trim();
  if (!trimmed || trimmed.startsWith('//')) {
    return defaultTarget;
  }

  const origin = typeof window !== 'undefined' ? window.location.origin : '';
  const normalizeFromUrl = (url) => {
    const path = `${url.pathname}${url.search}${url.hash}` || defaultTarget;
    return path === '/login' ? defaultTarget : path;
  };

  if (/^https?:\/\//i.test(trimmed)) {
    try {
      const abs = new URL(trimmed);
      if (origin && abs.origin !== origin) {
        return defaultTarget;
      }
      return normalizeFromUrl(abs);
    } catch (_) {
      return defaultTarget;
    }
  }

  try {
    const base = origin || 'http://localhost';
    const prepared = trimmed.startsWith('/') ? trimmed : `/${trimmed}`;
    const resolved = new URL(prepared, base);
    return normalizeFromUrl(resolved);
  } catch (_) {
    const fallback = trimmed.startsWith('/') ? trimmed : `/${trimmed}`;
    return fallback === '/login' ? defaultTarget : fallback;
  }
};

const Login = () => {
  const [credentials, setCredentials] = useState({ username: '', password: '' });
  const [showPassword, setShowPassword] = useState(false);
  const { user, fetchUserData } = useContext(AuthContext);
  const [error, setError] = useState(null);
  const [idps, setIdps] = useState([]);
  const navigate = useNavigate();
  const location = useLocation();
  const stateRedirect = typeof location.state?.from === 'string' ? location.state.from : null;
  const redirectTarget = useMemo(() => {
    const searchParams = new URLSearchParams(location.search || '');
    const queryRedirect = searchParams.get('redirect');
    return sanitizeRedirectTarget(stateRedirect || queryRedirect);
  }, [stateRedirect, location.search]);
  const absoluteRedirectTarget = useMemo(() => {
    if (typeof window === 'undefined') {
      return redirectTarget;
    }
    return `${window.location.origin}${redirectTarget}`;
  }, [redirectTarget]);

  useEffect(() => {
    // If user is already logged in, redirect to intended destination
    if (user) {
      // Paths served by Traefik to backend services (not React app routes)
      // These require full page navigation, not React Router client-side nav
      const externalPathPrefixes = ['/runtime/', '/lab/', '/admin/'];
      const needsFullNavigation = externalPathPrefixes.some(prefix =>
        redirectTarget.startsWith(prefix)
      );

      if (needsFullNavigation) {
        // Full page navigation for non-React paths
        window.location.href = redirectTarget;
      } else {
        // React Router navigation for React app routes
        navigate(redirectTarget, { replace: true });
      }
    }
  }, [user, navigate, redirectTarget]);

  useEffect(() => {
    // Load configured IdPs to show SSO buttons
    const fetchIdps = async () => {
      try {
        const resp = await fetch(`${BASE_URL}/auth/idp/public/providers`);
        const data = await resp.json();
        if (resp.ok && data.providers) {
          setIdps(data.providers);
        }
      } catch (_) {
        // ignore
      }
    };
    fetchIdps();
  }, []);

  const handleChange = (e) => {
    setCredentials({ ...credentials, [e.target.name]: e.target.value });
  };

  const handleTogglePasswordVisibility = () => {
    setShowPassword(!showPassword);
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setError(null);
    
    try {
      // POST credentials to the backend token endpoint
      const response = await fetch(`${BASE_URL}/auth/token`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/x-www-form-urlencoded',
        },
        credentials: 'include',
        body: new URLSearchParams({
          username: credentials.username,
          password: credentials.password,
          grant_type: 'password',
          scope: 'openid email profile roles offline_access'
        })
      });

      if (!response.ok) {
        let message = 'Login failed';
        try {
          const ct = response.headers.get('content-type') || '';
          if (ct.includes('application/json')) {
            const errorData = await response.json();
            message = errorData.detail || errorData.error || message;
          } else {
            const text = await response.text();
            // Common during startup: Bad Gateway / Service Unavailable
            if (response.status === 502 || response.status === 503) {
              message = 'Authentication service is starting up. Please retry in a moment.';
            } else {
              message = text?.substring(0, 200) || message;
            }
          }
        } catch (e) {
          // Network or parse error; show a friendly message instead of raw exception
          if (response.status === 502 || response.status === 503) {
            message = 'Authentication service is starting up. Please retry shortly.';
          } else {
            message = 'Unable to contact authentication service. Check connectivity and try again.';
          }
        }
        throw new Error(message);
      }

      await response.json();

      // Populate AuthContext from the server-side session cookie
      try {
        await fetchUserData();
      } catch (e) {
        // Non-fatal; continue to redirect
      }

      // Redirect to the intended destination (defaults to home)
      // Paths served by Traefik to backend services (not React app routes)
      // require full page navigation, not React Router client-side nav
      const externalPathPrefixes = ['/runtime/', '/lab/', '/admin/'];
      const needsFullNavigation = externalPathPrefixes.some(prefix =>
        redirectTarget.startsWith(prefix)
      );

      if (needsFullNavigation) {
        window.location.href = redirectTarget;
      } else {
        navigate(redirectTarget, { replace: true });
      }
      
    } catch (error) {
      console.error('Login error:', error);
      // Normalize noisy messages like JSON parse errors
      let msg = error?.message || 'Invalid username or password';
      if (/Unexpected token/i.test(msg) || /JSON/.test(msg)) {
        msg = 'Authentication service not ready. Please try again in a few seconds.';
      }
      setError(msg);
    }
  };

  // If user is already logged in, don't render the form
  if (user) {
    return null;
  }

  return (
    <LoginContainer>
      <Container maxWidth="sm" sx={{ display: 'flex', flexDirection: 'column', alignItems: 'center' }}>
        <Box sx={{ width: '100%', display: 'flex', flexDirection: 'column', alignItems: 'center', justifyContent: 'center', mb: 4 }}>
          <Logo src="/kamiwaza-logo.png" alt="Kamiwaza Logo" />
        </Box>
        
        <LoginForm elevation={6}>
          <Typography variant="h5" fontWeight="600" align="center" gutterBottom sx={{ fontFamily: fontStack }}>
            Sign In
          </Typography>
          
          {error && (
            <Alert severity="error" sx={{ borderRadius: 2 }}>
              {error}
            </Alert>
          )}
          
          <form onSubmit={handleSubmit} style={{ display: 'flex', flexDirection: 'column', gap: 20 }}>
            <StyledTextField
              label="Username"
              name="username"
              value={credentials.username}
              onChange={handleChange}
              fullWidth
              required
              autoFocus
              variant="outlined"
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <PersonOutline sx={{ color: 'text.secondary' }} />
                  </InputAdornment>
                ),
              }}
            />
            
            <StyledTextField
              label="Password"
              name="password"
              type={showPassword ? 'text' : 'password'}
              value={credentials.password}
              onChange={handleChange}
              fullWidth
              required
              variant="outlined"
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <LockOutlined sx={{ color: 'text.secondary' }} />
                  </InputAdornment>
                ),
                endAdornment: (
                  <InputAdornment position="end">
                    <IconButton
                      onClick={handleTogglePasswordVisibility}
                      edge="end"
                      sx={{ color: 'text.secondary' }}
                    >
                      {showPassword ? <VisibilityOff /> : <Visibility />}
                    </IconButton>
                  </InputAdornment>
                ),
              }}
            />
            
          <StyledButton 
            type="submit" 
            variant="contained" 
            fullWidth
            size="large"
          >
            Login
          </StyledButton>
          {idps && idps.length > 0 && (
            <Stack spacing={1.5} sx={{ mt: 2 }}>
              <Typography variant="body2" color="text.secondary" align="center">
                Or continue with
              </Typography>
              {idps.map((provider) => {
                const isGoogle = (provider.providerId || '').toLowerCase() === 'google' || provider.alias === 'google';
                const label = isGoogle
                  ? 'Sign in with Google'
                  : `Sign in with ${provider.displayName || provider.alias}`;
                const IconComponent = isGoogle ? Google : LoginIcon;
                return (
                  <StyledButton
                    key={provider.alias}
                    type="button"
                    variant="contained"
                    fullWidth
                    size="large"
                    onClick={() => {
                      const params = new URLSearchParams({ provider: provider.alias });
                      if (absoluteRedirectTarget) {
                        params.set('redirect_uri', absoluteRedirectTarget);
                      }
                      window.location.href = `${BASE_URL}/auth/login?${params.toString()}`;
                    }}
                    startIcon={<IconComponent fontSize="small" />}
                  >
                    {label}
                  </StyledButton>
                );
              })}
            </Stack>
          )}
          </form>
        </LoginForm>
      </Container>
    </LoginContainer>
  );
};

export default Login;
