import React, { useContext } from 'react';
import PropTypes from 'prop-types';
import { Navigate, useLocation } from 'react-router-dom';
import { AuthContext } from '../../context/AuthContext';

/**
 * AdminRoute - Protects routes that require admin role
 *
 * Usage: Wrap routes that should only be accessible to admin users
 * - If loading: shows loading indicator
 * - If not authenticated: redirects to /login
 * - If authenticated but not admin: redirects to /
 * - If admin: renders children
 */
const AdminRoute = ({ children }) => {
  const { user, loading } = useContext(AuthContext);
  const location = useLocation();

  if (loading) {
    return <div>Loading...</div>;
  }

  if (!user) {
    const attemptedPath = `${location.pathname}${location.search}${location.hash}`;
    return (
      <Navigate
        to="/login"
        replace
        state={{ from: attemptedPath || '/' }}
      />
    );
  }

  // Check if user has admin role
  const isAdmin = user && Array.isArray(user.roles) && user.roles.includes('admin');

  if (!isAdmin) {
    // Non-admin users are redirected to homepage
    return <Navigate to="/" replace />;
  }

  return children;
};

AdminRoute.propTypes = {
  children: PropTypes.node.isRequired,
};

export default AdminRoute;
