// App Garden Template Variables Constants

// Template variable regex pattern - matches {variable_name}
export const TEMPLATE_VARIABLE_PATTERN = /\{([a-zA-Z_][a-zA-Z0-9_]*)\}/g;

// Available template variables
export const TEMPLATE_VARIABLES = {
  app_port: {
    name: 'app_port',
    description: 'External port (443 in single-port routing) used for app runtime URLs',
    example: '443',
    deprecated: false,
  },
  model_name: {
    name: 'model_name',
    description: 'Resolved model provider/name (e.g., qwen/qwen0.6b)',
    example: 'openai/gpt-4o-mini',
    deprecated: false,
    availability: 'Only when a model is selected for the deployment',
  },
  lb_port: {
    name: 'lb_port',
    description: 'Legacy name for app_port',
    example: '443',
    deprecated: true,
    deprecationMessage: 'Use {app_port} instead',
  },
  deployment_id: {
    name: 'deployment_id',
    description: 'Unique identifier for the deployment',
    example: '09c6ce9a-0ce6-415c-922e-d8de686e1b7d',
    deprecated: false,
  },
  app_name: {
    name: 'app_name',
    description: 'Name of the deployed application',
    example: 'my-app',
    deprecated: false,
  },
  model_port: {
    name: 'model_port',
    description: 'Port of the deployed AI model (443 when path routing is active)',
    example: '443',
    deprecated: false,
    availability: 'Only when model is deployed',
  },
  app_access_path: {
    name: 'app_access_path',
    description: 'Path prefix exposed via Traefik for this app runtime',
    example: '/runtime/apps/123e4567-e89b-12d3-a456-426614174000',
    deprecated: false,
  },
  app_path: {
    name: 'app_path',
    description: 'Final serve_path assigned to the application.',
    example: '/runtime/apps/123e4567-e89b-12d3-a456-426614174000',
    deprecated: false,
  },
  model_access_path: {
    name: 'model_access_path',
    description: 'Path prefix exposed via Traefik for the selected model',
    example: '/runtime/models/123e4567-e89b-12d3-a456-426614174000',
    deprecated: false,
    availability: 'Only when model is deployed',
  },
  model_path: {
    name: 'model_path',
    description: 'Serve_path assigned to the backing model deployment.',
    example: '/runtime/models/123e4567-e89b-12d3-a456-426614174000',
    deprecated: false,
    availability: 'Only when model is deployed',
  },
  openai_base_url: {
    name: 'openai_base_url',
    description: 'Legacy port-based OpenAI-compatible base URL without trailing /v1',
    example: 'https://host.docker.internal:61111',
    deprecated: false,
    availability: 'Only when model is deployed',
  },
  openai_path_base_url: {
    name: 'openai_path_base_url',
    description: 'Path-based OpenAI-compatible base URL without trailing /v1',
    example: 'https://localhost/runtime/models/123e4567-e89b-12d3-a456-426614174000',
    deprecated: false,
    availability: 'Only when model is deployed',
  },
  app_path_url: {
    name: 'app_path_url',
    description: 'Full HTTPS URL for accessing the app via path routing',
    example: 'https://localhost/runtime/apps/123e4567-e89b-12d3-a456-426614174000',
    deprecated: false,
  },
  model_path_url: {
    name: 'model_path_url',
    description: 'Full HTTPS URL for accessing the model via path routing',
    example: 'https://localhost/runtime/models/123e4567-e89b-12d3-a456-426614174000',
    deprecated: false,
    availability: 'Only when model is deployed',
  },
  auth_gateway_tls_insecure: {
    name: 'auth_gateway_tls_insecure',
    description: 'TLS certificate validation setting (0=accept self-signed, 1=reject)',
    example: '0',
    deprecated: false,
  },
  kamiwaza_origin: {
    name: 'kamiwaza_origin',
    description: 'Origin used for console/auth redirects.',
    example: 'https://console.kamiwaza.ai',
    deprecated: false,
  },
  use_auth: {
    name: 'use_auth',
    description: 'ForwardAuth enforcement flag (true/false).',
    example: 'true',
    deprecated: false,
  },
  forwardauth_signature_secret: {
    name: 'forwardauth_signature_secret',
    description: 'ForwardAuth signature secret injected by Kamiwaza.',
    example: '***',
    deprecated: false,
    availability: 'Only when FORWARDAUTH_SIGNATURE_SECRET is configured',
  },
};

// System variables that are automatically injected
export const SYSTEM_VARIABLES = [
  {
    key: 'KAMIWAZA_DEPLOYMENT_ID',
    description: 'Unique identifier for this deployment.',
    placeholder: '{deployment_id}',
  },
  {
    key: 'KAMIWAZA_APP_NAME',
    description: 'Matches the name you choose when launching the app.',
    placeholder: '{app_name}',
  },
  {
    key: 'KAMIWAZA_APP_PORT',
    description: 'Port assigned for runtime access (443 when path routing is enabled).',
    placeholder: '{app_port}',
  },
  {
    key: 'KAMIWAZA_APP_URL',
    description: 'Port-based URL for reaching the deployed app through Traefik.',
    placeholder: '{kamiwaza_origin}:{app_port}',
  },
  {
    key: 'KAMIWAZA_API_URL',
    description: 'Base URL for the Kamiwaza API.',
    placeholder: 'http://host.docker.internal:7777/api',
  },
  {
    key: 'KAMIWAZA_PUBLIC_API_URL',
    description: 'External API base (public host + /api) for browser-facing calls.',
    placeholder: '{kamiwaza_origin}/api',
  },
  {
    key: 'KAMIWAZA_APP_INTERNAL_URL',
    description: 'Port-based internal URL for reaching the app from within the cluster.',
    placeholder: 'https://host.docker.internal:{app_port}',
  },
  {
    key: 'KAMIWAZA_APP_PATH',
    description: 'Path prefix served by Traefik for this deployment.',
    placeholder: '/runtime/apps/{deployment_id}',
  },
  {
    key: 'KAMIWAZA_APP_PATH_URL',
    description: 'Full path-based URL composed from {kamiwaza_origin} and {app_path}.',
    placeholder: '{kamiwaza_origin}{app_path}',
  },
  {
    key: 'KAMIWAZA_ORIGIN',
    description: 'Console origin exposed to templates and system vars.',
    placeholder: '{kamiwaza_origin}',
  },
  {
    key: 'KAMIWAZA_USE_AUTH',
    description: 'ForwardAuth enforcement flag (true/false).',
    placeholder: '{use_auth}',
  },
  {
    key: 'KAMIWAZA_MODEL_PORT',
    description: 'Available when a model preference is resolved; points to the backing model.',
    placeholder: '{model_port}',
  },
  {
    key: 'KAMIWAZA_MODEL_NAME',
    description: 'Provider/name of the model selected for this deployment.',
    placeholder: '{model_name}',
  },
  {
    key: 'KAMIWAZA_MODEL_URL',
    description: 'Port-based URL for the configured model via HTTPS.',
    placeholder: 'https://host.docker.internal:{model_port}',
  },
  {
    key: 'KAMIWAZA_MODEL_PATH',
    description: 'Serve_path for the configured model deployment.',
    placeholder: '{model_path}',
  },
  {
    key: 'KAMIWAZA_MODEL_PATH_URL',
    description: 'Full path-based URL derived from {kamiwaza_origin} + {model_path}.',
    placeholder: '{kamiwaza_origin}{model_path}',
  },
  {
    key: 'OPENAI_BASE_URL',
    description: 'Port-based OpenAI-compatible base URL for the selected model.',
    placeholder: '{openai_base_url}/v1',
  },
  {
    key: 'OPENAI_PATH_BASE_URL',
    description: 'Path-based OpenAI-compatible base URL for the selected model.',
    placeholder: '{openai_path_base_url}/v1',
  },
  {
    key: 'KAMIWAZA_TLS_REJECT_UNAUTHORIZED',
    description: 'TLS certificate validation control (0=accept self-signed certs, 1=strict validation).',
    placeholder: '{auth_gateway_tls_insecure}',
  },
  {
    key: 'FORWARDAUTH_SIGNATURE_SECRET',
    description: 'ForwardAuth HMAC secret passed through when configured.',
    placeholder: '{forwardauth_signature_secret}',
  },
  {
    key: 'KAMIWAZA_APP_SESSION_TOKEN',
    description: 'Opaque token used by the runtime/UI to signal app session events.',
    placeholder: 'random-token',
  },
  {
    key: 'KAMIWAZA_APP_SESSION_ENDPOINT',
    description: 'Relative API path for ending the app session (purge on logout). Use KAMIWAZA_APP_SESSION_TOKEN in Authorization header.',
    placeholder: '/api/apps/sessions/end',
  },
  {
    key: 'KAMIWAZA_APP_SESSION_HEARTBEAT_ENDPOINT',
    description: 'Relative API path for heartbeats to keep the session alive. Use KAMIWAZA_APP_SESSION_TOKEN in Authorization header.',
    placeholder: '/api/apps/sessions/heartbeat',
  },
];

export const SYSTEM_VARIABLE_KEYS = SYSTEM_VARIABLES.map((variable) => variable.key);

// Helper function to detect template variables in a string
export const detectTemplateVariables = (value) => {
  if (!value || typeof value !== 'string') {
    return [];
  }

  const matches = [];
  let match;

  // Reset regex lastIndex
  TEMPLATE_VARIABLE_PATTERN.lastIndex = 0;

  while ((match = TEMPLATE_VARIABLE_PATTERN.exec(value)) !== null) {
    matches.push({
      fullMatch: match[0],
      variableName: match[1],
      index: match.index,
      isValid: Object.prototype.hasOwnProperty.call(TEMPLATE_VARIABLES, match[1]),
      isDeprecated: TEMPLATE_VARIABLES[match[1]]?.deprecated || false,
    });
  }

  return matches;
};

// Helper function to check if a string contains template variables
export const hasTemplateVariables = (value) => {
  if (!value || typeof value !== 'string') {
    return false;
  }

  TEMPLATE_VARIABLE_PATTERN.lastIndex = 0;
  return TEMPLATE_VARIABLE_PATTERN.test(value);
};

// Helper function to get variable info
export const getTemplateVariableInfo = (variableName) => {
  return TEMPLATE_VARIABLES[variableName] || null;
};

// Helper function to get all non-deprecated variables
export const getActiveTemplateVariables = () => {
  return Object.values(TEMPLATE_VARIABLES).filter(v => !v.deprecated);
};

// Helper function to format example with template
export const formatTemplateExample = (template, exampleValues = {}) => {
  let result = template;

  Object.entries(exampleValues).forEach(([varName, value]) => {
    result = result.replace(new RegExp(`\\{${varName}\\}`, 'g'), value);
  });

  return result;
};
