import { ThemeProvider } from "@mui/material/styles";
import { render, screen, waitFor } from "@testing-library/react";
// biome-ignore lint/correctness/noUnusedImports: React is required for JSX transform
import React from "react";
import { BrowserRouter } from "react-router-dom";
import { TourProvider } from "../../../contexts/TourContext";
import theme from "../../../Theme";
import AppGarden from "../AppGarden";

// Mock the api module used by AppGarden
jest.mock("../../../utils/api", () => ({
	__esModule: true,
	default: {
		get: jest.fn(),
		post: jest.fn(),
	},
}));

// Import the mocked api module
import api from "../../../utils/api";

// Mock child components to isolate AppGarden logic
jest.mock("../AppTemplateList", () => {
	return function MockAppTemplateList() {
		return <div data-testid="app-template-list">App Template List</div>;
	};
});

jest.mock("../AppDeploymentList", () => {
	return function MockAppDeploymentList() {
		return <div data-testid="app-deployment-list">App Deployment List</div>;
	};
});

const renderAppGarden = () =>
	render(
		<BrowserRouter>
			<ThemeProvider theme={theme}>
				<TourProvider>
					<AppGarden />
				</TourProvider>
			</ThemeProvider>
		</BrowserRouter>,
	);

describe("AppGarden", () => {
	beforeEach(() => {
		jest.clearAllMocks();
		// Default mock responses
		api.get.mockImplementation((url) => {
			if (url === "/apps/garden/status") {
				return Promise.resolve({
					data: { garden_apps_available: true, missing_count: 0 },
				});
			}
			if (url === "/serving/deployments") {
				return Promise.resolve({ data: [] });
			}
			return Promise.reject(new Error(`Unhandled GET: ${url}`));
		});
	});

	it("renders the App Garden page", async () => {
		renderAppGarden();

		await waitFor(() => {
			expect(screen.getByText("App Garden")).toBeInTheDocument();
		});
	});

	describe("Model Deployment Detection", () => {
		describe("shows warning when no models deployed", () => {
			it("displays warning when deployments array is empty", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({ data: [] });
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(
						screen.getByText(/No models are currently deployed/),
					).toBeInTheDocument();
				});
			});

			it("displays warning when deployment status is not DEPLOYED", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "STARTING",
									lb_port: 8080,
									instances: [{ status: "DEPLOYED" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(
						screen.getByText(/No models are currently deployed/),
					).toBeInTheDocument();
				});
			});

			it("displays warning when no instances have DEPLOYED status", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "DEPLOYED",
									lb_port: 8080,
									instances: [{ status: "STARTING" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(
						screen.getByText(/No models are currently deployed/),
					).toBeInTheDocument();
				});
			});

			it("displays warning when instances array is missing", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "DEPLOYED",
									lb_port: 8080,
									// no instances field
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(
						screen.getByText(/No models are currently deployed/),
					).toBeInTheDocument();
				});
			});
		});

		describe("hides warning when models are properly deployed", () => {
			it("hides warning when deployment has access_path (path routing)", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "DEPLOYED",
									access_path: "/runtime/models/dep-1",
									instances: [{ status: "DEPLOYED" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(screen.getByText("App Garden")).toBeInTheDocument();
				});

				// Warning should NOT be present
				expect(
					screen.queryByText(/No models are currently deployed/),
				).not.toBeInTheDocument();
			});

			it("hides warning when deployment has lb_port (port routing)", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "DEPLOYED",
									lb_port: 8080,
									// no access_path - port-only routing
									instances: [{ status: "DEPLOYED" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(screen.getByText("App Garden")).toBeInTheDocument();
				});

				// Warning should NOT be present - port routing is valid
				expect(
					screen.queryByText(/No models are currently deployed/),
				).not.toBeInTheDocument();
			});

			it("hides warning when deployment has both path and port (dual routing)", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "DEPLOYED",
									access_path: "/runtime/models/dep-1",
									lb_port: 8080,
									instances: [{ status: "DEPLOYED" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(screen.getByText("App Garden")).toBeInTheDocument();
				});

				expect(
					screen.queryByText(/No models are currently deployed/),
				).not.toBeInTheDocument();
			});
		});

		describe("edge cases for port detection", () => {
			it("treats port 443 as no dedicated port (shows warning)", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "DEPLOYED",
									lb_port: 443, // Default HTTPS port, not a dedicated deployment port
									// no access_path
									instances: [{ status: "DEPLOYED" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(
						screen.getByText(/No models are currently deployed/),
					).toBeInTheDocument();
				});
			});

			it("treats port 0 as no dedicated port (shows warning)", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "DEPLOYED",
									lb_port: 0,
									instances: [{ status: "DEPLOYED" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(
						screen.getByText(/No models are currently deployed/),
					).toBeInTheDocument();
				});
			});

			it("accepts high ports as valid dedicated ports", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "DEPLOYED",
									lb_port: 61234, // High port typical for dynamic allocation
									instances: [{ status: "DEPLOYED" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(screen.getByText("App Garden")).toBeInTheDocument();
				});

				expect(
					screen.queryByText(/No models are currently deployed/),
				).not.toBeInTheDocument();
			});
		});

		describe("error handling", () => {
			it("does not show warning when API call fails (avoids false warning)", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.reject(new Error("Network error"));
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(screen.getByText("App Garden")).toBeInTheDocument();
				});

				// On error, warning should NOT be shown (hasDeployedModels = null)
				expect(
					screen.queryByText(/No models are currently deployed/),
				).not.toBeInTheDocument();
			});
		});

		describe("multiple deployments", () => {
			it("hides warning if at least one deployment is valid", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "STOPPED",
									lb_port: 8080,
									instances: [],
								},
								{
									id: "dep-2",
									status: "DEPLOYED",
									lb_port: 8081,
									instances: [{ status: "DEPLOYED" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(screen.getByText("App Garden")).toBeInTheDocument();
				});

				expect(
					screen.queryByText(/No models are currently deployed/),
				).not.toBeInTheDocument();
			});

			it("shows warning if all deployments are invalid", async () => {
				api.get.mockImplementation((url) => {
					if (url === "/apps/garden/status") {
						return Promise.resolve({
							data: { garden_apps_available: true, missing_count: 0 },
						});
					}
					if (url === "/serving/deployments") {
						return Promise.resolve({
							data: [
								{
									id: "dep-1",
									status: "STOPPED",
									lb_port: 8080,
									instances: [],
								},
								{
									id: "dep-2",
									status: "FAILED",
									lb_port: 8081,
									instances: [{ status: "FAILED" }],
								},
							],
						});
					}
					return Promise.reject(new Error(`Unhandled GET: ${url}`));
				});

				renderAppGarden();

				await waitFor(() => {
					expect(
						screen.getByText(/No models are currently deployed/),
					).toBeInTheDocument();
				});
			});
		});
	});
});
