import React from 'react';
import { render, screen, fireEvent, waitFor } from '@testing-library/react';
import { ThemeProvider } from '@mui/material/styles';
import axios from 'axios';

import AppDeploymentList from '../AppDeploymentList';
import theme from '../../../Theme';
import { RoutingConfigContext } from '../../../context/RoutingConfigContext';

jest.mock('axios');

// Configurable routing mode for tests
let testRoutingMode = 'dual';

const getRoutingConfig = () => ({
  routingMode: testRoutingMode,
  rawRoutingMode: testRoutingMode,
  isLoading: false,
  error: null,
  isDualMode: testRoutingMode === 'dual',
  normalizedMode: testRoutingMode === 'path' ? 'path' : 'port',
});

const renderList = (props = {}) =>
  render(
    <ThemeProvider theme={theme}>
      <RoutingConfigContext.Provider value={getRoutingConfig()}>
        <AppDeploymentList refreshDeployments={() => {}} {...props} />
      </RoutingConfigContext.Provider>
    </ThemeProvider>
  );

describe('AppDeploymentList', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  it('surfaces failure details via tooltip for failed deployments', async () => {
    axios.get.mockResolvedValueOnce({
      data: [
        {
          id: 'deployment-1',
          name: 'failed-app',
          status: 'FAILED',
          last_error_code: 'APP_HTTP_PROBE_FAILED',
          last_error_message: 'HTTP probe failed at https://example.com',
          access_path_prefix: '/runtime/apps',
          access_path: '/runtime/apps/deployment-1',
          instances: [],
          starting_copies: 1,
          min_copies: 1,
          max_copies: 1,
        },
      ],
    });

    renderList();

    const statusChip = await screen.findByText('FAILED');
    fireEvent.mouseEnter(statusChip);

    await waitFor(() => {
      expect(
        screen.getByText('APP_HTTP_PROBE_FAILED: HTTP probe failed at https://example.com')
      ).toBeInTheDocument();
    });
  });

  it('displays runtime path and open link for deployed apps', async () => {
    axios.get.mockResolvedValueOnce({
      data: [
        {
          id: 'deployment-2',
          name: 'path-app',
          status: 'DEPLOYED',
          lb_port: 443,
          access_path_prefix: '/runtime/apps',
          access_path: '/runtime/apps/deployment-2',
          instances: [{ status: 'DEPLOYED' }],
          starting_copies: 1,
          runtime_artifacts: {
            resolved_env: {
              KAMIWAZA_APP_PATH: '/runtime/apps/deployment-2',
              KAMIWAZA_APP_PATH_URL: 'http://localhost/runtime/apps/deployment-2',
              KAMIWAZA_APP_URL: 'http://localhost:443',
              KAMIWAZA_APP_PORT: '443',
            },
          },
        },
      ],
    });

    renderList();

    await waitFor(() => {
      // Path-based routing shows the full URL
      expect(screen.getByText('http://localhost/runtime/apps/deployment-2')).toBeInTheDocument();
    });

    const link = screen.getByRole('link', { name: /open app/i });
    expect(link).toHaveAttribute('href', 'http://localhost/runtime/apps/deployment-2');
  });

  describe('Routing Mode Presentation', () => {
    afterEach(() => {
      // Reset to default routing mode after each test
      testRoutingMode = 'dual';
    });

    const createDeployment = (overrides = {}) => ({
      id: 'deployment-test',
      name: 'test-app',
      status: 'DEPLOYED',
      lb_port: 9090,
      access_path_prefix: '/runtime/apps',
      access_path: '/runtime/apps/deployment-test',
      instances: [{ status: 'DEPLOYED' }],
      starting_copies: 1,
      min_copies: 1,
      max_copies: 1,
      runtime_artifacts: {
        resolved_env: {
          KAMIWAZA_APP_PATH: '/runtime/apps/deployment-test',
          KAMIWAZA_APP_PATH_URL: 'https://localhost/runtime/apps/deployment-test',
          KAMIWAZA_APP_URL: 'https://localhost:9090',
          KAMIWAZA_APP_PORT: '9090',
        },
      },
      ...overrides,
    });

    describe('port mode', () => {
      beforeEach(() => {
        testRoutingMode = 'port';
      });

      it('displays port-based URL when in port mode', async () => {
        axios.get.mockResolvedValueOnce({ data: [createDeployment()] });

        renderList();

        await waitFor(() => {
          expect(screen.getByText('https://localhost:9090')).toBeInTheDocument();
        });
      });

      it('falls back to path URL when no port available in port mode', async () => {
        const deployment = createDeployment({
          lb_port: 443,
          runtime_artifacts: {
            resolved_env: {
              KAMIWAZA_APP_PATH: '/runtime/apps/deployment-test',
              KAMIWAZA_APP_PATH_URL: 'https://localhost/runtime/apps/deployment-test',
              KAMIWAZA_APP_URL: 'https://localhost:443',
              KAMIWAZA_APP_PORT: '443',
            },
          },
        });

        axios.get.mockResolvedValueOnce({ data: [deployment] });

        renderList();

        // When port is 443 (default), it should fall back to path
        await waitFor(() => {
          expect(screen.getByText('https://localhost/runtime/apps/deployment-test')).toBeInTheDocument();
        });
      });
    });

    describe('path mode', () => {
      beforeEach(() => {
        testRoutingMode = 'path';
      });

      it('displays path-based URL when in path mode', async () => {
        axios.get.mockResolvedValueOnce({ data: [createDeployment()] });

        renderList();

        await waitFor(() => {
          expect(screen.getByText('https://localhost/runtime/apps/deployment-test')).toBeInTheDocument();
        });
      });

      it('does not show port URL even when port is available', async () => {
        axios.get.mockResolvedValueOnce({ data: [createDeployment()] });

        renderList();

        await waitFor(() => {
          // Component transforms "test-app" to "Test App"
          expect(screen.getByText('Test App')).toBeInTheDocument();
        });

        // Should show path URL, not port URL
        expect(screen.getByText('https://localhost/runtime/apps/deployment-test')).toBeInTheDocument();
        expect(screen.queryByText('https://localhost:9090')).not.toBeInTheDocument();
      });
    });

    describe('dual mode', () => {
      beforeEach(() => {
        testRoutingMode = 'dual';
      });

      it('displays both port and path URLs when in dual mode', async () => {
        axios.get.mockResolvedValueOnce({ data: [createDeployment()] });

        renderList();

        await waitFor(() => {
          // Component transforms "test-app" to "Test App"
          expect(screen.getByText('Test App')).toBeInTheDocument();
        });

        // Should show both URLs
        expect(screen.getByText('https://localhost:9090')).toBeInTheDocument();
        expect(screen.getByText('https://localhost/runtime/apps/deployment-test')).toBeInTheDocument();
      });

      it('shows only path when port is default (443) in dual mode', async () => {
        const deployment = createDeployment({
          lb_port: 443,
          runtime_artifacts: {
            resolved_env: {
              KAMIWAZA_APP_PATH: '/runtime/apps/deployment-test',
              KAMIWAZA_APP_PATH_URL: 'https://localhost/runtime/apps/deployment-test',
              KAMIWAZA_APP_URL: 'https://localhost:443',
              KAMIWAZA_APP_PORT: '443',
            },
          },
        });

        axios.get.mockResolvedValueOnce({ data: [deployment] });

        renderList();

        await waitFor(() => {
          // Component transforms "test-app" to "Test App"
          expect(screen.getByText('Test App')).toBeInTheDocument();
        });

        // Should show only path URL since port 443 is default
        expect(screen.getByText('https://localhost/runtime/apps/deployment-test')).toBeInTheDocument();
        expect(screen.queryByText('https://localhost:443')).not.toBeInTheDocument();
      });
    });
  });
});
