import React from 'react';
import { render, screen, fireEvent, waitFor } from '@testing-library/react';
import { ThemeProvider } from '@mui/material/styles';
import axios from 'axios';
import AppDeploymentDetail from '../AppDeploymentDetail';
import theme from '../../../Theme';
import { RoutingConfigContext } from '../../../context/RoutingConfigContext';

// Mock axios
jest.mock('axios');

// Configurable routing mode for tests
let testRoutingMode = 'dual';

const getRoutingConfig = () => ({
  routingMode: testRoutingMode,
  rawRoutingMode: testRoutingMode,
  isLoading: false,
  error: null,
  isDualMode: testRoutingMode === 'dual',
  normalizedMode: testRoutingMode === 'path' ? 'path' : 'port',
});

describe('AppDeploymentDetail', () => {
  const mockOnClose = jest.fn();
  const mockOnDeploymentStopped = jest.fn();

  const mockDeploymentData = {
    id: 'test-app-id',
    name: 'Test App',
    status: 'DEPLOYED',
    created_at: new Date().toISOString(),
    deployed_at: new Date().toISOString(),
    lb_port: 61123,  // Dedicated port (not 443) for dual routing
    access_path_prefix: '/runtime/apps',
    access_path: '/runtime/apps/test-app-id',
    last_error_code: null,
    last_error_message: null,
    env_vars: {
      NODE_ENV: 'production',
    },
    runtime_artifacts: {
      env_entries: [
        { key: 'NODE_ENV', value: 'production', source: 'user' },
        { key: 'KAMIWAZA_APP_PORT', value: '61123', source: 'system' },
        { key: 'KAMIWAZA_APP_PATH', value: '/runtime/apps/test-app-id', source: 'system' },
        { key: 'KAMIWAZA_APP_URL', value: 'https://localhost:61123', source: 'system' },
        { key: 'KAMIWAZA_APP_PATH_URL', value: 'https://localhost/runtime/apps/test-app-id', source: 'system' },
      ],
    },
    volumes: [],
    instances: [],
  };

  beforeEach(() => {
    jest.clearAllMocks();
    axios.get.mockResolvedValue({ data: mockDeploymentData });
  });

  const renderComponent = (props = {}) => {
    const defaultProps = {
      deploymentId: 'test-deployment-id',
      isOpen: true,
      onClose: mockOnClose,
      onDeploymentStopped: mockOnDeploymentStopped,
      ...props,
    };

    return render(
      <ThemeProvider theme={theme}>
        <RoutingConfigContext.Provider value={getRoutingConfig()}>
          <AppDeploymentDetail {...defaultProps} />
        </RoutingConfigContext.Provider>
      </ThemeProvider>
    );
  };

  it('renders the dialog when open', async () => {
    renderComponent();

    await waitFor(() => {
      expect(screen.getByText('Deployment Details')).toBeInTheDocument();
    });
  });

  it('does not render when isOpen is false', () => {
    renderComponent({ isOpen: false });

    expect(screen.queryByText('Deployment Details')).not.toBeInTheDocument();
  });

  describe('Close button functionality', () => {
    it('renders close button with proper attributes', async () => {
      renderComponent();

      await waitFor(() => {
        const closeButtons = screen.getAllByRole('button');
        const closeButton = closeButtons.find(button =>
          button.querySelector('[data-testid="CloseIcon"]')
        );
        expect(closeButton).toBeInTheDocument();
        expect(closeButton).toHaveClass('MuiIconButton-sizeSmall');
      });
    });

    it('calls onClose when close button is clicked', async () => {
      renderComponent();

      await waitFor(() => {
        const closeButtons = screen.getAllByRole('button');
        const closeButton = closeButtons.find(button =>
          button.querySelector('[data-testid="CloseIcon"]')
        );
        fireEvent.click(closeButton);
        expect(mockOnClose).toHaveBeenCalledTimes(1);
      });
    });

    it('close button is positioned correctly next to refresh button', async () => {
      renderComponent();

      await waitFor(() => {
        const refreshButton = screen
          .getAllByRole('button')
          .find(button => button.querySelector('[data-testid="RefreshIcon"]'));
        const closeButton = screen
          .getAllByRole('button')
          .find(button => button.querySelector('[data-testid="CloseIcon"]'));

        // Both buttons should be in the same container
        expect(refreshButton.parentElement).toBe(closeButton.parentElement);
      });
    });

    it('both icon buttons have correct size attribute', async () => {
      renderComponent();

      await waitFor(() => {
        const iconButtons = screen
          .getAllByRole('button')
          .filter(
            button =>
              button.querySelector('[data-testid="RefreshIcon"]') ||
              button.querySelector('[data-testid="CloseIcon"]')
          );

        iconButtons.forEach(button => {
          expect(button.className).toMatch(/MuiIconButton-sizeSmall/);
        });
      });
    });
  });

  it('shows loading spinner while fetching data', () => {
    axios.get.mockReturnValueOnce(new Promise(() => {})); // Never resolves
    renderComponent();

    expect(screen.getByRole('progressbar')).toBeInTheDocument();
  });

  it('displays deployment details when data is loaded', async () => {
    renderComponent();

    await waitFor(() => {
      expect(screen.getByText('Test App')).toBeInTheDocument();
      expect(screen.getByText('Running')).toBeInTheDocument(); // Status is shown as "Running" for DEPLOYED
    });

    expect(screen.getByText('User Variables')).toBeInTheDocument();
    expect(screen.getByText('System Variables')).toBeInTheDocument();
    expect(screen.getByText('NODE_ENV')).toBeInTheDocument();
    expect(screen.getByText('production')).toBeInTheDocument();
    expect(screen.getByText('KAMIWAZA_APP_PORT')).toBeInTheDocument();
    expect(screen.getAllByText('61123').length).toBeGreaterThan(0);
    expect(screen.getByText('Path URL')).toBeInTheDocument();
    expect(screen.getAllByText('/runtime/apps/test-app-id').length).toBeGreaterThan(0);
  });

  it('handles error state gracefully', async () => {
    const errorMessage = 'Failed to load deployment details';
    axios.get.mockRejectedValueOnce({
      response: { data: { detail: errorMessage } },
    });

    renderComponent();

    await waitFor(() => {
      expect(screen.getByText(errorMessage)).toBeInTheDocument();
    });
  });

  it('refresh button triggers data refetch', async () => {
    renderComponent();

    await waitFor(() => {
      const refreshButton = screen
        .getAllByRole('button')
        .find(button => button.querySelector('[data-testid="RefreshIcon"]'));

      // Initial call
      expect(axios.get).toHaveBeenCalledTimes(1);

      // Click refresh
      fireEvent.click(refreshButton);

      // Should make another call
      expect(axios.get).toHaveBeenCalledTimes(2);
    });
  });

  it('displays failure alert when deployment is failed', async () => {
    const failureMessage = 'HTTP probe failed at https://example.com';
    axios.get.mockResolvedValueOnce({
      data: {
        ...mockDeploymentData,
        status: 'FAILED',
        last_error_code: 'APP_HTTP_PROBE_FAILED',
        last_error_message: failureMessage,
      },
    });

    renderComponent();

    await waitFor(() => {
      expect(screen.getByText(failureMessage)).toBeInTheDocument();
    });

    const alert = screen.getByRole('alert');
    expect(alert).toHaveTextContent('APP_HTTP_PROBE_FAILED');
  });

  describe('Routing Mode Presentation', () => {
    afterEach(() => {
      // Reset to default routing mode after each test
      testRoutingMode = 'dual';
    });

    const createDeployment = (overrides = {}) => ({
      ...mockDeploymentData,
      ...overrides,
    });

    describe('port mode', () => {
      beforeEach(() => {
        testRoutingMode = 'port';
      });

      it('displays port-based URL when in port mode', async () => {
        axios.get.mockResolvedValueOnce({ data: createDeployment() });

        renderComponent();

        await waitFor(() => {
          expect(screen.getByText('Deployment Details')).toBeInTheDocument();
        });

        // Should show port URL in the details
        expect(screen.getByRole('link', { name: 'https://localhost:61123' })).toBeInTheDocument();
      });

      it('falls back to path URL when no port available in port mode', async () => {
        const deployment = createDeployment({
          lb_port: 443,
          runtime_artifacts: {
            env_entries: [
              { key: 'KAMIWAZA_APP_PORT', value: '443', source: 'system' },
              { key: 'KAMIWAZA_APP_PATH', value: '/runtime/apps/test-app-id', source: 'system' },
              { key: 'KAMIWAZA_APP_URL', value: 'https://localhost:443', source: 'system' },
              { key: 'KAMIWAZA_APP_PATH_URL', value: 'https://localhost/runtime/apps/test-app-id', source: 'system' },
            ],
          },
        });

        axios.get.mockResolvedValueOnce({ data: deployment });

        renderComponent();

        await waitFor(() => {
          expect(screen.getByText('Deployment Details')).toBeInTheDocument();
        });

        // When port is 443 (default), it should fall back to path
        expect(screen.getByRole('link', { name: 'https://localhost/runtime/apps/test-app-id' })).toBeInTheDocument();
      });
    });

    describe('path mode', () => {
      beforeEach(() => {
        testRoutingMode = 'path';
      });

      it('displays path-based URL when in path mode', async () => {
        axios.get.mockResolvedValueOnce({ data: createDeployment() });

        renderComponent();

        await waitFor(() => {
          expect(screen.getByText('Deployment Details')).toBeInTheDocument();
        });

        // Should show path URL
        expect(screen.getByRole('link', { name: 'https://localhost/runtime/apps/test-app-id' })).toBeInTheDocument();
      });

      it('does not show port URL even when port is available', async () => {
        axios.get.mockResolvedValueOnce({ data: createDeployment() });

        renderComponent();

        await waitFor(() => {
          expect(screen.getByText('Deployment Details')).toBeInTheDocument();
        });

        // Should show path URL, not port URL
        expect(screen.getByRole('link', { name: 'https://localhost/runtime/apps/test-app-id' })).toBeInTheDocument();
        expect(screen.queryByRole('link', { name: 'https://localhost:61123' })).not.toBeInTheDocument();
      });
    });

    describe('dual mode', () => {
      beforeEach(() => {
        testRoutingMode = 'dual';
      });

      it('displays both port and path URLs when in dual mode', async () => {
        axios.get.mockResolvedValueOnce({ data: createDeployment() });

        renderComponent();

        await waitFor(() => {
          expect(screen.getByText('Deployment Details')).toBeInTheDocument();
        });

        // Should show both URLs
        expect(screen.getByRole('link', { name: 'https://localhost:61123' })).toBeInTheDocument();
        expect(screen.getByRole('link', { name: 'https://localhost/runtime/apps/test-app-id' })).toBeInTheDocument();
      });

      it('displays Port URL and Path URL labels when both routes available', async () => {
        axios.get.mockResolvedValueOnce({ data: createDeployment() });

        renderComponent();

        await waitFor(() => {
          expect(screen.getByText('Deployment Details')).toBeInTheDocument();
        });

        // Should show labels for each route type
        expect(screen.getByText('Port URL')).toBeInTheDocument();
        expect(screen.getByText('Path URL')).toBeInTheDocument();
      });

      it('shows only path when port is default (443) in dual mode', async () => {
        const deployment = createDeployment({
          lb_port: 443,
          runtime_artifacts: {
            env_entries: [
              { key: 'KAMIWAZA_APP_PORT', value: '443', source: 'system' },
              { key: 'KAMIWAZA_APP_PATH', value: '/runtime/apps/test-app-id', source: 'system' },
              { key: 'KAMIWAZA_APP_URL', value: 'https://localhost:443', source: 'system' },
              { key: 'KAMIWAZA_APP_PATH_URL', value: 'https://localhost/runtime/apps/test-app-id', source: 'system' },
            ],
          },
        });

        axios.get.mockResolvedValueOnce({ data: deployment });

        renderComponent();

        await waitFor(() => {
          expect(screen.getByText('Deployment Details')).toBeInTheDocument();
        });

        // Should show only path URL since port 443 is default
        expect(screen.getByRole('link', { name: 'https://localhost/runtime/apps/test-app-id' })).toBeInTheDocument();
        // Port URL label should not be present since port is not available
        expect(screen.queryByText('Port URL')).not.toBeInTheDocument();
      });
    });
  });
});
