import React from 'react';
import { render, screen, fireEvent, waitFor, within } from '@testing-library/react';
import { ThemeProvider } from '@mui/material/styles';
import axios from 'axios';

import AppDeployModal from '../AppDeployModal';
import theme from '../../../Theme';
import { SYSTEM_VARIABLES } from '../constants';

jest.mock('axios');

const baseTemplate = {
  id: 'template-id',
  name: 'test-template',
  env_defaults: {
    FOO: 'bar',
    OPENAI_BASE_URI: 'https://host.docker.internal/runtime/models/{deployment_id}/v1',
    NODE_TLS_REJECT_UNAUTHORIZED: '0',
  },
  preferred_model_type: 'any',
  fail_if_model_type_unavailable: false,
  preferred_model_name: null,
  fail_if_model_name_unavailable: false,
  compose_yml: 'version: "3"\nservices:\n  web:\n    image: test\n    ports:\n      - "3000"\n',
};

const renderModal = (props = {}) => {
  const defaultProps = {
    template: baseTemplate,
    open: true,
    onClose: jest.fn(),
    onSuccess: jest.fn(),
    ...props,
  };

  return render(
    <ThemeProvider theme={theme}>
      <AppDeployModal {...defaultProps} />
    </ThemeProvider>
  );
};

describe('AppDeployModal', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  it('renders the system variable reference list', async () => {
    renderModal();

    await waitFor(() => {
      expect(
        screen.getByRole('table', { name: /system variables/i })
      ).toBeInTheDocument();
    });

    SYSTEM_VARIABLES.forEach(({ key }) => {
      const keyCell = screen.getByText(key);
      expect(keyCell).toBeInTheDocument();

      const row = keyCell.closest('tr');
      expect(row).not.toBeNull();

      const cells = within(row).getAllByRole('cell');
      expect(cells.length).toBeGreaterThanOrEqual(2);
      const valueCellText = cells[1].textContent?.trim() ?? '';
      expect(valueCellText).not.toHaveLength(0);
    });
  });

  it('surfaces backend validation errors to the user', async () => {
    const errorDetail = 'Validation failed: unresolved template variable {model_port}';
    axios.post.mockRejectedValueOnce({
      response: { data: { detail: errorDetail } },
    });

    renderModal();

    fireEvent.click(screen.getByRole('button', { name: 'Deploy' }));

    await waitFor(() => {
      expect(screen.getByText(errorDetail)).toBeInTheDocument();
    });

    expect(axios.post).toHaveBeenCalledTimes(1);
    const [, payload] = axios.post.mock.calls[0];
    expect(payload.env_vars).toEqual({
      FOO: 'bar',
      NODE_TLS_REJECT_UNAUTHORIZED: '0',
    });
    expect(payload.env_vars).not.toHaveProperty('OPENAI_BASE_URI');
    expect(payload.env_vars).toHaveProperty('NODE_TLS_REJECT_UNAUTHORIZED', '0');
    expect(payload.is_ephemeral_session).toBe(false);
  });
});
