import React from 'react';
import { Box, Chip, Typography } from '@mui/material';
import { styled } from '@mui/material/styles';
import { detectTemplateVariables, getTemplateVariableInfo } from './constants';

// Styled components
const HighlightedText = styled('span')(({ theme, deprecated }) => ({
  backgroundColor: deprecated
    ? 'rgba(255, 152, 0, 0.2)'
    : 'rgba(0, 192, 127, 0.2)',
  color: deprecated
    ? theme.palette.warning.main
    : theme.palette.primary.main,
  padding: '2px 4px',
  borderRadius: 4,
  fontFamily: 'monospace',
  fontSize: '0.9em',
  border: `1px solid ${deprecated
    ? theme.palette.warning.main
    : theme.palette.primary.main}`,
  cursor: 'help',
}));

const VariableChip = styled(Chip)(({ theme, deprecated }) => ({
  height: 20,
  fontSize: '0.75rem',
  marginLeft: theme.spacing(0.5),
  backgroundColor: deprecated
    ? theme.palette.warning.main
    : theme.palette.primary.main,
  color: theme.palette.common.white,
  '& .MuiChip-label': {
    padding: '0 6px',
  },
}));

const InvalidVariable = styled('span')(({ theme }) => ({
  backgroundColor: 'rgba(244, 67, 54, 0.1)',
  color: theme.palette.error.main,
  padding: '2px 4px',
  borderRadius: 4,
  fontFamily: 'monospace',
  fontSize: '0.9em',
  border: `1px dashed ${theme.palette.error.main}`,
  cursor: 'help',
}));

// Component to highlight template variables in text
export const TemplateVariableHighlight = ({ text, showChips = false }) => {
  if (!text || typeof text !== 'string') {
    return <>{text}</>;
  }

  const variables = detectTemplateVariables(text);

  if (variables.length === 0) {
    return <>{text}</>;
  }

  // Build the highlighted text
  const parts = [];
  let lastIndex = 0;

  variables.forEach((variable) => {
    // Add text before the variable
    if (variable.index > lastIndex) {
      parts.push(text.substring(lastIndex, variable.index));
    }

    // Add the highlighted variable
    const varInfo = getTemplateVariableInfo(variable.variableName);

    if (variable.isValid) {
      parts.push(
        <HighlightedText
          key={`${variable.index}-${variable.variableName}`}
          deprecated={variable.isDeprecated}
          title={varInfo ? varInfo.description : variable.variableName}
        >
          {variable.fullMatch}
        </HighlightedText>
      );

      if (showChips && varInfo) {
        parts.push(
          <VariableChip
            key={`chip-${variable.index}-${variable.variableName}`}
            label={varInfo.deprecated ? `${varInfo.name} (deprecated)` : varInfo.name}
            size="small"
            deprecated={varInfo.deprecated}
          />
        );
      }
    } else {
      // Invalid variable
      parts.push(
        <InvalidVariable
          key={`${variable.index}-${variable.variableName}`}
          title={`Unknown template variable: ${variable.variableName}`}
        >
          {variable.fullMatch}
        </InvalidVariable>
      );
    }

    lastIndex = variable.index + variable.fullMatch.length;
  });

  // Add remaining text
  if (lastIndex < text.length) {
    parts.push(text.substring(lastIndex));
  }

  return <>{parts}</>;
};

// Component to show a preview of how the value will look after substitution
export const TemplatePreview = ({ value, exampleValues = {} }) => {
  if (!value || typeof value !== 'string') {
    return null;
  }

  const variables = detectTemplateVariables(value);
  if (variables.length === 0) {
    return null;
  }

  // Create example substitution
  let preview = value;
  const defaults = {
    app_port: '443',
    lb_port: '443',
    deployment_id: '09c6ce9a-0ce6-415c-922e-d8de686e1b7d',
    app_name: 'my-app',
    model_port: '443',
    app_access_path: '/runtime/apps/09c6ce9a-0ce6-415c-922e-d8de686e1b7d',
    app_path_url: 'https://localhost/runtime/apps/09c6ce9a-0ce6-415c-922e-d8de686e1b7d',
    model_access_path: '/runtime/models/09c6ce9a-0ce6-415c-922e-d8de686e1b7d',
    model_path_url: 'https://localhost/runtime/models/09c6ce9a-0ce6-415c-922e-d8de686e1b7d',
    openai_base_url: 'https://host.docker.internal:61111',
    openai_path_base_url: 'https://localhost/runtime/models/09c6ce9a-0ce6-415c-922e-d8de686e1b7d',
  };

  const substitutionValues = { ...defaults, ...exampleValues };

  variables.forEach((variable) => {
    if (variable.isValid && substitutionValues[variable.variableName]) {
      preview = preview.replace(
        variable.fullMatch,
        substitutionValues[variable.variableName]
      );
    }
  });

  return (
    <Box sx={{ mt: 1, p: 1, backgroundColor: 'rgba(0, 0, 0, 0.05)', borderRadius: 1 }}>
      <Typography variant="caption" color="textSecondary" display="block" gutterBottom>
        Preview after substitution:
      </Typography>
      <Typography variant="body2" fontFamily="monospace" color="primary">
        {preview}
      </Typography>
    </Box>
  );
};

// Helper component to display template variable info inline
export const TemplateVariableInfo = ({ variableName }) => {
  const info = getTemplateVariableInfo(variableName);

  if (!info) {
    return null;
  }

  return (
    <Box sx={{ display: 'inline-flex', alignItems: 'center', gap: 1 }}>
      <HighlightedText deprecated={info.deprecated}>
        {`{${info.name}}`}
      </HighlightedText>
      <Typography variant="caption" color="textSecondary">
        {info.description}
      </Typography>
      {info.deprecated && (
        <Typography variant="caption" color="warning.main">
          (Deprecated: {info.deprecationMessage})
        </Typography>
      )}
    </Box>
  );
};
