import React, { useState } from 'react';
import {
  IconButton,
  Popover,
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Chip,
  Divider,
  Link,
  Alert,
} from '@mui/material';
import { styled } from '@mui/material/styles';
import HelpOutlineIcon from '@mui/icons-material/HelpOutline';
import InfoIcon from '@mui/icons-material/Info';
import ContentCopyIcon from '@mui/icons-material/ContentCopy';
import {
  getActiveTemplateVariables,
  TEMPLATE_VARIABLES,
  SYSTEM_VARIABLES,
} from './constants';

// Styled components
const StyledPopover = styled(Popover)(({ theme }) => ({
  '& .MuiPopover-paper': {
    borderRadius: 12,
    boxShadow: theme.shadows[8],
    maxWidth: 600,
    backgroundColor: theme.palette.background.paper,
  },
}));

const StyledTableCell = styled(TableCell)(({ theme }) => ({
  fontSize: '0.875rem',
  fontFamily: 'monospace',
  padding: theme.spacing(1.5),
}));

const VariableCode = styled('code')(({ theme }) => ({
  backgroundColor: 'rgba(0, 192, 127, 0.1)',
  color: theme.palette.primary.main,
  padding: '2px 6px',
  borderRadius: 4,
  fontFamily: 'monospace',
  fontSize: '0.875rem',
  cursor: 'pointer',
  '&:hover': {
    backgroundColor: 'rgba(0, 192, 127, 0.2)',
  },
}));

const DeprecatedChip = styled(Chip)(({ theme }) => ({
  height: 20,
  fontSize: '0.7rem',
  marginLeft: theme.spacing(1),
}));

const ExampleBox = styled(Box)(({ theme }) => ({
  backgroundColor: theme.palette.grey[50],
  padding: theme.spacing(2),
  borderRadius: 8,
  marginTop: theme.spacing(2),
  fontFamily: 'monospace',
  fontSize: '0.875rem',
}));

// Template Variable Help Tooltip Component
export const TemplateVariableHelp = ({ anchorEl, children }) => {
  const [popoverAnchor, setPopoverAnchor] = useState(null);
  const [copiedVar, setCopiedVar] = useState(null);

  const handleClick = (event) => {
    setPopoverAnchor(event.currentTarget);
  };

  const handleClose = () => {
    setPopoverAnchor(null);
    setCopiedVar(null);
  };

  const handleCopyVariable = async (varName) => {
    try {
      await navigator.clipboard.writeText(`{${varName}}`);
      setCopiedVar(varName);
      setTimeout(() => setCopiedVar(null), 2000);
    } catch (err) {
      console.error('Failed to copy:', err);
    }
  };

  const open = Boolean(popoverAnchor);

  const triggerElement = anchorEl || (
    <IconButton
      size="small"
      onClick={handleClick}
      color="primary"
      sx={{ ml: 1 }}
      title="Template Variables Help"
    >
      <HelpOutlineIcon fontSize="small" />
    </IconButton>
  );

  return (
    <>
      {React.cloneElement(triggerElement, { onClick: handleClick })}
      <StyledPopover
        open={open}
        anchorEl={popoverAnchor}
        onClose={handleClose}
        anchorOrigin={{
          vertical: 'bottom',
          horizontal: 'left',
        }}
        transformOrigin={{
          vertical: 'top',
          horizontal: 'left',
        }}
      >
        <Box sx={{ p: 3, maxHeight: '70vh', overflow: 'auto' }}>
          <Typography variant="h6" gutterBottom>
            Template Variables
          </Typography>
          <Typography variant="body2" color="textSecondary" paragraph>
            Use template variables in environment values to dynamically reference runtime values.
          </Typography>

          <TableContainer component={Paper} variant="outlined" sx={{ mb: 3 }}>
            <Table size="small">
              <TableHead>
                <TableRow>
                  <TableCell>Variable</TableCell>
                  <TableCell>Description</TableCell>
                  <TableCell>Example</TableCell>
                  <TableCell align="center">Action</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {Object.values(TEMPLATE_VARIABLES).map((variable) => (
                  <TableRow key={variable.name}>
                    <StyledTableCell>
                      <Box display="flex" alignItems="center">
                        <VariableCode onClick={() => handleCopyVariable(variable.name)}>
                          {`{${variable.name}}`}
                        </VariableCode>
                        {variable.deprecated && (
                          <DeprecatedChip
                            label="Deprecated"
                            size="small"
                            color="warning"
                            variant="outlined"
                          />
                        )}
                      </Box>
                    </StyledTableCell>
                    <TableCell>
                      <Typography variant="caption">
                        {variable.description}
                        {variable.deprecationMessage && (
                          <Typography variant="caption" color="warning.main" display="block">
                            {variable.deprecationMessage}
                          </Typography>
                        )}
                        {variable.availability && (
                          <Typography variant="caption" color="info.main" display="block">
                            {variable.availability}
                          </Typography>
                        )}
                      </Typography>
                    </TableCell>
                    <TableCell>
                      <Typography variant="caption" color="textSecondary">
                        {variable.example}
                      </Typography>
                    </TableCell>
                    <TableCell align="center">
                      <IconButton
                        size="small"
                        onClick={() => handleCopyVariable(variable.name)}
                        color={copiedVar === variable.name ? 'success' : 'default'}
                      >
                        <ContentCopyIcon fontSize="small" />
                      </IconButton>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </TableContainer>

          <Divider sx={{ my: 2 }} />

          <Typography variant="subtitle2" gutterBottom>
            System Variables (Automatically Injected)
          </Typography>
          <Typography variant="body2" color="textSecondary" paragraph>
            These read-only variables are automatically available in all deployments:
          </Typography>

          <Box sx={{ mb: 3 }}>
            {SYSTEM_VARIABLES.map((variable) => (
              <Chip
                key={variable.key}
                label={variable.key}
                size="small"
                variant="outlined"
                sx={{ m: 0.5, fontFamily: 'monospace' }}
              />
            ))}
          </Box>

          <Alert severity="info" icon={<InfoIcon />} sx={{ mb: 2 }}>
            Template substitution occurs during deployment. Unknown variables are preserved as-is.
          </Alert>

          <Typography variant="subtitle2" gutterBottom>
            Examples
          </Typography>

          <ExampleBox>
            <Typography variant="caption" color="textSecondary" display="block" gutterBottom>
              Environment Variable Configuration:
            </Typography>
            <Box component="pre" m={0}>
              {`CALLBACK_URL: "https://localhost/runtime/apps/{deployment_id}/callback"
MODEL_API: "{openai_base_url}/chat/completions"
APP_ID: "{deployment_id}"
SERVICE_NAME: "{app_name}-service"`}
            </Box>
          </ExampleBox>

          <ExampleBox>
            <Typography variant="caption" color="textSecondary" display="block" gutterBottom>
              After Deployment (Example):
            </Typography>
            <Box component="pre" m={0}>
              {`CALLBACK_URL: "https://localhost/runtime/apps/09c6ce9a-0ce6-415c-922e-d8de686e1b7d/callback"
MODEL_API: "https://localhost/runtime/models/09c6ce9a-0ce6-415c-922e-d8de686e1b7d/v1/chat/completions"
APP_ID: "09c6ce9a-0ce6-415c-922e-d8de686e1b7d"
SERVICE_NAME: "my-app-service"`}
            </Box>
          </ExampleBox>

          <Box sx={{ mt: 2, display: 'flex', justifyContent: 'space-between' }}>
            <Link
              href="/docs/template-variables"
              target="_blank"
              variant="caption"
              sx={{ textDecoration: 'none' }}
            >
              View Full Documentation
            </Link>
            <Typography variant="caption" color="textSecondary">
              Press ESC to close
            </Typography>
          </Box>
        </Box>
      </StyledPopover>
    </>
  );
};

// Compact inline help icon
export const TemplateVariableHelpIcon = ({ size = 'small' }) => {
  return <TemplateVariableHelp />;
};
