import React, { useState, useEffect } from 'react';
import {
  TextField,
  Box,
  InputAdornment,
  Tooltip,
  IconButton,
  Typography,
  Chip,
} from '@mui/material';
import { styled } from '@mui/material/styles';
import VisibilityIcon from '@mui/icons-material/Visibility';
import CheckCircleIcon from '@mui/icons-material/CheckCircle';
import WarningIcon from '@mui/icons-material/Warning';
import {
  hasTemplateVariables,
  detectTemplateVariables,
  getTemplateVariableInfo
} from './constants';
import { TemplateVariableHighlight, TemplatePreview } from './TemplateVariableHighlight';
import { TemplateVariableHelpIcon } from './TemplateVariableHelp';

// Styled components
const StyledTextField = styled(TextField)(({ theme, hasTemplates }) => ({
  '& .MuiOutlinedInput-root': {
    backgroundColor: hasTemplates
      ? 'rgba(0, 192, 127, 0.03)'
      : 'transparent',
    '& fieldset': {
      borderColor: hasTemplates
        ? theme.palette.primary.main
        : theme.palette.divider,
    },
  },
}));

const PreviewBox = styled(Box)(({ theme }) => ({
  marginTop: theme.spacing(0.5),
  padding: theme.spacing(1),
  backgroundColor: 'rgba(0, 0, 0, 0.03)',
  borderRadius: 4,
  border: '1px solid rgba(0, 0, 0, 0.1)',
}));

const TemplateChip = styled(Chip)(({ theme }) => ({
  height: 22,
  fontSize: '0.75rem',
  marginRight: theme.spacing(0.5),
}));

// Environment Variable Input Component
export const EnvVarInput = ({
  label,
  value,
  onChange,
  onBlur,
  disabled,
  error,
  helperText,
  size = 'small',
  fullWidth = true,
  showPreview = true,
  showHelp = true,
  type = 'text',
  ...otherProps
}) => {
  const [showPreviewState, setShowPreviewState] = useState(false);
  const [templateInfo, setTemplateInfo] = useState({
    hasTemplates: false,
    variables: [],
    hasDeprecated: false,
    hasInvalid: false,
  });

  useEffect(() => {
    if (type === 'value' && value) {
      const variables = detectTemplateVariables(value);
      const hasTemplates = variables.length > 0;
      const hasDeprecated = variables.some(v => v.isDeprecated);
      const hasInvalid = variables.some(v => !v.isValid);

      setTemplateInfo({
        hasTemplates,
        variables,
        hasDeprecated,
        hasInvalid,
      });
    } else {
      setTemplateInfo({
        hasTemplates: false,
        variables: [],
        hasDeprecated: false,
        hasInvalid: false,
      });
    }
  }, [value, type]);

  const getEndAdornment = () => {
    const adornments = [];

    // Template help icon
    if (showHelp && type === 'value') {
      adornments.push(
        <TemplateVariableHelpIcon key="help" size="small" />
      );
    }

    // Template status indicators
    if (templateInfo.hasTemplates) {
      if (templateInfo.hasInvalid) {
        adornments.push(
          <Tooltip key="invalid" title="Contains unknown template variables">
            <WarningIcon color="error" fontSize="small" />
          </Tooltip>
        );
      } else if (templateInfo.hasDeprecated) {
        adornments.push(
          <Tooltip key="deprecated" title="Contains deprecated template variables">
            <WarningIcon color="warning" fontSize="small" />
          </Tooltip>
        );
      } else {
        adornments.push(
          <Tooltip key="valid" title="Contains valid template variables">
            <CheckCircleIcon color="success" fontSize="small" />
          </Tooltip>
        );
      }

      // Preview toggle
      if (showPreview) {
        adornments.push(
          <Tooltip key="preview" title={showPreviewState ? "Hide preview" : "Show preview"}>
            <IconButton
              size="small"
              onClick={() => setShowPreviewState(!showPreviewState)}
              color={showPreviewState ? "primary" : "default"}
            >
              <VisibilityIcon fontSize="small" />
            </IconButton>
          </Tooltip>
        );
      }
    }

    return adornments.length > 0 ? (
      <InputAdornment position="end">
        <Box display="flex" alignItems="center" gap={0.5}>
          {adornments}
        </Box>
      </InputAdornment>
    ) : null;
  };

  const getHelperTextContent = () => {
    const messages = [];

    if (helperText) {
      messages.push(helperText);
    }

    if (templateInfo.hasTemplates && !error) {
      const varNames = templateInfo.variables
        .filter(v => v.isValid)
        .map(v => v.variableName);

      if (varNames.length > 0) {
        messages.push(
          <Box key="templates" component="span" sx={{ display: 'inline-flex', alignItems: 'center', gap: 0.5 }}>
            <Typography variant="caption" component="span">
              Using templates:
            </Typography>
            {varNames.map(name => (
              <TemplateChip
                key={name}
                label={`{${name}}`}
                size="small"
                variant="outlined"
                color={getTemplateVariableInfo(name)?.deprecated ? "warning" : "primary"}
              />
            ))}
          </Box>
        );
      }
    }

    return messages.length > 0 ? (
      <Box display="flex" flexDirection="column" gap={0.5}>
        {messages}
      </Box>
    ) : null;
  };

  return (
    <Box>
      <StyledTextField
        label={label}
        value={value}
        onChange={onChange}
        onBlur={onBlur}
        disabled={disabled}
        error={error || templateInfo.hasInvalid}
        size={size}
        fullWidth={fullWidth}
        hasTemplates={templateInfo.hasTemplates}
        InputProps={{
          endAdornment: getEndAdornment(),
        }}
        helperText={getHelperTextContent()}
        {...otherProps}
      />

      {showPreviewState && templateInfo.hasTemplates && (
        <PreviewBox>
          <TemplatePreview value={value} />
        </PreviewBox>
      )}
    </Box>
  );
};

// Simplified version for key input (no template support)
export const EnvVarKeyInput = (props) => {
  return <EnvVarInput {...props} type="key" showPreview={false} showHelp={false} />;
};

// Enhanced version for value input (with template support)
export const EnvVarValueInput = (props) => {
  return <EnvVarInput {...props} type="value" />;
};