import React, { useState, useEffect } from 'react';
import { styled } from '@mui/material/styles';
import { Typography, Box, Grid, Tooltip, Alert } from '@mui/material';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import axios from 'axios';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import AppTemplateCard from './AppTemplateCard';
import AppDeployModal from './AppDeployModal';
import { BASE_URL } from '../../const';

// Styled components
const SubTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.5rem',
  fontWeight: 600,
  marginTop: theme.spacing(4),
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '2px',
    background: 'linear-gradient(90deg, #00c07f, transparent)',
  },
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const AppTemplateList = ({ onDeploymentChange, refreshTemplates, ephemeralDefault = false }) => {
  const [templates, setTemplates] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [selectedTemplate, setSelectedTemplate] = useState(null);
  const [showDeployModal, setShowDeployModal] = useState(false);

  useEffect(() => {
    // If we have new templates from import, add them directly
    if (refreshTemplates?.newTemplates && refreshTemplates.newTemplates.length > 0) {
      setTemplates(prevTemplates => {
        // Add new templates to existing ones, avoiding duplicates
        const existingIds = new Set(prevTemplates.map(t => t.id));
        const newUniqueTemplates = refreshTemplates.newTemplates.filter(t => !existingIds.has(t.id));
        return [...prevTemplates, ...newUniqueTemplates];
      });
    } else {
      // Fallback to full fetch if no new templates provided or on initial load
      fetchTemplates();
    }
  }, [refreshTemplates]);

  const fetchTemplates = async () => {
    setLoading(true);
    try {
      const response = await axios.get(`${BASE_URL}/apps/app_templates`);
      setTemplates(response.data);
    } catch (err) {
      console.error('Error fetching app templates:', err);
      setError(err);
    } finally {
      setLoading(false);
    }
  };

  const handleDeployClick = template => {
    setSelectedTemplate(template);
    setShowDeployModal(true);
  };

  const handleDeploySuccess = () => {
    setShowDeployModal(false);
    setSelectedTemplate(null);
    if (onDeploymentChange) {
      onDeploymentChange();
    }
  };

  if (loading) return <Spinner />;
  if (error) return <ErrorComponent message={error?.message || 'Failed to load app templates'} />;

  return (
    <Box sx={{ mb: 6 }} data-tour="app-gallery">
      <Box sx={{ display: 'flex', alignItems: 'center', mb: 3 }} data-tour="risk-tiers">
        <SubTitle variant="h5">
          Available Applications
          <Tooltip title="Pre-configured applications ready to deploy">
            <InfoIcon />
          </Tooltip>
        </SubTitle>
      </Box>

      {templates.length === 0 ? (
        <Alert severity="info" sx={{ maxWidth: 600 }}>
          No app templates available yet. Check back soon for exciting applications!
        </Alert>
      ) : (
        <Grid container spacing={3}>
          {templates.map((template, index) => (
            <Grid item xs={12} sm={6} md={4} lg={3} key={template.id}>
              <AppTemplateCard
                template={template}
                onDeploy={() => handleDeployClick(template)}
                data-tour={index === 0 ? 'app-card' : undefined}
                isFirstCard={index === 0}
              />
            </Grid>
          ))}
        </Grid>
      )}

      {showDeployModal && selectedTemplate && (
        <AppDeployModal
          template={selectedTemplate}
          open={showDeployModal}
          onClose={() => setShowDeployModal(false)}
          onSuccess={handleDeploySuccess}
          ephemeralDefault={ephemeralDefault}
        />
      )}
    </Box>
  );
};

export default AppTemplateList;
