import React, { useState, useEffect } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import {
  Card,
  CardContent,
  CardMedia,
  CardActions,
  Typography,
  Button,
  Box,
  Chip,
  IconButton,
  Tooltip,
  CircularProgress,
  Menu,
  MenuItem,
  ListItemIcon,
  ListItemText,
  Badge,
} from '@mui/material';
import RocketLaunchIcon from '@mui/icons-material/RocketLaunch';
import DownloadIcon from '@mui/icons-material/Download';
import PreviewIcon from '@mui/icons-material/Preview';
import AppsIcon from '@mui/icons-material/Apps';
import MoreVertIcon from '@mui/icons-material/MoreVert';
import UpdateIcon from '@mui/icons-material/Update';
import DeleteIcon from '@mui/icons-material/Delete';
import PlayArrowIcon from '@mui/icons-material/PlayArrow';
import AppPreviewModal from './AppPreviewModal';
import axios from 'axios';
import { BASE_URL } from '../../const';

// Styled components
const StyledCard = styled(Card)(({ theme }) => ({ 
  height: '100%',
  display: 'flex',
  flexDirection: 'column',
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  borderRadius: 12,
  transition: 'all 0.3s ease',
  overflow: 'hidden',
  '&:hover': {
    transform: 'translateY(-4px)',
    boxShadow: '0 8px 24px rgba(0, 192, 127, 0.15)',
    borderColor: 'rgba(0, 192, 127, 0.3)',
  },
}));

const StyledCardMedia = styled(CardMedia)(() => ({
  height: 180,
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  position: 'relative',
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'center',
  '& img': {
    width: '100%',
    height: '100%',
    objectFit: 'cover', // This will crop/scale images to fill the container
    objectPosition: 'center', // Center the image within the container
  },
}));

const FallbackIcon = styled(AppsIcon)(({ theme }) => ({
  fontSize: 80,
  color: theme.palette.primary.main,
  opacity: 0.3,
}));

const PreviewButton = styled(IconButton)(({ theme }) => ({
  position: 'absolute',
  top: theme.spacing(1),
  right: theme.spacing(1),
  backgroundColor: 'rgba(0, 0, 0, 0.6)',
  color: theme.palette.common.white,
  '&:hover': {
    backgroundColor: 'rgba(0, 0, 0, 0.8)',
  },
}));

const StyledCardContent = styled(CardContent)(({ theme }) => ({
  flexGrow: 1,
  padding: theme.spacing(2),
}));

const AppName = styled(Typography)(({ theme }) => ({
  fontWeight: 600,
  fontSize: '1.1rem',
  marginBottom: theme.spacing(0.5),
  color: theme.palette.text.primary,
}));

const AppDescription = styled(Typography)(({ theme }) => ({
  fontSize: '0.9rem',
  color: theme.palette.text.secondary,
  marginBottom: theme.spacing(1.5),
  display: '-webkit-box',
  WebkitLineClamp: 2,
  WebkitBoxOrient: 'vertical',
  overflow: 'hidden',
  textOverflow: 'ellipsis',
  minHeight: '2.8em',
}));

const TagsContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  flexWrap: 'wrap',
  gap: theme.spacing(0.5),
  marginTop: theme.spacing(1),
}));

const StyledChip = styled(Chip)(({ theme }) => ({
  height: 24,
  fontSize: '0.75rem',
  backgroundColor: 'rgba(0, 192, 127, 0.1)',
  color: theme.palette.primary.main,
  borderColor: 'rgba(0, 192, 127, 0.3)',
}));

const StyledCardActions = styled(CardActions)(({ theme }) => ({
  padding: theme.spacing(2),
  paddingTop: 0,
  flexDirection: 'column',
  gap: theme.spacing(1),
}));

const DeployButton = styled(Button)(({ theme }) => ({
  textTransform: 'none',
  fontWeight: 500,
  backgroundColor: theme.palette.primary.main,
  color: theme.palette.primary.contrastText,
  '&:hover': {
    backgroundColor: theme.palette.primary.dark,
  },
}));

const PullButton = styled(Button)(() => ({
  textTransform: 'none',
  fontWeight: 500,
  backgroundColor: 'rgba(33, 150, 243, 0.1)',
  color: '#2196f3',
  border: '1px solid rgba(33, 150, 243, 0.3)',
  '&:hover': {
    backgroundColor: 'rgba(33, 150, 243, 0.2)',
  },
}));

const VersionChip = styled(Chip)(({ theme }) => ({
  position: 'absolute',
  bottom: theme.spacing(1),
  left: theme.spacing(1),
  height: 24,
  fontSize: '0.75rem',
  backgroundColor: 'rgba(0, 0, 0, 0.6)',
  color: theme.palette.common.white,
}));

// Helper function to generate app description based on template name
const generateDescription = name => {
  const descriptions = {
    'hello-web': 'A simple hello world web application to test your deployment',
    nginx: 'High-performance web server and reverse proxy',
    postgres: 'Powerful open-source relational database',
    redis: 'In-memory data structure store and cache',
    mongodb: 'Popular NoSQL document database',
    mysql: "World's most popular open source database",
    wordpress: 'Content management system for websites',
    jupyter: 'Interactive computing environment',
  };

  // Try to match partial names
  const lowerName = name.toLowerCase();
  for (const [key, desc] of Object.entries(descriptions)) {
    if (lowerName.includes(key)) {
      return desc;
    }
  }

  return 'Deploy this application to your cluster';
};

// Helper function to generate tags based on template
const generateTags = template => {
  const tags = [];

  // Add source type tag
  if (template.source_type === 'kamiwaza') {
    tags.push('Official');
  } else if (template.source_type === 'user_repo') {
    tags.push('Community');
  }

  // Add risk tier tag
  if (template.risk_tier === 0) {
    tags.push('Guided');
  } else if (template.risk_tier === 1) {
    tags.push('Scanned');
  }

  // Try to infer tags from name
  const lowerName = template.name.toLowerCase();
  if (lowerName.includes('web') || lowerName.includes('nginx')) {
    tags.push('Web');
  }
  if (
    lowerName.includes('db') ||
    lowerName.includes('database') ||
    lowerName.includes('postgres') ||
    lowerName.includes('mysql') ||
    lowerName.includes('mongo')
  ) {
    tags.push('Database');
  }

  return tags;
};

const AppTemplateCard = ({ 
  template, 
  onDeploy, 
  onDelete, 
  onUpdate,
  hasUpdate = false,
  deployments = [],
  isFirstCard = false 
}) => {
  const [showPreview, setShowPreview] = useState(false);
  const [imageStatus, setImageStatus] = useState(null);
  const [pullingImages, setPullingImages] = useState(false);
  const [menuAnchor, setMenuAnchor] = useState(null);

  // Extract or generate metadata
  const description = template.description || generateDescription(template.name);
  const tags = template.tags || generateTags(template);
  const previewImageUrl = template.preview_image;
  const screenshotUrl = template.screenshot_url;

  useEffect(() => {
    checkImageStatus();
  }, [template.id]);

  const checkImageStatus = async () => {
    try {
      const response = await axios.get(`${BASE_URL}/apps/images/status/${template.id}`);
      setImageStatus(response.data);
    } catch (error) {
      console.error('Error checking image status:', error);
      // On error, assume images need pulling (safer than assuming they're available)
      setImageStatus({
        all_images_pulled: false,
        images: ['unknown'],
        error: true
      });
    }
  };

  const handlePullImages = async () => {
    setPullingImages(true);
    try {
      const response = await axios.post(`${BASE_URL}/apps/images/pull/${template.id}`);
      if (response.data.all_successful) {
        // Update image status to reflect successful pull
        setImageStatus({
          all_images_pulled: true,
          images: response.data.results.map(r => r.image),
        });
      } else {
        console.error('Some images failed to pull:', response.data);
      }
    } catch (error) {
      console.error('Error pulling images:', error);
    } finally {
      setPullingImages(false);
    }
  };

  const handleImageError = e => {
    // Hide broken image icon
    e.target.style.display = 'none';
  };

  const shouldShowPullButton =
    imageStatus && !imageStatus.all_images_pulled && imageStatus.images?.length > 0;

  return (
    <>
      <StyledCard data-tour={isFirstCard ? 'app-card' : undefined}>
        {previewImageUrl ? (
          // When we have an image, use a div container with the image as background
          <StyledCardMedia component="div">
            <img 
              src={previewImageUrl}
              alt={template.name}
              style={{ width: '100%', height: '100%', objectFit: 'cover' }}
              onError={handleImageError}
            />
            {screenshotUrl && (
              <Tooltip title="Preview application">
                <PreviewButton size="small" onClick={() => setShowPreview(true)}>
                  <PreviewIcon />
                </PreviewButton>
              </Tooltip>
            )}
            <VersionChip label={`v${template.version || '1.0.0'}`} size="small" />
          </StyledCardMedia>
        ) : (
          // When no image, show the fallback
          <StyledCardMedia component="div">
            <FallbackIcon />
            {screenshotUrl && (
              <Tooltip title="Preview application">
                <PreviewButton size="small" onClick={() => setShowPreview(true)}>
                  <PreviewIcon />
                </PreviewButton>
              </Tooltip>
            )}
            <VersionChip label={`v${template.version || '1.0.0'}`} size="small" />
          </StyledCardMedia>
        )}

        <StyledCardContent>
          <AppName variant="h6">
            {template.name.replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}
          </AppName>

          <AppDescription variant="body2">{description}</AppDescription>

          <TagsContainer>
            {tags.slice(0, 3).map((tag, index) => (
              <StyledChip key={index} label={tag} size="small" variant="outlined" />
            ))}
          </TagsContainer>
        </StyledCardContent>

        <StyledCardActions>
          {deployments.length > 0 && (
            <Box display="flex" alignItems="center" gap={1} mb={1}>
              <PlayArrowIcon color="success" fontSize="small" />
              <Typography variant="caption" color="success.main">
                {deployments.length} active deployment{deployments.length > 1 ? 's' : ''}
              </Typography>
            </Box>
          )}
          
          {shouldShowPullButton && (
            <PullButton
              fullWidth
              variant="outlined"
              startIcon={pullingImages ? <CircularProgress size={16} /> : <DownloadIcon />}
              onClick={handlePullImages}
              disabled={pullingImages}
              data-tour={isFirstCard ? 'app-pull-images' : undefined}
            >
              {pullingImages ? 'Pulling Images...' : 'Pull Images'}
            </PullButton>
          )}

          <Box display="flex" gap={1}>
            <DeployButton
              variant="contained"
              startIcon={<RocketLaunchIcon />}
              onClick={onDeploy}
              data-tour={isFirstCard ? 'app-deploy' : undefined}
              disabled={shouldShowPullButton}
              sx={{ flex: 1 }}
            >
              Deploy
            </DeployButton>

            {(onUpdate || onDelete) && (
              <Tooltip title="More actions">
                <IconButton
                  onClick={(e) => setMenuAnchor(e.currentTarget)}
                  size="small"
                  sx={{ 
                    border: '1px solid rgba(255, 255, 255, 0.23)',
                    color: 'rgba(255, 255, 255, 0.7)',
                    '&:hover': {
                      backgroundColor: 'rgba(255, 255, 255, 0.08)',
                    }
                  }}
                >
                  {hasUpdate ? (
                    <Badge badgeContent="!" color="warning">
                      <MoreVertIcon fontSize="small" />
                    </Badge>
                  ) : (
                    <MoreVertIcon fontSize="small" />
                  )}
                </IconButton>
              </Tooltip>
            )}
          </Box>

          {shouldShowPullButton && (
            <Typography
              variant="caption"
              sx={{ color: 'text.secondary', textAlign: 'center', mt: 0.5 }}
            >
              Download images first for faster deployment
            </Typography>
          )}
        </StyledCardActions>

        {/* Actions Menu */}
        <Menu
          anchorEl={menuAnchor}
          open={Boolean(menuAnchor)}
          onClose={() => setMenuAnchor(null)}
          transformOrigin={{ horizontal: 'right', vertical: 'top' }}
          anchorOrigin={{ horizontal: 'right', vertical: 'bottom' }}
        >
          {onUpdate && (
            <MenuItem onClick={() => { onUpdate(); setMenuAnchor(null); }}>
              <ListItemIcon>
                {hasUpdate ? (
                  <Badge badgeContent="!" color="warning">
                    <UpdateIcon fontSize="small" />
                  </Badge>
                ) : (
                  <UpdateIcon fontSize="small" />
                )}
              </ListItemIcon>
              <ListItemText>
                {hasUpdate ? 'Update Available' : 'Check for Updates'}
              </ListItemText>
            </MenuItem>
          )}
          {onDelete && (
            <MenuItem 
              onClick={() => { onDelete(); setMenuAnchor(null); }}
              sx={{ color: 'error.main' }}
            >
              <ListItemIcon>
                <DeleteIcon fontSize="small" color="error" />
              </ListItemIcon>
              <ListItemText>Delete Template</ListItemText>
            </MenuItem>
          )}
        </Menu>
      </StyledCard>

      {showPreview && (
        <AppPreviewModal
          template={template}
          open={showPreview}
          onClose={() => setShowPreview(false)}
        />
      )}
    </>
  );
};

AppTemplateCard.propTypes = {
  template: PropTypes.shape({
    id: PropTypes.string.isRequired,
    name: PropTypes.string.isRequired,
    description: PropTypes.string,
    tags: PropTypes.array,
    preview_image: PropTypes.string,
    screenshot_url: PropTypes.string,
    version: PropTypes.string,
  }).isRequired,
  onDeploy: PropTypes.func.isRequired,
  onDelete: PropTypes.func,
  onUpdate: PropTypes.func,
  hasUpdate: PropTypes.bool,
  deployments: PropTypes.array,
  isFirstCard: PropTypes.bool,
};

AppTemplateCard.defaultProps = {
  isFirstCard: false,
  hasUpdate: false,
  deployments: [],
  onDelete: null,
  onUpdate: null,
};

export default AppTemplateCard; 