import React from 'react';
import { styled } from '@mui/material/styles';
import {
  Dialog,
  DialogContent,
  IconButton,
  Box,
  Typography
} from '@mui/material';
import CloseIcon from '@mui/icons-material/Close';
import AppsIcon from '@mui/icons-material/Apps';

// Styled components
const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    borderRadius: 12,
    backgroundColor: theme.palette.background.paper,
    maxWidth: '90vw',
    maxHeight: '90vh',
  },
}));

const CloseButton = styled(IconButton)(({ theme }) => ({
  position: 'absolute',
  right: theme.spacing(1),
  top: theme.spacing(1),
  backgroundColor: 'rgba(0, 0, 0, 0.6)',
  color: theme.palette.common.white,
  zIndex: 1,
  '&:hover': {
    backgroundColor: 'rgba(0, 0, 0, 0.8)',
  },
}));

const ImageContainer = styled(Box)(({ theme }) => ({
  position: 'relative',
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'center',
  minHeight: 400,
  backgroundColor: 'rgba(0, 0, 0, 0.9)',
}));

const PreviewImage = styled('img')(({ theme }) => ({
  maxWidth: '100%',
  maxHeight: '80vh',
  objectFit: 'contain',
}));

const FallbackContainer = styled(Box)(({ theme }) => ({
  display: 'flex',
  flexDirection: 'column',
  alignItems: 'center',
  justifyContent: 'center',
  padding: theme.spacing(8),
  textAlign: 'center',
}));

const FallbackIcon = styled(AppsIcon)(({ theme }) => ({
  fontSize: 120,
  color: theme.palette.primary.main,
  opacity: 0.3,
  marginBottom: theme.spacing(2),
}));

const AppPreviewModal = ({ template, open, onClose }) => {
  const screenshotUrl = template.screenshot_url;
  const appName = template.name.replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase());

  const handleImageError = (e) => {
    // Hide the broken image and show fallback
    e.target.style.display = 'none';
  };

  return (
    <StyledDialog 
      open={open} 
      onClose={onClose}
      maxWidth={false}
    >
      <CloseButton onClick={onClose}>
        <CloseIcon />
      </CloseButton>
      
      <DialogContent sx={{ p: 0 }}>
        <ImageContainer>
          {screenshotUrl ? (
            <PreviewImage
              src={screenshotUrl}
              alt={`${appName} preview`}
              onError={handleImageError}
            />
          ) : (
            <FallbackContainer>
              <FallbackIcon />
              <Typography variant="h5" color="text.secondary" gutterBottom>
                No Preview Available
              </Typography>
              <Typography variant="body1" color="text.secondary">
                A preview image for {appName} will be available soon.
              </Typography>
            </FallbackContainer>
          )}
        </ImageContainer>
      </DialogContent>
    </StyledDialog>
  );
};

export default AppPreviewModal; 