import DownloadIcon from "@mui/icons-material/Download";
import RefreshIcon from "@mui/icons-material/Refresh";
import {
	Alert,
	Box,
	Button,
	CircularProgress,
	IconButton,
	Tooltip,
	Typography,
} from "@mui/material";
import { styled } from "@mui/material/styles";
import React, { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { useTour } from "../../contexts/TourContext";
import api from "../../utils/api";
import AppDeploymentList from "./AppDeploymentList";
import AppTemplateList from "./AppTemplateList";

// Styled components
const PageContainer = styled(Box)(({ theme }) => ({
	maxWidth: "100%",
	margin: "0 auto",
	padding: theme.spacing(3, 2),
}));

const PageTitle = styled(Typography)(({ theme }) => ({
	color: theme.palette.primary.main,
	fontSize: "2.5rem",
	fontWeight: 700,
	marginBottom: theme.spacing(1),
	background: "linear-gradient(90deg, #00c07f, #00c0a0)",
	backgroundClip: "text",
	textFillColor: "transparent",
	WebkitBackgroundClip: "text",
	WebkitTextFillColor: "transparent",
}));

const PageSubtitle = styled(Typography)(({ theme }) => ({
	color: theme.palette.text.secondary,
	fontSize: "1.1rem",
	marginBottom: theme.spacing(4),
}));

const ImportButton = styled(Button)(({ theme }) => ({
	marginBottom: theme.spacing(3),
	background: "linear-gradient(90deg, #00c07f, #00c0a0)",
	color: "white",
	fontWeight: 600,
	padding: theme.spacing(1.5, 3),
	borderRadius: 8,
	"&:hover": {
		background: "linear-gradient(90deg, #00a66b, #00a089)",
	},
	"&:disabled": {
		background: theme.palette.grey[300],
		color: theme.palette.grey[500],
	},
}));

const AppGarden = () => {
	const { startTour } = useTour();
	const navigate = useNavigate();
	const [refreshDeployments, setRefreshDeployments] = useState(false);
	const [refreshTemplates, setRefreshTemplates] = useState({ trigger: false });
	const [gardenStatus, setGardenStatus] = useState(null);
	const [importing, setImporting] = useState(false);
	const [importResult, setImportResult] = useState(null);
	const [hasDeployedModels, setHasDeployedModels] = useState(null);
	const [syncing, setSyncing] = useState(false);

	useEffect(() => {
		checkGardenStatus();
		// Note: No auto-sync on page load - use manual refresh button to avoid hitting remote every load
		checkModelDeployments();

		// Start app garden tour after a delay
		const timer = setTimeout(() => {
			startTour("appGarden");
		}, 1500);

		return () => clearTimeout(timer);
	}, [startTour]);

	const checkGardenStatus = async () => {
		try {
			const response = await api.get("/apps/garden/status");
			setGardenStatus(response.data);
		} catch (error) {
			console.error("Error checking garden status:", error);
		}
	};

	const syncRemoteApps = async () => {
		setSyncing(true);
		try {
			// Call remote sync endpoint - bypasses cache to fetch fresh from remote
			const response = await api.post("/apps/remote/sync");

			if (response.data.success) {
				if (
					response.data.imported_count > 0 ||
					response.data.updated_count > 0
				) {
					console.log(
						`Synced remote apps: ${response.data.imported_count} imported, ${response.data.updated_count || 0} updated`,
					);
				}

				// Refresh garden status and templates after sync
				await checkGardenStatus();
				setRefreshTemplates({ trigger: !refreshTemplates.trigger });
				handleDeploymentChange();
			}
		} catch (error) {
			console.warn("Remote app sync failed:", error);
		} finally {
			setSyncing(false);
		}
	};

	const handleImportGardenApps = async () => {
		setImporting(true);
		setImportResult(null);

		try {
			const response = await api.post("/apps/garden/import");
			setImportResult(response.data);

			if (response.data.success) {
				// Refresh garden status after successful import
				await checkGardenStatus();

				// If we have imported templates, pass them to avoid extra API call
				if (
					response.data.imported_templates &&
					response.data.imported_templates.length > 0
				) {
					setRefreshTemplates({
						trigger: !refreshTemplates.trigger,
						newTemplates: response.data.imported_templates,
					});
				} else {
					// Fallback to trigger refresh if no templates returned
					setRefreshTemplates({ trigger: !refreshTemplates.trigger });
				}

				handleDeploymentChange();
			}
		} catch (error) {
			console.error("Error importing garden apps:", error);
			setImportResult({
				success: false,
				errors: [
					error.response?.data?.detail || "Failed to import garden apps",
				],
			});
		} finally {
			setImporting(false);
		}
	};

	const handleDeploymentChange = () => {
		// Toggle to trigger refresh in AppDeploymentList
		setRefreshDeployments(!refreshDeployments);
	};

	const checkModelDeployments = async () => {
		try {
			const response = await api.get("/serving/deployments");
			const deployments = Array.isArray(response.data) ? response.data : [];

			// Consider a model deployed if deployment is DEPLOYED, has a route (path or port),
			// and at least one instance is DEPLOYED (matches server-side selection logic)
			const anyDeployed = deployments.some((dep) => {
				const hasPath = Boolean(dep.access_path || dep.accessPath);
				const hasPort = Boolean(dep.lb_port && dep.lb_port !== 443);
				const hasRoute = hasPath || hasPort;
				const isDeployed = dep.status === "DEPLOYED";
				const hasDeployedInstance =
					Array.isArray(dep.instances) &&
					dep.instances.some((inst) => inst.status === "DEPLOYED");
				return isDeployed && hasRoute && hasDeployedInstance;
			});

			setHasDeployedModels(anyDeployed);
		} catch (error) {
			console.warn(
				"Error checking model deployments for App Garden banner:",
				error,
			);
			// On error, do not show the banner (avoid false warning)
			setHasDeployedModels(null);
		}
	};

	const shouldShowImportButton =
		gardenStatus?.garden_apps_available && gardenStatus?.missing_count > 0;

	return (
		<PageContainer>
			<Box
				sx={{
					mb: 4,
					display: "flex",
					justifyContent: "space-between",
					alignItems: "flex-start",
				}}
			>
				<Box>
					<PageTitle variant="h1" data-tour="app-garden-title">
						App Garden
					</PageTitle>
					<PageSubtitle variant="body1">
						Deploy AI-Powered Applications with Ease
					</PageSubtitle>
				</Box>
				<Tooltip title="Refresh from remote catalog">
					<IconButton
						onClick={syncRemoteApps}
						disabled={syncing}
						sx={{
							color: "text.secondary",
							"&:hover": { color: "primary.main" },
						}}
					>
						{syncing ? <CircularProgress size={24} /> : <RefreshIcon />}
					</IconButton>
				</Tooltip>
			</Box>

			{/* Garden Import Section */}
			{hasDeployedModels === false && (
				<Box sx={{ mb: 3 }}>
					<Alert
						severity="warning"
						sx={{ display: "flex", alignItems: "center" }}
					>
						No models are currently deployed. Most apps require a deployed model
						to function.
						<Button
							color="warning"
							size="small"
							sx={{ ml: 2 }}
							onClick={() => navigate("/models")}
						>
							Go to Models
						</Button>
					</Alert>
				</Box>
			)}
			{shouldShowImportButton && (
				<Box sx={{ mb: 3 }}>
					<ImportButton
						startIcon={
							importing ? (
								<CircularProgress size={20} color="inherit" />
							) : (
								<DownloadIcon />
							)
						}
						onClick={handleImportGardenApps}
						disabled={importing}
						data-tour="pull-images-button"
					>
						{importing
							? "Importing..."
							: `Import ${gardenStatus.missing_count} ${gardenStatus.missing_count === 1 ? "App" : "Apps"}`}
					</ImportButton>

					{gardenStatus.missing_apps?.length > 0 && (
						<Typography
							variant="caption"
							sx={{ display: "block", color: "text.secondary", mt: 1 }}
						>
							Available apps: {gardenStatus.missing_apps.join(", ")}
						</Typography>
					)}
				</Box>
			)}

			{/* Import Results */}
			{importResult && (
				<Box sx={{ mb: 3 }}>
					<Alert
						severity={importResult.success ? "success" : "error"}
						onClose={() => setImportResult(null)}
						sx={{ mb: 2 }}
					>
						{importResult.success
							? `Successfully imported ${importResult.imported_count} app${importResult.imported_count !== 1 ? "s" : ""}!`
							: "Import failed"}
						{importResult.errors?.length > 0 && (
							<Box sx={{ mt: 1 }}>
								{importResult.errors.map((error, index) => (
									<Typography
										key={index}
										variant="caption"
										sx={{ display: "block" }}
									>
										• {error}
									</Typography>
								))}
							</Box>
						)}
					</Alert>
				</Box>
			)}

			{/* Available App Templates */}
			<AppTemplateList
				onDeploymentChange={handleDeploymentChange}
				refreshTemplates={refreshTemplates}
				ephemeralDefault={gardenStatus?.app_session_ephemeral_default}
			/>

			{/* Deployed Applications */}
			<AppDeploymentList refreshDeployments={refreshDeployments} />
		</PageContainer>
	);
};

export default AppGarden;
