import React, { useState, useEffect, useCallback } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import {
  Typography,
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Tooltip,
  Chip,
  CircularProgress
} from '@mui/material';
import InfoOutlinedIcon from '@mui/icons-material/InfoOutlined';
import OpenInNewIcon from '@mui/icons-material/OpenInNew';
import StopIcon from '@mui/icons-material/Stop';
import VisibilityIcon from '@mui/icons-material/Visibility';
import axios from 'axios';
import Spinner from '../common/Spinner';
import ErrorComponent from '../common/ErrorComponent';
import AppDeploymentDetail from './AppDeploymentDetail';
import ConfirmModal from '../common/ConfirmModal';
import { BASE_URL } from '../../const';
import { buildAppRuntimeInfo } from '../../utils/routing';
import { useRoutingConfig } from '../../context/RoutingConfigContext';

// Styled components
const SubTitle = styled(Typography)(({ theme }) => ({
  color: theme.palette.primary.main,
  fontSize: '1.5rem',
  fontWeight: 600,
  marginTop: theme.spacing(4),
  marginBottom: theme.spacing(2),
  paddingBottom: theme.spacing(1),
  position: 'relative',
  '&:after': {
    content: '""',
    position: 'absolute',
    left: 0,
    bottom: 0,
    width: '60px',
    height: '2px',
    background: 'linear-gradient(90deg, #00c07f, transparent)'
  }
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 12,
  marginBottom: theme.spacing(4),
  boxShadow: '0 4px 20px rgba(0, 0, 0, 0.15)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  overflow: 'auto',
}));

const StyledTable = styled(Table)(({ theme }) => ({
  width: '100%',
  '& .MuiTableCell-root': {
    borderColor: 'rgba(255, 255, 255, 0.08)',
    whiteSpace: 'nowrap',
  },
}));

const TableHeaderCell = styled(TableCell)(({ theme }) => ({
  backgroundColor: 'rgba(0, 0, 0, 0.2)',
  color: theme.palette.primary.main,
  fontWeight: 600,
  fontSize: '0.95rem',
  padding: theme.spacing(1.5, 3),
}));

const TableBodyCell = styled(TableCell)(({ theme }) => ({
  padding: theme.spacing(2, 3),
  fontSize: '0.9rem',
}));

const StyledTableRow = styled(TableRow)(({ theme }) => ({
  transition: 'background-color 0.2s',
  '&:nth-of-type(odd)': {
    backgroundColor: 'rgba(255, 255, 255, 0.02)',
  },
  '&:hover': {
    backgroundColor: 'rgba(255, 255, 255, 0.05)',
  },
  '&.empty-row': {
    height: 100,
    '& td': {
      textAlign: 'center',
      color: theme.palette.text.secondary,
    }
  }
}));

const ActionButton = styled(Button)(({ theme, color = 'primary' }) => ({
  textTransform: 'none',
  fontWeight: 500,
  padding: theme.spacing(0.5, 1.5),
  borderRadius: theme.shape.borderRadius,
  fontSize: '0.8rem',
  transition: 'all 0.2s',
  backgroundColor: color === 'primary'
    ? theme.palette.primary.main
    : theme.palette.error.main,
  color: theme.palette.primary.contrastText,
  '&:hover': {
    backgroundColor: color === 'primary'
      ? theme.palette.primary.dark
      : theme.palette.error.dark,
    transform: 'translateY(-2px)',
    boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
  },
  '&:disabled': {
    backgroundColor: theme.palette.action.disabledBackground,
  },
}));

const InfoIcon = styled(InfoOutlinedIcon)(({ theme }) => ({
  fontSize: '1.1rem',
  color: theme.palette.text.secondary,
  marginLeft: theme.spacing(1),
  cursor: 'pointer',
  transition: 'color 0.2s',
  '&:hover': {
    color: theme.palette.primary.main,
  },
}));

const StatusChip = styled(Chip)(({ theme, status }) => ({
  fontWeight: 500,
  ...(status === 'RUNNING' && {
    backgroundColor: 'rgba(76, 175, 80, 0.1)',
    color: '#4caf50',
    borderColor: '#4caf50',
  }),
  ...(status === 'STOPPED' && {
    backgroundColor: 'rgba(158, 158, 158, 0.1)',
    color: '#9e9e9e',
    borderColor: '#9e9e9e',
  }),
  ...(status === 'FAILED' && {
    backgroundColor: 'rgba(244, 67, 54, 0.1)',
    color: '#f44336',
    borderColor: '#f44336',
  }),
  ...(status === 'STARTING' && {
    backgroundColor: 'rgba(255, 152, 0, 0.1)',
    color: '#ff9800',
    borderColor: '#ff9800',
  }),
}));

const RouteLink = styled('a')(({ theme }) => ({
  color: theme.palette.primary.main,
  textDecoration: 'none',
  fontFamily: 'monospace',
  fontSize: '0.85rem',
  display: 'inline-block',
  maxWidth: '100%',
  overflow: 'hidden',
  textOverflow: 'ellipsis',
  whiteSpace: 'nowrap',
  verticalAlign: 'bottom',
  '&:hover': {
    textDecoration: 'underline',
    color: theme.palette.primary.light,
  },
}));

const AppDeploymentList = ({ refreshDeployments }) => {
  const [deployments, setDeployments] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [selectedDeploymentId, setSelectedDeploymentId] = useState(null);
  const [confirmStop, setConfirmStop] = useState(null);
  const [stoppingDeploymentId, setStoppingDeploymentId] = useState(null);

  // Get system routing mode: 'path', 'port', or 'dual'
  const { routingMode } = useRoutingConfig();

  const fetchDeployments = useCallback(async () => {
    setLoading(true);
    try {
      const response = await axios.get(`${BASE_URL}/apps/deployments`);
      setDeployments(response.data);
    } catch (err) {
      console.error('Error fetching app deployments:', err);
      setError(err);
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchDeployments();
  }, [fetchDeployments, refreshDeployments]);

  const handleOpenDetails = (deploymentId) => {
    setSelectedDeploymentId(deploymentId);
  };

  const handleStopDeployment = async (deploymentId) => {
    setStoppingDeploymentId(deploymentId);
    try {
      await axios.delete(`${BASE_URL}/apps/deployment/${deploymentId}`);
      fetchDeployments();
    } catch (err) {
      console.error('Error stopping deployment:', err);
      setError(err);
    } finally {
      setStoppingDeploymentId(null);
      setConfirmStop(null);
    }
  };

  if (loading && deployments.length === 0) return <Spinner />;
  if (error && deployments.length === 0) return <ErrorComponent message={error.message} />;

  return (
    <Box sx={{ mb: 4 }}>
      <SubTitle variant="h5">
        Deployed Applications
        <Tooltip title="Applications currently running on your cluster">
          <InfoIcon />
        </Tooltip>
      </SubTitle>

      <StyledTableContainer component={Paper}>
        <StyledTable>
          <TableHead>
            <TableRow>
              <TableHeaderCell>Application Name</TableHeaderCell>
              <TableHeaderCell>Status</TableHeaderCell>
              <TableHeaderCell>Instances</TableHeaderCell>
              <TableHeaderCell>Primary Route</TableHeaderCell>
              <TableHeaderCell sx={{ px: 1.5 }}>Open App</TableHeaderCell>
              <TableHeaderCell sx={{ px: 1.5 }}>Details</TableHeaderCell>
              <TableHeaderCell sx={{ px: 1.5, color: theme => theme.palette.error.main }}>Stop</TableHeaderCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {deployments.length === 0 ? (
              <StyledTableRow className="empty-row">
                <TableBodyCell colSpan={7}>No deployed applications found</TableBodyCell>
              </StyledTableRow>
            ) : (
              deployments
                .filter(deployment => deployment.status !== 'STOPPED')
                .map((deployment) => {
                  const runtimeInfo = buildAppRuntimeInfo(deployment);
                  const appUrl = runtimeInfo.baseUrl;
                  const activeInstances = deployment.instances?.filter(i => i.status === 'RUNNING' || i.status === 'DEPLOYED').length || 0;
                  const totalInstances = deployment.starting_copies || deployment.instances?.length || 0;

                  const hasFailureMessage =
                    deployment.status === 'FAILED' &&
                    (deployment.last_error_message || deployment.last_error_code);
                  const statusChip = (
                    <StatusChip
                      label={deployment.status}
                      size="small"
                      variant="outlined"
                      status={deployment.status}
                    />
                  );

                  const failureTitle = hasFailureMessage
                    ? `${deployment.last_error_code ? `${deployment.last_error_code}: ` : ''}${
                        deployment.last_error_message || 'Deployment failed'
                      }`
                    : '';

                  return (
                    <StyledTableRow key={deployment.id}>
                      <TableBodyCell>
                        {deployment.name.replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}
                      </TableBodyCell>
                      <TableBodyCell>
                        {hasFailureMessage ? (
                          <Tooltip title={failureTitle} placement="top">
                            <span>{statusChip}</span>
                          </Tooltip>
                        ) : (
                          statusChip
                        )}
                      </TableBodyCell>
                      <TableBodyCell>
                        {activeInstances} / {totalInstances}
                      </TableBodyCell>
                      <TableBodyCell>
                        {(() => {
                          // Determine which routes to show based on routing mode
                          const hasPort = Boolean(runtimeInfo.portUrl);
                          const hasPath = Boolean(runtimeInfo.pathUrl || runtimeInfo.publicPath);
                          const isDualMode = routingMode === 'dual';
                          const showPort = isDualMode || routingMode !== 'path';
                          // Show path in path mode, dual mode, or as fallback when no port available
                          const needsPathFallback = showPort && !hasPort;
                          const showPath = isDualMode || routingMode === 'path' || needsPathFallback;

                          if (!hasPort && !hasPath) {
                            return <span style={{ color: '#666' }}>Pending</span>;
                          }

                          const routes = [];
                          if (showPort && hasPort) {
                            routes.push({ url: runtimeInfo.portUrl, type: 'port' });
                          }
                          if (showPath && hasPath) {
                            const pathDisplay = runtimeInfo.pathUrl || runtimeInfo.publicPath;
                            routes.push({ url: pathDisplay, type: 'path' });
                          }

                          if (routes.length === 0) {
                            return <span style={{ color: '#666' }}>Unavailable</span>;
                          }

                          return (
                            <Box>
                              {routes.map((route, idx) => (
                                <Box key={route.type} sx={{ mb: idx < routes.length - 1 ? 0.5 : 0 }}>
                                  <Tooltip
                                    title={route.url}
                                    placement="top"
                                    arrow
                                    enterDelay={300}
                                  >
                                    <RouteLink
                                      href={route.url}
                                      target="_blank"
                                      rel="noopener noreferrer"
                                    >
                                      {route.url}
                                    </RouteLink>
                                  </Tooltip>
                                </Box>
                              ))}
                            </Box>
                          );
                        })()}
                      </TableBodyCell>
                      <TableBodyCell sx={{ px: 1.5 }}>
                        {(deployment.status === 'RUNNING' || deployment.status === 'DEPLOYED') && appUrl ? (
                          <ActionButton
                            component="a"
                            href={appUrl}
                            target="_blank"
                            size="small"
                            startIcon={<OpenInNewIcon />}
                          >
                            Open App
                          </ActionButton>
                        ) : (
                          <span style={{ color: '#666' }}>
                            {deployment.status === 'STARTING' ? 'Starting...' : (deployment.status === 'FAILED' ? 'Failed' : 'Not available')}
                          </span>
                        )}
                      </TableBodyCell>
                      <TableBodyCell sx={{ px: 1.5 }}>
                        <ActionButton
                          onClick={() => handleOpenDetails(deployment.id)}
                          size="small"
                          startIcon={<VisibilityIcon />}
                        >
                          Details
                        </ActionButton>
                      </TableBodyCell>
                      <TableBodyCell sx={{ px: 1.5 }}>
                        <ActionButton
                          color="error"
                          size="small"
                          startIcon={stoppingDeploymentId === deployment.id ?
                            <CircularProgress size={16} color="inherit" /> :
                            <StopIcon />
                          }
                          onClick={() => setConfirmStop(deployment)}
                          disabled={deployment.status === 'STOPPING' || stoppingDeploymentId === deployment.id}
                        >
                          {stoppingDeploymentId === deployment.id ? 'Stopping...' : 'Stop'}
                        </ActionButton>
                      </TableBodyCell>
                    </StyledTableRow>
                  );
                })
            )}
          </TableBody>
        </StyledTable>
      </StyledTableContainer>

      {selectedDeploymentId && (
        <AppDeploymentDetail
          deploymentId={selectedDeploymentId}
          isOpen={!!selectedDeploymentId}
          onClose={() => setSelectedDeploymentId(null)}
          onDeploymentStopped={fetchDeployments}
        />
      )}

      {confirmStop && (
        <ConfirmModal
          title="Stop Application"
          message={`Are you sure you want to stop "${confirmStop.name}"? This will terminate all running instances.`}
          showModal={!!confirmStop}
          onConfirm={() => handleStopDeployment(confirmStop.id)}
          onCancel={() => setConfirmStop(null)}
        />
      )}
    </Box>
  );
};

AppDeploymentList.propTypes = {
  refreshDeployments: PropTypes.func.isRequired,
};

export default AppDeploymentList;
