import React, { useState, useEffect, useMemo } from 'react';
import PropTypes from 'prop-types';
import { styled } from '@mui/material/styles';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Chip,
  IconButton,
  Divider,
  Alert,
} from '@mui/material';
import CloseIcon from '@mui/icons-material/Close';
import RefreshIcon from '@mui/icons-material/Refresh';
import OpenInNewIcon from '@mui/icons-material/OpenInNew';
import StopIcon from '@mui/icons-material/Stop';
import axios from 'axios';
import Spinner from '../common/Spinner';
import { BASE_URL } from '../../const';
import { buildAppRuntimeInfo } from '../../utils/routing';
import { useRoutingConfig } from '../../context/RoutingConfigContext';

// Styled components
const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    borderRadius: 12,
    backgroundColor: theme.palette.background.paper,
    backgroundImage: 'linear-gradient(rgba(0, 192, 127, 0.03), rgba(0, 0, 0, 0.03))',
    minWidth: 800,
  },
}));

const StyledDialogTitle = styled(DialogTitle)(({ theme }) => ({
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'space-between',
  paddingBottom: theme.spacing(1),
}));

const TitleText = styled(Typography)(({ theme }) => ({
  fontSize: '1.5rem',
  fontWeight: 600,
  color: theme.palette.primary.main,
}));

const InfoSection = styled(Box)(({ theme }) => ({
  marginBottom: theme.spacing(3),
}));

const SectionTitle = styled(Typography)(({ theme }) => ({
  fontSize: '1.1rem',
  fontWeight: 600,
  marginBottom: theme.spacing(1.5),
  color: theme.palette.text.primary,
}));

const InfoGrid = styled(Box)(({ theme }) => ({
  display: 'grid',
  gridTemplateColumns: 'repeat(2, 1fr)',
  gap: theme.spacing(2),
  marginBottom: theme.spacing(2),
}));

const InfoItem = styled(Box)(({ theme }) => ({
  display: 'flex',
  flexDirection: 'column',
}));

const InfoLabel = styled(Typography)(({ theme }) => ({
  fontSize: '0.875rem',
  color: theme.palette.text.secondary,
  marginBottom: theme.spacing(0.5),
}));

const InfoValue = styled(Typography)(({ theme }) => ({
  fontSize: '1rem',
  fontWeight: 500,
}));

const EnvVarBox = styled(Box)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  border: '1px solid rgba(255, 255, 255, 0.08)',
  borderRadius: 8,
  padding: theme.spacing(2),
  fontFamily: 'monospace',
  fontSize: '0.875rem',
  overflowX: 'auto',
}));

const StatusChip = styled(Chip)(({ theme, status }) => ({
  fontWeight: 500,
  ...(status === 'RUNNING' && {
    backgroundColor: 'rgba(76, 175, 80, 0.1)',
    color: '#4caf50',
    borderColor: '#4caf50',
  }),
  ...(status === 'DEPLOYED' && {
    backgroundColor: 'rgba(76, 175, 80, 0.1)',
    color: '#4caf50',
    borderColor: '#4caf50',
  }),
  ...(status === 'STOPPED' && {
    backgroundColor: 'rgba(158, 158, 158, 0.1)',
    color: '#9e9e9e',
    borderColor: '#9e9e9e',
  }),
  ...(status === 'FAILED' && {
    backgroundColor: 'rgba(244, 67, 54, 0.1)',
    color: '#f44336',
    borderColor: '#f44336',
  }),
  ...(status === 'STARTING' && {
    backgroundColor: 'rgba(255, 152, 0, 0.1)',
    color: '#ff9800',
    borderColor: '#ff9800',
  }),
  ...(status === 'PULLING_IMAGES' && {
    backgroundColor: 'rgba(33, 150, 243, 0.1)',
    color: '#2196f3',
    borderColor: '#2196f3',
  }),
  ...(status === 'REQUESTED' && {
    backgroundColor: 'rgba(156, 39, 176, 0.1)',
    color: '#9c27b0',
    borderColor: '#9c27b0',
  }),
}));

const StyledTableContainer = styled(TableContainer)(({ theme }) => ({
  backgroundColor: 'rgba(255, 255, 255, 0.03)',
  borderRadius: 8,
  border: '1px solid rgba(255, 255, 255, 0.08)',
}));

const getStatusMessage = (status, imageStatus) => {
  switch (status) {
    case 'PULLING_IMAGES':
      if (imageStatus?.images?.length > 0) {
        const pulledCount = Object.values(imageStatus.image_status || {}).filter(Boolean).length;
        const totalCount = imageStatus.images.length;
        return `Pulling Images... (${pulledCount}/${totalCount})`;
      }
      return 'Pulling Images...';
    case 'REQUESTED':
      return 'Preparing Deployment...';
    case 'DEPLOYED':
      return 'Running';
    case 'FAILED':
      return 'Failed';
    case 'STOPPED':
      return 'Stopped';
    default:
      return status;
  }
};

const AppDeploymentDetail = ({ deploymentId, isOpen, onClose, onDeploymentStopped }) => {
  const [deployment, setDeployment] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [confirmStop, setConfirmStop] = useState(false);
  const [isStoppingDeployment, setIsStoppingDeployment] = useState(false);
  const runtimeInfo = useMemo(() => buildAppRuntimeInfo(deployment), [deployment]);

  // Get system routing mode: 'path', 'port', or 'dual'
  const { routingMode } = useRoutingConfig();

  useEffect(() => {
    if (isOpen && deploymentId) {
      fetchDeploymentDetails();
    }
  }, [isOpen, deploymentId]);

  const fetchDeploymentDetails = async () => {
    setLoading(true);
    setError(null);
    try {
      const response = await axios.get(`${BASE_URL}/apps/deployment/${deploymentId}`);
      setDeployment(response.data);
    } catch (err) {
      console.error('Error fetching deployment details:', err);
      setError(err.response?.data?.detail || 'Failed to load deployment details');
    } finally {
      setLoading(false);
    }
  };

  const formatDate = dateString => {
    if (!dateString) return 'N/A';
    return new Date(dateString).toLocaleString();
  };

  const handleStopDeployment = async () => {
    setIsStoppingDeployment(true);
    setConfirmStop(false);
    try {
      await axios.delete(`${BASE_URL}/apps/deployment/${deploymentId}`);
      onClose();
      onDeploymentStopped?.();
    } catch (err) {
      console.error('Error stopping deployment:', err);
      setError(err.response?.data?.detail || 'Failed to stop deployment');
      setIsStoppingDeployment(false);
    }
  };

  const envEntries = React.useMemo(() => {
    if (!deployment) return [];
    const runtimeEntries = deployment.runtime_artifacts?.env_entries;
    if (Array.isArray(runtimeEntries) && runtimeEntries.length > 0) {
      return runtimeEntries.map(entry => ({
        key: entry.key,
        value: entry.value,
        source: entry.source || 'user',
      }));
    }

    if (deployment.env_vars && Object.keys(deployment.env_vars).length > 0) {
      return Object.entries(deployment.env_vars).map(([key, value]) => ({
        key,
        value,
        source: 'user',
      }));
    }

    return [];
  }, [deployment]);

  const sortedEnvEntries = React.useMemo(
    () => envEntries.slice().sort((a, b) => a.key.localeCompare(b.key)),
    [envEntries],
  );

  const systemEnvEntries = React.useMemo(
    () => sortedEnvEntries.filter(entry => entry.source === 'system'),
    [sortedEnvEntries],
  );

  const templateEnvEntries = React.useMemo(
    () => sortedEnvEntries.filter(entry => entry.source === 'template'),
    [sortedEnvEntries],
  );

  const userEnvEntries = React.useMemo(
    () =>
      sortedEnvEntries.filter(
        entry => entry.source !== 'system' && entry.source !== 'template',
      ),
    [sortedEnvEntries],
  );

  if (!isOpen) return null;

  return (
    <StyledDialog open={isOpen} onClose={onClose} maxWidth="md" fullWidth>
      <StyledDialogTitle>
        <TitleText>Deployment Details</TitleText>
        <Box>
          <IconButton onClick={fetchDeploymentDetails} disabled={loading} size="small">
            <RefreshIcon />
          </IconButton>
          <IconButton onClick={onClose} size="small">
            <CloseIcon />
          </IconButton>
        </Box>
      </StyledDialogTitle>

      <Divider />

      <DialogContent>
        {loading && <Spinner />}
        {error && (
          <Alert severity="error" sx={{ mb: 2 }}>
            {error}
          </Alert>
        )}

        {deployment && !loading && (
          <>
            <InfoSection>
              <SectionTitle>General Information</SectionTitle>
              <InfoGrid>
                <InfoItem>
                  <InfoLabel>Name</InfoLabel>
                  <InfoValue>{deployment.name}</InfoValue>
                </InfoItem>
                <InfoItem>
                  <InfoLabel>Status</InfoLabel>
                  <StatusChip
                    label={getStatusMessage(deployment.status, deployment.image_status)}
                    size="small"
                    variant="outlined"
                    status={deployment.status}
                  />
                </InfoItem>
                <InfoItem>
                  <InfoLabel>Created At</InfoLabel>
                  <InfoValue>{formatDate(deployment.created_at)}</InfoValue>
                </InfoItem>
                <InfoItem>
                  <InfoLabel>Deployed At</InfoLabel>
                  <InfoValue>{formatDate(deployment.deployed_at)}</InfoValue>
                </InfoItem>
                <InfoItem>
                  <InfoLabel>Session Lifecycle</InfoLabel>
                  <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                    <StatusChip
                      label={deployment.is_ephemeral_session ? 'Ephemeral' : 'Persistent'}
                      size="small"
                      variant="outlined"
                      status={deployment.is_ephemeral_session ? 'REQUESTED' : 'DEPLOYED'}
                    />
                  </Box>
                </InfoItem>
              </InfoGrid>

              {deployment.is_ephemeral_session && (
                <Alert severity="info" sx={{ mt: 1, fontSize: '0.875rem' }}>
                  This app is ephemeral and will be automatically removed when you log out.
                </Alert>
              )}

              {deployment.status === 'FAILED' && (
                <Alert severity="error" sx={{ mt: 2 }}>
                  {deployment.last_error_code ? (
                    <strong>{deployment.last_error_code}: </strong>
                  ) : null}
                  {deployment.last_error_message ||
                    'Deployment failed during validation checks'}
                </Alert>
              )}

              {deployment.status === 'DEPLOYED' && runtimeInfo.baseUrl && (
                <Box sx={{ mt: 2 }}>
                  <Button
                    variant="contained"
                    color="primary"
                    startIcon={<OpenInNewIcon />}
                    href={runtimeInfo.baseUrl}
                    target="_blank"
                    sx={{ textTransform: 'none' }}
                  >
                    Open Application
                  </Button>
                </Box>
              )}

              {/* Image pulling information */}
              {(deployment.status === 'PULLING_IMAGES' ||
                (deployment.image_status && !deployment.image_status.all_images_pulled)) && (
                <Box sx={{ mt: 2 }}>
                  <Alert severity="info" sx={{ fontSize: '0.875rem' }}>
                    {deployment.status === 'PULLING_IMAGES' ? (
                      <>
                        <strong>Pulling container images...</strong>
                        <br />
                        This may take some time depending on your bandwidth and image sizes.
                        {deployment.image_status?.images && (
                          <Box sx={{ mt: 1, fontSize: '0.8rem' }}>
                            Images: {deployment.image_status.images.join(', ')}
                          </Box>
                        )}
                      </>
                    ) : (
                      <>
                        <strong>Note:</strong> Some container images may need to be downloaded on
                        first deployment, which could extend deployment time.
                      </>
                    )}
                  </Alert>
                </Box>
              )}
            </InfoSection>

            <InfoSection>
              <SectionTitle>Configuration</SectionTitle>
              <InfoGrid>
                <InfoItem>
                  <InfoLabel>Min Instances</InfoLabel>
                  <InfoValue>{deployment.min_copies}</InfoValue>
                </InfoItem>
                <InfoItem>
                  <InfoLabel>Max Instances</InfoLabel>
                  <InfoValue>{deployment.max_copies || 'Unlimited'}</InfoValue>
                </InfoItem>
                <InfoItem>
                  <InfoLabel>Starting Instances</InfoLabel>
                  <InfoValue>{deployment.starting_copies}</InfoValue>
                </InfoItem>
                {/* URL display based on routing mode */}
                {(() => {
                  const hasPort = Boolean(runtimeInfo.portUrl);
                  const hasPath = Boolean(runtimeInfo.pathUrl || runtimeInfo.publicPath);
                  const isDualMode = routingMode === 'dual';
                  const showPort = isDualMode || routingMode !== 'path';
                  // Show path in path mode, dual mode, or as fallback when no port available
                  const needsPathFallback = showPort && !hasPort;
                  const showPath = isDualMode || routingMode === 'path' || needsPathFallback;

                  const routes = [];
                  if (showPort && hasPort) {
                    routes.push({ url: runtimeInfo.portUrl, type: 'port' });
                  }
                  if (showPath && hasPath) {
                    const pathDisplay = runtimeInfo.pathUrl || runtimeInfo.publicPath;
                    routes.push({ url: pathDisplay, type: 'path' });
                  }

                  if (routes.length === 0) {
                    return (
                      <InfoItem>
                        <InfoLabel>URL</InfoLabel>
                        <InfoValue>Unavailable</InfoValue>
                      </InfoItem>
                    );
                  }

                  if (isDualMode && routes.length === 2) {
                    // Dual mode: show both with separate labels
                    return routes.map((route) => (
                      <InfoItem key={route.type}>
                        <InfoLabel>{route.type === 'port' ? 'Port URL' : 'Path URL'}</InfoLabel>
                        <InfoValue sx={{ wordBreak: 'break-all' }}>
                          <Typography
                            component="a"
                            href={route.url}
                            target="_blank"
                            rel="noopener noreferrer"
                            sx={{ color: '#00c07f', textDecoration: 'none' }}
                          >
                            {route.url}
                          </Typography>
                        </InfoValue>
                      </InfoItem>
                    ));
                  }

                  // Single route mode (port or path)
                  const route = routes[0];
                  return (
                    <InfoItem>
                      <InfoLabel>URL</InfoLabel>
                      <InfoValue sx={{ wordBreak: 'break-all' }}>
                        <Typography
                          component="a"
                          href={route.url}
                          target="_blank"
                          rel="noopener noreferrer"
                          sx={{ color: '#00c07f', textDecoration: 'none' }}
                        >
                          {route.url}
                        </Typography>
                      </InfoValue>
                    </InfoItem>
                  );
                })()}
              </InfoGrid>
            </InfoSection>

            {sortedEnvEntries.length > 0 && (
              <InfoSection>
                <SectionTitle>Environment Variables</SectionTitle>
                <EnvVarBox>
                  {systemEnvEntries.length > 0 && (
                    <Box sx={{ mb: templateEnvEntries.length > 0 || userEnvEntries.length > 0 ? 2 : 0 }}>
                      <Typography variant="subtitle2" sx={{ color: '#00c07f', mb: 1 }}>
                        System Variables
                      </Typography>
                      {systemEnvEntries.map((entry, index) => (
                        <Box key={`system-${index}`} sx={{ mb: 0.5 }}>
                          <span style={{ color: '#00c07f' }}>{entry.key}</span>
                          <span style={{ color: '#666' }}>=</span>
                          <span>{entry.value}</span>
                        </Box>
                      ))}
                    </Box>
                  )}

                  {templateEnvEntries.length > 0 && (
                    <Box sx={{ mb: userEnvEntries.length > 0 ? 2 : 0 }}>
                      <Typography variant="subtitle2" sx={{ color: '#00c07f', mb: 1 }}>
                        Template Variables
                      </Typography>
                      {templateEnvEntries.map((entry, index) => (
                        <Box key={`template-${index}`} sx={{ mb: 0.5 }}>
                          <span style={{ color: '#00c07f' }}>{entry.key}</span>
                          <span style={{ color: '#666' }}>=</span>
                          <span>{entry.value}</span>
                        </Box>
                      ))}
                    </Box>
                  )}

                  {userEnvEntries.length > 0 && (
                    <Box>
                      <Typography variant="subtitle2" sx={{ color: '#00c07f', mb: 1 }}>
                        User Variables
                      </Typography>
                      {userEnvEntries.map((entry, index) => (
                        <Box key={`user-${index}`} sx={{ mb: 0.5 }}>
                          <span style={{ color: '#00c07f' }}>{entry.key}</span>
                          <span style={{ color: '#666' }}>=</span>
                          <span>{entry.value}</span>
                        </Box>
                      ))}
                    </Box>
                  )}
                </EnvVarBox>
              </InfoSection>
            )}

            {deployment.instances && deployment.instances.length > 0 && (
              <InfoSection>
                <SectionTitle>Instances</SectionTitle>
                <StyledTableContainer component={Paper}>
                  <Table size="small">
                    <TableHead>
                      <TableRow>
                        <TableCell>Instance ID</TableCell>
                        <TableCell>Host</TableCell>
                        <TableCell>Status</TableCell>
                      </TableRow>
                    </TableHead>
                    <TableBody>
                      {deployment.instances.map(instance => (
                        <TableRow key={instance.id}>
                          <TableCell>{instance.id.slice(0, 8)}...</TableCell>
                          <TableCell>{instance.host_name || 'N/A'}</TableCell>
                          <TableCell>
                            <StatusChip
                              label={getStatusMessage(instance.status, deployment.image_status)}
                              size="small"
                              variant="outlined"
                              status={instance.status}
                            />
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </StyledTableContainer>
              </InfoSection>
            )}
          </>
        )}
      </DialogContent>

      <DialogActions sx={{ p: 2, justifyContent: 'space-between' }}>
        <Box>
          {deployment && ['DEPLOYED', 'RUNNING', 'STARTING', 'PULLING_IMAGES', 'REQUESTED'].includes(deployment.status) && (
            <Button
              color="error"
              variant="outlined"
              onClick={() => setConfirmStop(true)}
              startIcon={<StopIcon />}
              disabled={isStoppingDeployment}
            >
              {isStoppingDeployment ? 'Stopping...' : 'Stop'}
            </Button>
          )}
        </Box>
        <Button onClick={onClose}>Close</Button>
      </DialogActions>

      {/* Stop Confirmation Dialog */}
      <Dialog open={confirmStop} onClose={() => setConfirmStop(false)}>
        <DialogTitle>Confirm Stop Deployment</DialogTitle>
        <DialogContent>
          <Typography>
            This will stop this app deployment. It will immediately become unavailable. Are you sure?
          </Typography>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setConfirmStop(false)}>Cancel</Button>
          <Button onClick={handleStopDeployment} color="error" variant="contained">
            Yes, Stop
          </Button>
        </DialogActions>
      </Dialog>
    </StyledDialog>
  );
};

AppDeploymentDetail.propTypes = {
  deploymentId: PropTypes.string,
  isOpen: PropTypes.bool.isRequired,
  onClose: PropTypes.func.isRequired,
  onDeploymentStopped: PropTypes.func,
};

export default AppDeploymentDetail;
